import { useParams, useLocation, Link } from "wouter";
import { useQuery } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import { ArrowLeft, Loader2, Package, ExternalLink, Image as ImageIcon } from "lucide-react";
import { Skeleton } from "@/components/ui/skeleton";

interface ProductSet {
  id: number;
  setMatrixId: number | null;
  sku: string;
  title: string;
  shortDescription: string | null;
  fullDescription: string | null;
  color: string | null;
  depth: string | null;
  panelCount: number | null;
  hookLength: string | null;
  basePrice: string | null;
  calculatedPrice: string | null;
  images: string[];
  generatedFromMatrix: boolean;
  combinationKey: string | null;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
}

interface SetMatrix {
  id: number;
  name: string;
  productGroup: string | null;
}

interface SetProductLink {
  linkId: number;
  productId: number | null;
  componentType: string;
  quantity: number;
  position: number;
  createdAt: string;
  product: {
    id: number;
    sku: string;
    title: string;
    basePrice: string | null;
    isActive: boolean;
    image: {
      url: string;
      altText: string | null;
      isPrimary: boolean;
    } | null;
  } | null;
}

export default function CatalogSetEditor() {
  const { id } = useParams();
  const [, navigate] = useLocation();

  // Fetch set data
  const { data: productSet, isLoading } = useQuery<ProductSet>({
    queryKey: [`/api/product-sets/${id}`],
    enabled: !!id,
  });

  // Fetch matrix info if set has setMatrixId
  const { data: matrix } = useQuery<SetMatrix>({
    queryKey: [`/api/set-matrices/${productSet?.setMatrixId}`],
    enabled: !!productSet?.setMatrixId,
  });

  // Fetch component products
  const { data: componentProducts = [] } = useQuery<SetProductLink[]>({
    queryKey: [`/api/product-sets/${id}/products`],
    enabled: !!id,
  });

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-full">
        <Loader2 className="h-8 w-8 animate-spin" />
      </div>
    );
  }

  if (!productSet) {
    return (
      <div className="flex flex-col items-center justify-center h-full gap-4">
        <p className="text-muted-foreground">Nie znaleziono zestawu</p>
        <Link href="/catalog-sets">
          <Button variant="outline">
            <ArrowLeft className="h-4 w-4 mr-2" />
            Powrót do listy
          </Button>
        </Link>
      </div>
    );
  }

  return (
    <div className="flex flex-col h-full w-full max-w-full overflow-x-hidden">
      {/* Header */}
      <div className="border-b bg-card">
        <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between p-4 gap-4 max-w-full">
          <div className="flex items-center gap-4 w-full sm:w-auto min-w-0 flex-1">
            <Link href="/catalog-sets">
              <Button variant="ghost" size="icon" data-testid="button-back" className="flex-shrink-0">
                <ArrowLeft className="h-4 w-4" />
              </Button>
            </Link>
            <div className="flex-1 min-w-0 overflow-hidden">
              <h1 className="text-xl sm:text-2xl font-bold break-words line-clamp-2">
                {productSet.title}
              </h1>
              <p className="text-sm text-muted-foreground truncate">SKU: {productSet.sku}</p>
            </div>
          </div>
          <div className="flex flex-wrap items-center gap-1.5 sm:gap-2 w-full sm:w-auto justify-end">
            {matrix && (
              <Link href={`/set-matrices/settings/${matrix.id}`}>
                <Button variant="outline" size="sm" data-testid="link-matrix">
                  <Package className="h-4 w-4 mr-1.5" />
                  Macierz: {matrix.name}
                </Button>
              </Link>
            )}
            <Badge variant={productSet.isActive ? "default" : "secondary"}>
              {productSet.isActive ? "Aktywny" : "Nieaktywny"}
            </Badge>
            {productSet.generatedFromMatrix && (
              <Badge variant="outline">
                <Package className="h-3 w-3 mr-1" />
                Wygenerowany
              </Badge>
            )}
          </div>
        </div>
      </div>

      {/* Content */}
      <div className="flex-1 overflow-auto">
        <div className="p-4 max-w-7xl mx-auto space-y-6">
          <Tabs defaultValue="podstawowe" className="w-full">
            <TabsList className="grid w-full grid-cols-3">
              <TabsTrigger value="podstawowe" data-testid="tab-podstawowe">Podstawowe</TabsTrigger>
              <TabsTrigger value="opis" data-testid="tab-opis">Opis</TabsTrigger>
              <TabsTrigger value="obrazy" data-testid="tab-obrazy">Obrazy</TabsTrigger>
            </TabsList>

            {/* Podstawowe */}
            <TabsContent value="podstawowe" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>Informacje podstawowe</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                      <label className="text-sm font-medium">SKU</label>
                      <p className="text-sm text-muted-foreground mt-1">{productSet.sku}</p>
                    </div>
                    <div>
                      <label className="text-sm font-medium">Tytuł</label>
                      <p className="text-sm text-muted-foreground mt-1">{productSet.title}</p>
                    </div>
                    {productSet.color && (
                      <div>
                        <label className="text-sm font-medium">Kolor</label>
                        <p className="text-sm text-muted-foreground mt-1">{productSet.color}</p>
                      </div>
                    )}
                    {productSet.depth && (
                      <div>
                        <label className="text-sm font-medium">Głębokość</label>
                        <p className="text-sm text-muted-foreground mt-1">{productSet.depth}</p>
                      </div>
                    )}
                    {productSet.panelCount !== null && (
                      <div>
                        <label className="text-sm font-medium">Liczba paneli</label>
                        <p className="text-sm text-muted-foreground mt-1">{productSet.panelCount}</p>
                      </div>
                    )}
                    {productSet.hookLength && (
                      <div>
                        <label className="text-sm font-medium">Długość wieszaka</label>
                        <p className="text-sm text-muted-foreground mt-1">{productSet.hookLength}</p>
                      </div>
                    )}
                  </div>

                  {/* Komponenty zestawu */}
                  {componentProducts.length > 0 && (
                    <div className="pt-4 border-t">
                      <label className="text-sm font-medium mb-3 block">Komponenty zestawu ({componentProducts.length})</label>
                      <div className="space-y-3">
                        {componentProducts.map((link) => {
                          const formattedDate = new Date(link.createdAt).toLocaleDateString('pl-PL', {
                            year: 'numeric',
                            month: '2-digit',
                            day: '2-digit'
                          });

                          // Jeśli produkt istnieje - pokaż z linkiem
                          if (link.product) {
                            return (
                              <Link 
                                key={link.linkId} 
                                href={`/catalog-products/${link.product.id}`}
                                className="flex items-center gap-3 p-3 rounded-lg border hover-elevate active-elevate-2 transition-all"
                                data-testid={`component-product-${link.product.id}`}
                              >
                                {/* Zdjęcie */}
                                <div className="flex-shrink-0 w-16 h-16 rounded-md overflow-hidden bg-muted border">
                                  {link.product.image ? (
                                    <img 
                                      src={link.product.image.url} 
                                      alt={link.product.image.altText || link.product.title}
                                      className="w-full h-full object-cover"
                                    />
                                  ) : (
                                    <div className="w-full h-full flex items-center justify-center">
                                      <ImageIcon className="h-6 w-6 text-muted-foreground" />
                                    </div>
                                  )}
                                </div>
                                
                                {/* Informacje */}
                                <div className="flex-1 min-w-0">
                                  <p className="text-sm font-medium line-clamp-2 hover:text-primary transition-colors">
                                    {link.product.title}
                                  </p>
                                  <p className="text-xs text-muted-foreground mt-1">
                                    SKU: {link.product.sku} • Ilość: {link.quantity}
                                    <span className="ml-2 text-[11px]">
                                      (dodano: {formattedDate})
                                    </span>
                                  </p>
                                </div>
                              </Link>
                            );
                          }

                          // Jeśli produkt nie istnieje - pokaż jako templateowy komponent (bez linku)
                          return (
                            <div 
                              key={link.linkId}
                              className="flex items-center gap-3 p-3 rounded-lg border bg-muted/30"
                              data-testid={`component-template-${link.linkId}`}
                            >
                              {/* Placeholder zdjęcia */}
                              <div className="flex-shrink-0 w-16 h-16 rounded-md overflow-hidden bg-muted border">
                                <div className="w-full h-full flex items-center justify-center">
                                  <Package className="h-6 w-6 text-muted-foreground" />
                                </div>
                              </div>
                              
                              {/* Informacje */}
                              <div className="flex-1 min-w-0">
                                <p className="text-sm font-medium text-muted-foreground">
                                  {link.componentType} <span className="text-xs">(szablon)</span>
                                </p>
                                <p className="text-xs text-muted-foreground mt-1">
                                  Ilość: {link.quantity}
                                  <span className="ml-2 text-[11px]">
                                    (dodano: {formattedDate})
                                  </span>
                                </p>
                              </div>
                            </div>
                          );
                        })}
                      </div>
                    </div>
                  )}
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Ceny</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    {productSet.basePrice && (
                      <div>
                        <label className="text-sm font-medium">Cena bazowa</label>
                        <p className="text-sm text-muted-foreground mt-1">{productSet.basePrice} PLN</p>
                      </div>
                    )}
                    {productSet.calculatedPrice && (
                      <div>
                        <label className="text-sm font-medium">Cena kalkulowana</label>
                        <p className="text-sm text-muted-foreground mt-1">{productSet.calculatedPrice} PLN</p>
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>

              {matrix && (
                <Card>
                  <CardHeader>
                    <CardTitle>Macierz zestawu</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="font-medium">{matrix.name}</p>
                        {matrix.productGroup && (
                          <p className="text-sm text-muted-foreground">Grupa: {matrix.productGroup}</p>
                        )}
                      </div>
                      <Link href={`/set-matrices/settings/${matrix.id}`}>
                        <Button variant="outline" size="sm">
                          <ExternalLink className="h-4 w-4 mr-2" />
                          Otwórz macierz
                        </Button>
                      </Link>
                    </div>
                  </CardContent>
                </Card>
              )}
            </TabsContent>

            {/* Opis */}
            <TabsContent value="opis" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>Krótki opis</CardTitle>
                </CardHeader>
                <CardContent>
                  {productSet.shortDescription ? (
                    <p className="text-sm">{productSet.shortDescription}</p>
                  ) : (
                    <p className="text-sm text-muted-foreground">Brak opisu</p>
                  )}
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Pełny opis</CardTitle>
                </CardHeader>
                <CardContent>
                  {productSet.fullDescription ? (
                    <div 
                      className="prose prose-sm dark:prose-invert max-w-none"
                      dangerouslySetInnerHTML={{ __html: productSet.fullDescription }}
                    />
                  ) : (
                    <p className="text-sm text-muted-foreground">Brak pełnego opisu</p>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            {/* Obrazy */}
            <TabsContent value="obrazy" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>Galeria zdjęć</CardTitle>
                </CardHeader>
                <CardContent>
                  {productSet.images && productSet.images.length > 0 ? (
                    <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                      {productSet.images.map((imageUrl, index) => (
                        <div 
                          key={index} 
                          className="relative aspect-square rounded-lg overflow-hidden border bg-muted group"
                        >
                          <img
                            src={imageUrl}
                            alt={`${productSet.title} - obraz ${index + 1}`}
                            className="w-full h-full object-cover"
                            onError={(e) => {
                              (e.target as HTMLImageElement).src = 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAwIiBoZWlnaHQ9IjIwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMjAwIiBoZWlnaHQ9IjIwMCIgZmlsbD0iI2UwZTBlMCIvPjx0ZXh0IHg9IjUwJSIgeT0iNTAlIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMTQiIGZpbGw9IiM5OTk5OTkiIHRleHQtYW5jaG9yPSJtaWRkbGUiIGR5PSIuM2VtIj5CcmFrIG9icmF6dTwvdGV4dD48L3N2Zz4=';
                            }}
                          />
                          {index === 0 && (
                            <Badge className="absolute top-2 left-2" variant="secondary">
                              Główne
                            </Badge>
                          )}
                        </div>
                      ))}
                    </div>
                  ) : (
                    <div className="flex flex-col items-center justify-center py-12 text-muted-foreground">
                      <ImageIcon className="h-12 w-12 mb-2 opacity-20" />
                      <p className="text-sm">Brak zdjęć</p>
                    </div>
                  )}
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>

          {/* Metadata */}
          <Card>
            <CardHeader>
              <CardTitle>Informacje systemowe</CardTitle>
            </CardHeader>
            <CardContent className="space-y-2">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                <div>
                  <span className="font-medium">ID:</span> <span className="text-muted-foreground">{productSet.id}</span>
                </div>
                {productSet.combinationKey && (
                  <div>
                    <span className="font-medium">Klucz kombinacji:</span>{" "}
                    <span className="text-muted-foreground">{productSet.combinationKey}</span>
                  </div>
                )}
                <div>
                  <span className="font-medium">Utworzono:</span>{" "}
                  <span className="text-muted-foreground">
                    {new Date(productSet.createdAt).toLocaleString("pl-PL")}
                  </span>
                </div>
                <div>
                  <span className="font-medium">Zaktualizowano:</span>{" "}
                  <span className="text-muted-foreground">
                    {new Date(productSet.updatedAt).toLocaleString("pl-PL")}
                  </span>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
