import { useParams, useLocation, Link } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { useState, useRef, useEffect } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { DictionaryCombobox } from "@/components/dictionary-combobox";
import { useToast } from "@/hooks/use-toast";
import { ArrowLeft, Save, Loader2, Upload, X, Edit2, Star, Plus, Trash2, Package, FileDown, FileText, FileSpreadsheet } from "lucide-react";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { insertProductSchema } from "@shared/schema";
import type { Product, ProductPlatformData, ProductImage, ProductAddonAssignment } from "@shared/schema";
import { AIGenerationDialog } from "@/components/ai-generation-dialog";
import { AIGenerationHistory } from "@/components/ai-generation-history";
import { TemplatePickerDialog } from "@/components/template-picker-dialog";
import { RichTextEditor } from "@/components/rich-text-editor";
import { TagInput } from "@/components/tag-input";
import { Eye, Code2 } from "lucide-react";

// Platform data schema
const platformDataSchema = z.object({
  customTitle: z.string().optional(),
  categoryId: z.string().optional(),
  categoryName: z.string().optional(),
  description: z.string().optional(),
  customPrice: z.coerce.number().optional(),
  isPublished: z.boolean().default(false),
}).optional();

// Form schema extending the shared insert schema
const productFormSchema = insertProductSchema.extend({
  // Convert string inputs to numbers for numeric fields
  length: z.coerce.number().optional(),
  width: z.coerce.number().optional(),
  height: z.coerce.number().optional(),
  weight: z.coerce.number().optional(),
  basePrice: z.coerce.number().optional(),
  longDescriptionHtml: z.string().optional(),
  
  // Platform-specific data
  allegro: platformDataSchema,
  shoper: platformDataSchema,
  odoo: platformDataSchema,
});

type ProductFormData = z.infer<typeof productFormSchema>;

export default function CatalogProductEditor() {
  const { id } = useParams();
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const isNew = id === "new";

  // Fetch product data if editing
  const { data: product, isLoading } = useQuery<Product & {
    platformData?: ProductPlatformData[];
    images?: ProductImage[];
    addons?: ProductAddonAssignment[];
  }>({
    queryKey: [`/api/catalog-products/${id}`],
    enabled: !isNew,
  });

  // Fetch matrix info if product has matrixId
  interface ProductMatrix {
    id: number;
    name: string;
  }
  
  const { data: matrix } = useQuery<ProductMatrix>({
    queryKey: [`/api/product-matrices/${product?.matrixId}`],
    enabled: !!product?.matrixId,
  });

  // Fetch product sets where this product is used
  interface ProductSet {
    id: number;
    sku: string;
    title: string;
    setMatrixId: number | null;
    quantity: number;
    componentType: string;
  }
  
  const { data: productSets = [] } = useQuery<ProductSet[]>({
    queryKey: [`/api/catalog-products/${id}/sets`],
    enabled: !isNew && !!id,
  });

  // Fetch dictionaries
  interface ProductCreatorDictionary {
    id: number;
    dictionaryType: string;
    code: string;
    name: string;
    readableName: string | null;
    isActive: boolean;
  }

  const { data: colors } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=color"],
  });
  const { data: materials } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=material"],
  });
  const { data: dimensionLengths } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=dimension_length"],
  });
  const { data: dimensionWidths } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=dimension_width"],
  });
  const { data: dimensionHeights } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=dimension_height"],
  });
  const { data: productTypes } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=product_type"],
  });
  const { data: productGroups } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=product_group"],
  });
  const { data: doorsOptions } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=door"],
  });
  const { data: legsOptions } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=leg"],
  });

  // Initialize form
  const form = useForm<ProductFormData>({
    resolver: zodResolver(productFormSchema),
    defaultValues: {
      sku: "",
      title: "",
      shortDescription: "",
      longDescriptionHtml: "",
      length: null,
      width: null,
      height: null,
      weight: null,
      color: "",
      material: "",
      productType: "",
      productGroup: "",
      doors: "",
      legs: "",
      basePrice: null,
      currency: "PLN",
      isActive: true,
      allegro: {
        customTitle: "",
        categoryId: "",
        categoryName: "",
        description: "",
        customPrice: null,
        isPublished: false,
      },
      shoper: {
        customTitle: "",
        categoryId: "",
        categoryName: "",
        description: "",
        customPrice: null,
        isPublished: false,
      },
      odoo: {
        customTitle: "",
        categoryId: "",
        categoryName: "",
        description: "",
        customPrice: null,
        isPublished: false,
      },
    },
  });

  // Template Picker Dialog State
  const [isTemplatePickerOpen, setIsTemplatePickerOpen] = useState(false);
  
  // Description Editor Mode: 'code' for HTML textarea, 'editor' for WYSIWYG
  const [descriptionMode, setDescriptionMode] = useState<'code' | 'editor'>('editor');
  
  // Description Preview Dialog State
  const [isPreviewDialogOpen, setIsPreviewDialogOpen] = useState(false);

  // Update form when product data loads
  useEffect(() => {
    if (product && !isNew) {
      const allegroPlatform = product.platformData?.find(p => p.platform === "allegro");
      const shoperPlatform = product.platformData?.find(p => p.platform === "shoper");
      const odooPlatform = product.platformData?.find(p => p.platform === "odoo");

      form.reset({
        sku: product.sku,
        title: product.title,
        shortDescription: product.shortDescription ?? "",
        longDescriptionHtml: product.longDescriptionHtml ?? "",
        length: product.length ? Number(product.length) : undefined,
        width: product.width ? Number(product.width) : undefined,
        height: product.height ? Number(product.height) : undefined,
        weight: product.weight ? Number(product.weight) : undefined,
        color: product.color ?? "",
        material: product.material ?? "",
        productType: product.productType ?? "",
        productGroup: product.productGroup ?? "",
        doors: product.doors ?? "",
        legs: product.legs ?? "",
        basePrice: product.basePrice ? Number(product.basePrice) : undefined,
        currency: product.currency ?? "PLN",
        isActive: product.isActive ?? true,
        allegro: allegroPlatform ? {
          customTitle: allegroPlatform.customTitle ?? "",
          categoryId: allegroPlatform.categoryId ?? "",
          categoryName: allegroPlatform.categoryName ?? "",
          description: allegroPlatform.description ?? "",
          customPrice: allegroPlatform.customPrice ? Number(allegroPlatform.customPrice) : undefined,
          isPublished: allegroPlatform.isPublished ?? false,
        } : undefined,
        shoper: shoperPlatform ? {
          customTitle: shoperPlatform.customTitle ?? "",
          categoryId: shoperPlatform.categoryId ?? "",
          categoryName: shoperPlatform.categoryName ?? "",
          description: shoperPlatform.description ?? "",
          customPrice: shoperPlatform.customPrice ? Number(shoperPlatform.customPrice) : undefined,
          isPublished: shoperPlatform.isPublished ?? false,
        } : undefined,
        odoo: odooPlatform ? {
          customTitle: odooPlatform.customTitle ?? "",
          categoryId: odooPlatform.categoryId ?? "",
          categoryName: odooPlatform.categoryName ?? "",
          description: odooPlatform.description ?? "",
          customPrice: odooPlatform.customPrice ? Number(odooPlatform.customPrice) : undefined,
          isPublished: odooPlatform.isPublished ?? false,
        } : undefined,
      });
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [product]);

  // Save mutation
  const saveMutation = useMutation({
    mutationFn: async (data: ProductFormData) => {
      const response = isNew
        ? await apiRequest("POST", "/api/catalog-products", data)
        : await apiRequest("PUT", `/api/catalog-products/${id}`, data);
      
      // Parse the response to get the product data
      return await response.json();
    },
    onSuccess: (savedProduct: any) => {
      toast({
        title: "Sukces",
        description: isNew ? "Produkt został utworzony" : "Produkt został zaktualizowany",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/catalog-products"] });
      
      if (isNew && savedProduct?.id) {
        // For new products, navigate to edit page with the new ID
        navigate(`/catalog-products/${savedProduct.id}`);
      } else if (!isNew) {
        // For existing products, stay on the page and refresh data
        queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${id}`] });
      }
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const onSubmit = (data: ProductFormData) => {
    saveMutation.mutate(data);
  };

  // Copy shared data to platform-specific fields
  const copySharedToPlatform = (platform: "allegro" | "shoper" | "odoo") => {
    const sharedData = form.getValues();
    
    form.setValue(`${platform}.customTitle`, sharedData.title || "");
    form.setValue(`${platform}.description`, sharedData.longDescriptionHtml || "");
    form.setValue(`${platform}.customPrice`, sharedData.basePrice);
    
    toast({
      title: "Skopiowano dane",
      description: `Dane wspólne zostały skopiowane do zakładki ${platform.toUpperCase()}`,
    });
  };

  // Export functions
  const handleExport = async (format: 'html' | 'csv' | 'pdf') => {
    if (isNew) {
      toast({
        title: "Zapisz produkt",
        description: "Najpierw zapisz produkt, aby go wyeksportować",
        variant: "destructive",
      });
      return;
    }

    try {
      const response = await fetch(`/api/catalog-products/${id}/export?format=${format}`, {
        method: 'GET',
      });

      if (!response.ok) {
        throw new Error('Export failed');
      }

      // Get filename from Content-Disposition header or use default
      const contentDisposition = response.headers.get('Content-Disposition');
      let filename = `product-${product?.sku || id}.${format}`;
      if (contentDisposition) {
        const filenameMatch = contentDisposition.match(/filename="?(.+)"?/i);
        if (filenameMatch) {
          filename = filenameMatch[1];
        }
      }

      // Download file
      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      window.URL.revokeObjectURL(url);
      document.body.removeChild(a);

      toast({
        title: "Eksport zakończony",
        description: `Plik ${filename} został pobrany`,
      });
    } catch (error) {
      console.error('Export error:', error);
      toast({
        title: "Błąd eksportu",
        description: "Nie udało się wyeksportować produktu",
        variant: "destructive",
      });
    }
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-full">
        <Loader2 className="h-8 w-8 animate-spin" />
      </div>
    );
  }

  return (
    <div className="flex flex-col h-full w-full max-w-full overflow-x-hidden">
      {/* Header */}
      <div className="border-b bg-card">
        <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between p-4 gap-4 max-w-full">
          <div className="flex items-center gap-4 w-full sm:w-auto min-w-0 flex-1">
            <Link href="/catalog-products">
              <Button variant="ghost" size="icon" data-testid="button-back" className="flex-shrink-0">
                <ArrowLeft className="h-4 w-4" />
              </Button>
            </Link>
            <div className="flex-1 min-w-0 overflow-hidden">
              <h1 className="text-xl sm:text-2xl font-bold break-words line-clamp-2">
                {isNew ? "Nowy produkt" : `Edycja: ${product?.title}`}
              </h1>
              {!isNew && product && (
                <p className="text-sm text-muted-foreground truncate">SKU: {product.sku}</p>
              )}
            </div>
          </div>
          <div className="flex flex-wrap items-center gap-1.5 sm:gap-2 w-full sm:w-auto justify-end">
            {matrix && (
              <Link href={`/product-matrices/${matrix.id}/edit`}>
                <Button variant="outline" size="sm" data-testid="link-matrix">
                  <Package className="h-4 w-4 mr-1.5" />
                  <span className="text-xs sm:text-sm">{matrix.name}</span>
                </Button>
              </Link>
            )}
            {!isNew && (
              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button variant="outline" size="sm" data-testid="button-export" aria-label="Eksport produktu">
                    <FileDown className="h-4 w-4" />
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent align="end">
                  <DropdownMenuItem onClick={() => handleExport('html')} data-testid="export-html">
                    <FileText className="h-4 w-4 mr-2" />
                    Pobierz HTML
                  </DropdownMenuItem>
                  <DropdownMenuItem onClick={() => handleExport('csv')} data-testid="export-csv">
                    <FileSpreadsheet className="h-4 w-4 mr-2" />
                    Pobierz CSV
                  </DropdownMenuItem>
                  <DropdownMenuItem onClick={() => handleExport('pdf')} data-testid="export-pdf">
                    <FileText className="h-4 w-4 mr-2" />
                    Pobierz PDF
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>
            )}
            <Button
              onClick={form.handleSubmit(onSubmit)}
              disabled={saveMutation.isPending}
              size="sm"
              data-testid="button-save"
            >
              {saveMutation.isPending ? (
                <>
                  <Loader2 className="h-4 w-4 mr-1.5 animate-spin" />
                  <span className="text-xs sm:text-sm">Zapisz</span>
                </>
              ) : (
                <>
                  <Save className="h-4 w-4 mr-1.5" />
                  <span className="text-xs sm:text-sm">Zapisz</span>
                </>
              )}
            </Button>
          </div>
        </div>
      </div>

      {/* Content */}
      <div className="flex-1 overflow-x-hidden overflow-y-auto w-full">
        <div className="max-w-7xl mx-auto w-full px-2 sm:px-4 lg:px-6 py-2 sm:py-4 lg:py-6">
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4 sm:space-y-6 w-full max-w-full">
              {/* Zestawy, w których występuje produkt */}
              {!isNew && productSets.length > 0 && (
                <Card className="border-l-4 border-l-blue-500 bg-blue-50/50 dark:bg-blue-950/20">
                  <CardHeader>
                    <CardTitle className="text-base flex items-center gap-2">
                      <Package className="h-5 w-5" />
                      Występowanie w zestawach ({productSets.length})
                    </CardTitle>
                    <CardDescription>
                      Ten produkt jest komponentem w następujących zestawach:
                    </CardDescription>
                  </CardHeader>
                  <CardContent>
                    <ol className="list-decimal list-inside space-y-2">
                      {productSets.map((set, index) => (
                        <li key={set.id} className="text-sm">
                          <Link href={`/catalog-sets/${set.id}`} className="text-primary hover:underline font-medium">
                            {set.title}
                          </Link>
                          <span className="text-muted-foreground ml-2">
                            (SKU: {set.sku}, Ilość: {set.quantity})
                          </span>
                        </li>
                      ))}
                    </ol>
                  </CardContent>
                </Card>
              )}
              
              <Tabs defaultValue="shared" className="w-full max-w-full overflow-hidden">
              <div className="w-full overflow-x-auto overflow-y-hidden pb-2">
                <TabsList className="flex sm:grid w-max sm:w-full sm:grid-cols-6 gap-1 min-w-0">
                  <TabsTrigger value="shared" data-testid="tab-shared" className="whitespace-nowrap">
                    Wspólne
                  </TabsTrigger>
                  <TabsTrigger value="allegro" data-testid="tab-allegro" className="whitespace-nowrap">
                    Allegro
                  </TabsTrigger>
                  <TabsTrigger value="shoper" data-testid="tab-shoper" className="whitespace-nowrap">
                    Shoper
                  </TabsTrigger>
                  <TabsTrigger value="odoo" data-testid="tab-odoo" className="whitespace-nowrap">
                    Odoo
                  </TabsTrigger>
                  <TabsTrigger value="gallery" data-testid="tab-gallery" className="whitespace-nowrap">
                    Galeria
                  </TabsTrigger>
                  <TabsTrigger value="addons" data-testid="tab-addons" className="whitespace-nowrap">
                    Dodatki
                  </TabsTrigger>
                </TabsList>
              </div>

              {/* Zakładka: Wspólne */}
              <TabsContent value="shared" className="space-y-4">
                <Card className="overflow-hidden">
                  <CardHeader>
                    <CardTitle>Podstawowe informacje</CardTitle>
                    <CardDescription>
                      Dane wspólne dla wszystkich platform
                    </CardDescription>
                  </CardHeader>
                  <CardContent className="space-y-4 overflow-x-auto">
                    {/* Tytuł - większy na górze */}
                    <FormField
                      control={form.control}
                      name="title"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel className="text-lg font-semibold">Tytuł produktu *</FormLabel>
                          <FormControl>
                            <Input {...field} data-testid="input-title" className="text-lg" />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    {/* SKU z wyróżnionym tłem */}
                    <FormField
                      control={form.control}
                      name="sku"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel className="text-sm font-semibold">SKU (identyfikator) *</FormLabel>
                          <FormControl>
                            <div className="bg-muted/50 p-3 rounded-md border border-muted-foreground/20">
                              <Input {...field} data-testid="input-sku" className="bg-background" />
                            </div>
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    {/* Layout: Zdjęcie + Parametry */}
                    <div className="flex flex-col lg:flex-row gap-4">
                      {/* Pierwsze zdjęcie produktu */}
                      <div className="w-full lg:w-64 flex-shrink-0">
                        <div className="space-y-3">
                          <p className="text-sm font-medium">Zdjęcie podglądowe</p>
                          {product?.images && product.images.length > 0 ? (
                            <div className="aspect-square rounded-lg border bg-muted overflow-hidden max-w-[280px] mx-auto lg:mx-0">
                              <img
                                src={product.images[0].url}
                                alt={product.images[0].altText || product.title || 'Zdjęcie produktu'}
                                className="w-full h-full object-cover"
                              />
                            </div>
                          ) : (
                            <div className="aspect-square rounded-lg border bg-muted flex items-center justify-center text-muted-foreground max-w-[280px] mx-auto lg:mx-0">
                              <div className="text-center p-4">
                                <Package className="h-12 w-12 mx-auto mb-2 opacity-50" />
                                <p className="text-sm">Brak zdjęcia</p>
                                <p className="text-xs mt-1">Dodaj w zakładce Galeria</p>
                              </div>
                            </div>
                          )}
                          
                          {/* Metadata: daty utworzenia i aktualizacji */}
                          {!isNew && product && product.createdAt && product.updatedAt && (
                            <div className="space-y-2 p-3 bg-muted/30 rounded-md border text-xs max-w-[280px] mx-auto lg:mx-0" data-testid="product-metadata">
                              <div className="space-y-1">
                                <p className="font-medium text-muted-foreground">Utworzono:</p>
                                <p className="font-mono" data-testid="created-at">
                                  {new Date(product.createdAt).toLocaleString('pl-PL', {
                                    year: 'numeric',
                                    month: '2-digit',
                                    day: '2-digit',
                                    hour: '2-digit',
                                    minute: '2-digit',
                                  })}
                                </p>
                              </div>
                              <div className="space-y-1">
                                <p className="font-medium text-muted-foreground">Zaktualizowano:</p>
                                <p className="font-mono" data-testid="updated-at">
                                  {new Date(product.updatedAt).toLocaleString('pl-PL', {
                                    year: 'numeric',
                                    month: '2-digit',
                                    day: '2-digit',
                                    hour: '2-digit',
                                    minute: '2-digit',
                                  })}
                                </p>
                              </div>
                            </div>
                          )}
                          
                          {/* Opcje kolorystyczne (tylko do odczytu) */}
                          {!isNew && product?.colorOptions && product.colorOptions.length > 0 && (
                            <div className="space-y-2 p-3 bg-muted/30 rounded-md border text-xs max-w-[280px] mx-auto lg:mx-0" data-testid="color-options-card">
                              <p className="font-medium text-muted-foreground">Opcje kolorystyczne:</p>
                              <div className="flex flex-wrap gap-1.5">
                                {product.colorOptions.map((option, index) => (
                                  <Badge 
                                    key={index} 
                                    variant="secondary"
                                    className="text-xs"
                                    data-testid={`color-option-${index}`}
                                  >
                                    {option}
                                  </Badge>
                                ))}
                              </div>
                              <p className="text-[10px] text-muted-foreground italic">
                                Z matrycy produktów
                              </p>
                            </div>
                          )}
                        </div>
                      </div>

                      {/* Parametry produktu */}
                      <div className="flex-1 space-y-2">
                        {/* Długość + Szerokość */}
                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-2">
                          <FormField
                            control={form.control}
                            name="length"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Długość</FormLabel>
                                <FormControl>
                                  <DictionaryCombobox
                                    items={dimensionLengths}
                                    value={field.value?.toString() || ""}
                                    onChange={(value) => field.onChange(value ? parseFloat(value) : undefined)}
                                    placeholder="Wybierz"
                                    searchPlaceholder="Szukaj długości..."
                                    testId="select-length"
                                    valueField="code"
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />

                          <FormField
                            control={form.control}
                            name="width"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Szerokość</FormLabel>
                                <FormControl>
                                  <DictionaryCombobox
                                    items={dimensionWidths}
                                    value={field.value?.toString() || ""}
                                    onChange={(value) => field.onChange(value ? parseFloat(value) : undefined)}
                                    placeholder="Wybierz"
                                    searchPlaceholder="Szukaj szerokości..."
                                    testId="select-width"
                                    valueField="code"
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        </div>

                        {/* Wysokość + Waga */}
                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-2">
                          <FormField
                            control={form.control}
                            name="height"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Wysokość</FormLabel>
                                <FormControl>
                                  <DictionaryCombobox
                                    items={dimensionHeights}
                                    value={field.value?.toString() || ""}
                                    onChange={(value) => field.onChange(value ? parseFloat(value) : undefined)}
                                    placeholder="Wybierz"
                                    searchPlaceholder="Szukaj wysokości..."
                                    testId="select-height"
                                    valueField="code"
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />

                          <FormField
                            control={form.control}
                            name="weight"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Waga (kg)</FormLabel>
                                <FormControl>
                                  <Input
                                    {...field}
                                    type="number"
                                    step="0.01"
                                    placeholder="0.00"
                                    className="h-8"
                                    data-testid="input-weight"
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        </div>

                        {/* Kolor + Materiał (obowiązkowe) */}
                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-2">
                          <FormField
                            control={form.control}
                            name="color"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs font-semibold">Kolor *</FormLabel>
                                <div className="bg-muted/30 rounded-md">
                                  <FormControl>
                                    <DictionaryCombobox
                                      items={colors}
                                      value={field.value || ""}
                                      onChange={field.onChange}
                                      placeholder="Wybierz kolor"
                                      searchPlaceholder="Szukaj koloru..."
                                      testId="select-color"
                                      displayField="name"
                                    />
                                  </FormControl>
                                </div>
                                <FormMessage />
                              </FormItem>
                            )}
                          />

                          <FormField
                            control={form.control}
                            name="material"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs font-semibold">Materiał *</FormLabel>
                                <div className="bg-muted/30 rounded-md">
                                  <FormControl>
                                    <DictionaryCombobox
                                      items={materials}
                                      value={field.value || ""}
                                      onChange={field.onChange}
                                      placeholder="Wybierz materiał"
                                      searchPlaceholder="Szukaj materiału..."
                                      testId="select-material"
                                    />
                                  </FormControl>
                                </div>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        </div>

                        {/* Rodzaj + Grupa produktu */}
                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-2">
                          <FormField
                            control={form.control}
                            name="productType"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Rodzaj mebla</FormLabel>
                                <FormControl>
                                  <DictionaryCombobox
                                    items={productTypes}
                                    value={field.value || ""}
                                    onChange={field.onChange}
                                    placeholder="Wybierz rodzaj"
                                    searchPlaceholder="Szukaj rodzaju..."
                                    testId="select-product-type"
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />

                          <FormField
                            control={form.control}
                            name="productGroup"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Grupa produktu</FormLabel>
                                <FormControl>
                                  <DictionaryCombobox
                                    items={productGroups}
                                    value={field.value || ""}
                                    onChange={field.onChange}
                                    placeholder="Wybierz grupę"
                                    searchPlaceholder="Szukaj grupy..."
                                    testId="select-product-group"
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        </div>

                        {/* Drzwi + Nóżki */}
                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-2">
                          <FormField
                            control={form.control}
                            name="doors"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Drzwi</FormLabel>
                                <FormControl>
                                  <DictionaryCombobox
                                    items={doorsOptions}
                                    value={field.value || ""}
                                    onChange={field.onChange}
                                    placeholder="Wybierz drzwi"
                                    searchPlaceholder="Szukaj drzwi..."
                                    testId="select-doors"
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />

                          <FormField
                            control={form.control}
                            name="legs"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Nóżki</FormLabel>
                                <FormControl>
                                  <DictionaryCombobox
                                    items={legsOptions}
                                    value={field.value || ""}
                                    onChange={field.onChange}
                                    placeholder="Wybierz nóżki"
                                    searchPlaceholder="Szukaj nóżek..."
                                    testId="select-legs"
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        </div>
                      </div>
                    </div>

                    {/* Cena + Waluta + Aktywny */}
                    <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-2">
                      <FormField
                        control={form.control}
                        name="basePrice"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel className="text-xs">Cena bazowa</FormLabel>
                            <FormControl>
                              <Input
                                {...field}
                                type="number"
                                step="0.01"
                                className="h-8"
                                data-testid="input-base-price"
                              />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="currency"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel className="text-xs">Waluta</FormLabel>
                            <FormControl>
                              <Input {...field} disabled className="h-8" data-testid="input-currency" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="isActive"
                        render={({ field }) => (
                          <FormItem className="flex flex-row items-center justify-between rounded-lg border py-1 px-2">
                            <div className="space-y-0">
                              <FormLabel className="text-xs">Aktywny</FormLabel>
                            </div>
                            <FormControl>
                              <Switch
                                checked={field.value}
                                onCheckedChange={field.onChange}
                                data-testid="switch-active"
                              />
                            </FormControl>
                          </FormItem>
                        )}
                      />
                    </div>

                    {/* Opisy - na końcu po wszystkich parametrach */}
                    <FormField
                      control={form.control}
                      name="shortDescription"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Krótki opis</FormLabel>
                          <FormControl>
                            <Textarea
                              {...field}
                              rows={3}
                              data-testid="textarea-short-description"
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="longDescriptionHtml"
                      render={({ field }) => (
                        <FormItem>
                          <div className="flex flex-col sm:flex-row items-start sm:items-center gap-2 sm:justify-between mb-2">
                            <FormLabel className="flex-shrink-0">Długi opis</FormLabel>
                            <div className="flex flex-wrap gap-1.5 sm:gap-2">
                              {/* Mode Toggle */}
                              <div className="flex gap-1 border rounded-md p-1">
                                <Button
                                  type="button"
                                  variant={descriptionMode === 'editor' ? 'default' : 'ghost'}
                                  size="sm"
                                  onClick={() => setDescriptionMode('editor')}
                                  data-testid="button-mode-editor"
                                  className="h-7 px-2"
                                >
                                  <Eye className="h-3 w-3 mr-1" />
                                  Edytor
                                </Button>
                                <Button
                                  type="button"
                                  variant={descriptionMode === 'code' ? 'default' : 'ghost'}
                                  size="sm"
                                  onClick={() => setDescriptionMode('code')}
                                  data-testid="button-mode-code"
                                  className="h-7 px-2"
                                >
                                  <Code2 className="h-3 w-3 mr-1" />
                                  Kod
                                </Button>
                              </div>
                              
                              <Button
                                type="button"
                                variant="outline"
                                size="sm"
                                onClick={() => setIsTemplatePickerOpen(true)}
                                data-testid="button-use-template"
                              >
                                <Package className="h-4 w-4 mr-2" />
                                Użyj szablonu
                              </Button>
                              <Button
                                type="button"
                                variant="outline"
                                size="sm"
                                onClick={() => setIsPreviewDialogOpen(true)}
                                disabled={!form.watch('longDescriptionHtml')}
                                data-testid="button-preview-description"
                              >
                                <Eye className="h-4 w-4 mr-2" />
                                Podgląd opisu
                              </Button>
                              {!isNew && (
                                <>
                                  <AIGenerationHistory
                                    productId={parseInt(id!)}
                                    onReuse={(description) => {
                                      form.setValue("longDescriptionHtml", description);
                                    }}
                                  />
                                  <AIGenerationDialog
                                    productId={parseInt(id!)}
                                    onGenerated={(description) => {
                                      form.setValue("longDescriptionHtml", description);
                                    }}
                                  />
                                </>
                              )}
                            </div>
                          </div>
                          <FormDescription>
                            {descriptionMode === 'editor' 
                              ? 'Edytor WYSIWYG z możliwością formatowania tekstu, dodawania tabel i obrazów'
                              : 'Tryb kodu HTML - bezpośrednia edycja znaczników HTML'}
                          </FormDescription>
                          <FormControl>
                            {descriptionMode === 'code' ? (
                              <Textarea
                                {...field}
                                rows={10}
                                placeholder="<p>Wprowadź szczegółowy opis produktu...</p>"
                                data-testid="textarea-long-description"
                                className="font-mono text-sm"
                              />
                            ) : (
                              <RichTextEditor
                                content={field.value || ''}
                                onContentChange={(html, json) => {
                                  field.onChange(html);
                                }}
                                placeholder="Wprowadź szczegółowy opis produktu..."
                                enableGalleryPicker={!isNew}
                                enableAddonPicker={!isNew}
                              />
                            )}
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </CardContent>
                </Card>
              </TabsContent>

              {/* Zakładka: Allegro */}
              <TabsContent value="allegro" className="space-y-4">
                <Card>
                  <CardHeader className="flex flex-row items-center justify-between gap-2 sm:gap-4 space-y-0 pb-2">
                    <div>
                      <CardTitle>Allegro - dane specyficzne</CardTitle>
                      <CardDescription>
                        Konfiguracja produktu dla platformy Allegro
                      </CardDescription>
                    </div>
                    <Button
                      type="button"
                      variant="outline"
                      size="sm"
                      onClick={() => copySharedToPlatform("allegro")}
                      data-testid="button-copy-to-allegro"
                    >
                      Kopiuj z Wspólne
                    </Button>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    <FormField
                      control={form.control}
                      name="allegro.customTitle"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Tytuł (nadpisany)</FormLabel>
                          <FormControl>
                            <Input {...field} data-testid="input-allegro-title" />
                          </FormControl>
                          <FormDescription>
                            Zostaw puste aby użyć tytułu wspólnego
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                      <FormField
                        control={form.control}
                        name="allegro.categoryId"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>ID kategorii</FormLabel>
                            <FormControl>
                              <Input {...field} data-testid="input-allegro-category-id" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="allegro.categoryName"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Nazwa kategorii</FormLabel>
                            <FormControl>
                              <Input {...field} data-testid="input-allegro-category-name" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                    </div>

                    <FormField
                      control={form.control}
                      name="allegro.description"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Opis</FormLabel>
                          <FormControl>
                            <Textarea
                              {...field}
                              rows={10}
                              data-testid="textarea-allegro-description"
                            />
                          </FormControl>
                          <FormDescription>
                            HTML z osadzonymi obrazami i renderami
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="allegro.customPrice"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Cena (nadpisana)</FormLabel>
                          <FormControl>
                            <Input
                              {...field}
                              type="number"
                              step="0.01"
                              data-testid="input-allegro-price"
                            />
                          </FormControl>
                          <FormDescription>
                            Zostaw puste aby użyć ceny bazowej
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="allegro.isPublished"
                      render={({ field }) => (
                        <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                          <div className="space-y-0.5">
                            <FormLabel className="text-base">Opublikowany</FormLabel>
                            <FormDescription>
                              Czy produkt jest opublikowany na Allegro
                            </FormDescription>
                          </div>
                          <FormControl>
                            <Switch
                              checked={field.value}
                              onCheckedChange={field.onChange}
                              data-testid="switch-allegro-published"
                            />
                          </FormControl>
                        </FormItem>
                      )}
                    />
                  </CardContent>
                </Card>
              </TabsContent>

              {/* Zakładka: Shoper */}
              <TabsContent value="shoper" className="space-y-4">
                <Card>
                  <CardHeader className="flex flex-row items-center justify-between gap-2 sm:gap-4 space-y-0 pb-2">
                    <div>
                      <CardTitle>Shoper - dane specyficzne</CardTitle>
                      <CardDescription>
                        Konfiguracja produktu dla platformy Shoper
                      </CardDescription>
                    </div>
                    <Button
                      type="button"
                      variant="outline"
                      size="sm"
                      onClick={() => copySharedToPlatform("shoper")}
                      data-testid="button-copy-to-shoper"
                    >
                      Kopiuj z Wspólne
                    </Button>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    <FormField
                      control={form.control}
                      name="shoper.customTitle"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Tytuł (nadpisany)</FormLabel>
                          <FormControl>
                            <Input {...field} data-testid="input-shoper-title" />
                          </FormControl>
                          <FormDescription>
                            Zostaw puste aby użyć tytułu wspólnego
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                      <FormField
                        control={form.control}
                        name="shoper.categoryId"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>ID kategorii</FormLabel>
                            <FormControl>
                              <Input {...field} data-testid="input-shoper-category-id" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="shoper.categoryName"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Nazwa kategorii</FormLabel>
                            <FormControl>
                              <Input {...field} data-testid="input-shoper-category-name" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                    </div>

                    <FormField
                      control={form.control}
                      name="shoper.description"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Opis</FormLabel>
                          <FormControl>
                            <Textarea
                              {...field}
                              rows={10}
                              data-testid="textarea-shoper-description"
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="shoper.customPrice"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Cena (nadpisana)</FormLabel>
                          <FormControl>
                            <Input
                              {...field}
                              type="number"
                              step="0.01"
                              data-testid="input-shoper-price"
                            />
                          </FormControl>
                          <FormDescription>
                            Zostaw puste aby użyć ceny bazowej
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="shoper.isPublished"
                      render={({ field }) => (
                        <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                          <div className="space-y-0.5">
                            <FormLabel className="text-base">Opublikowany</FormLabel>
                            <FormDescription>
                              Czy produkt jest opublikowany w Shoper
                            </FormDescription>
                          </div>
                          <FormControl>
                            <Switch
                              checked={field.value}
                              onCheckedChange={field.onChange}
                              data-testid="switch-shoper-published"
                            />
                          </FormControl>
                        </FormItem>
                      )}
                    />
                  </CardContent>
                </Card>
              </TabsContent>

              {/* Zakładka: Odoo */}
              <TabsContent value="odoo" className="space-y-4">
                <Card>
                  <CardHeader className="flex flex-row items-center justify-between gap-2 sm:gap-4 space-y-0 pb-2">
                    <div>
                      <CardTitle>Odoo - dane specyficzne</CardTitle>
                      <CardDescription>
                        Konfiguracja produktu dla systemu Odoo
                      </CardDescription>
                    </div>
                    <Button
                      type="button"
                      variant="outline"
                      size="sm"
                      onClick={() => copySharedToPlatform("odoo")}
                      data-testid="button-copy-to-odoo"
                    >
                      Kopiuj z Wspólne
                    </Button>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    <FormField
                      control={form.control}
                      name="odoo.customTitle"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Nazwa produktu (nadpisana)</FormLabel>
                          <FormControl>
                            <Input {...field} data-testid="input-odoo-title" />
                          </FormControl>
                          <FormDescription>
                            Zostaw puste aby użyć tytułu wspólnego
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                      <FormField
                        control={form.control}
                        name="odoo.categoryId"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>ID kategorii</FormLabel>
                            <FormControl>
                              <Input {...field} data-testid="input-odoo-category-id" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="odoo.categoryName"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Nazwa kategorii</FormLabel>
                            <FormControl>
                              <Input {...field} data-testid="input-odoo-category-name" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                    </div>

                    <FormField
                      control={form.control}
                      name="odoo.description"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Opis</FormLabel>
                          <FormControl>
                            <Textarea
                              {...field}
                              rows={5}
                              data-testid="textarea-odoo-description"
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="odoo.customPrice"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Cena (nadpisana)</FormLabel>
                          <FormControl>
                            <Input
                              {...field}
                              type="number"
                              step="0.01"
                              data-testid="input-odoo-price"
                            />
                          </FormControl>
                          <FormDescription>
                            Zostaw puste aby użyć ceny bazowej
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="odoo.isPublished"
                      render={({ field }) => (
                        <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                          <div className="space-y-0.5">
                            <FormLabel className="text-base">Zsynchronizowany</FormLabel>
                            <FormDescription>
                              Czy produkt jest zsynchronizowany z Odoo
                            </FormDescription>
                          </div>
                          <FormControl>
                            <Switch
                              checked={field.value}
                              onCheckedChange={field.onChange}
                              data-testid="switch-odoo-published"
                            />
                          </FormControl>
                        </FormItem>
                      )}
                    />
                  </CardContent>
                </Card>
              </TabsContent>

              {/* Zakładka: Galeria */}
              <TabsContent value="gallery" className="space-y-4">
                {!isNew && <ProductGallery productId={parseInt(id!)} />}
                {isNew && (
                  <Card>
                    <CardContent className="pt-6">
                      <div className="text-center text-muted-foreground py-12">
                        <p>Zapisz najpierw produkt, aby dodać zdjęcia</p>
                      </div>
                    </CardContent>
                  </Card>
                )}
              </TabsContent>

              {/* Zakładka: Dodatki */}
              <TabsContent value="addons" className="space-y-4">
                {!isNew && <ProductAddons productId={parseInt(id!)} />}
                {isNew && (
                  <Card>
                    <CardContent className="pt-6">
                      <div className="text-center text-muted-foreground py-12">
                        <p>Zapisz najpierw produkt, aby przypisać dodatki</p>
                      </div>
                    </CardContent>
                  </Card>
                )}
              </TabsContent>
            </Tabs>
          </form>
        </Form>

        {/* Template Picker Dialog */}
        <TemplatePickerDialog
          open={isTemplatePickerOpen}
          onOpenChange={setIsTemplatePickerOpen}
          onSelect={(template) => {
            form.setValue("longDescriptionHtml", template.htmlContent);
            toast({
              title: "Szablon zastosowany",
              description: `Treść szablonu "${template.name}" została skopiowana do opisu produktu.`,
            });
          }}
        />

        {/* Description Preview Dialog */}
        <Dialog open={isPreviewDialogOpen} onOpenChange={setIsPreviewDialogOpen}>
          <DialogContent className="max-w-[95vw] sm:max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Podgląd opisu produktu</DialogTitle>
              <DialogDescription>
                Podgląd jak opis będzie wyglądał na platformach sprzedażowych (Allegro/Shoper)
              </DialogDescription>
            </DialogHeader>
            <div className="mt-4">
              {/* Preview container with Allegro/Shoper-like styling */}
              <div 
                className="prose prose-sm max-w-none bg-white dark:bg-gray-900 text-gray-900 dark:text-gray-300 p-4 sm:p-6 rounded-lg border"
                style={{
                  fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif',
                  fontSize: '14px',
                  lineHeight: '1.6',
                }}
                dangerouslySetInnerHTML={{ 
                  __html: form.watch('longDescriptionHtml') || '<p class="text-muted-foreground">Brak opisu</p>' 
                }}
              />
            </div>
            <div className="flex justify-end gap-2 mt-4">
              <Button
                type="button"
                variant="outline"
                onClick={() => setIsPreviewDialogOpen(false)}
                data-testid="button-close-preview"
              >
                Zamknij
              </Button>
            </div>
          </DialogContent>
        </Dialog>
        </div>
      </div>
    </div>
  );
}

// Product Gallery Component
function ProductGallery({ productId }: { productId: number }) {
  const { toast } = useToast();
  const fileInputRef = useRef<HTMLInputElement>(null);
  const [editingImage, setEditingImage] = useState<ProductImage | null>(null);
  const [editAltText, setEditAltText] = useState("");
  const [editImageType, setEditImageType] = useState<string>("packshot");
  const [editTags, setEditTags] = useState<string[]>([]);
  const [draggedImage, setDraggedImage] = useState<ProductImage | null>(null);
  const [isDraggingFiles, setIsDraggingFiles] = useState(false);
  const [uploadingCount, setUploadingCount] = useState(0);

  // Fetch images
  const { data: images, isLoading } = useQuery<ProductImage[]>({
    queryKey: [`/api/catalog-products/${productId}/images`],
    enabled: !!productId,
  });

  // Fetch color dictionary for tag suggestions
  interface ProductCreatorDictionary {
    id: number;
    dictionaryType: string;
    code: string;
    name: string;
    readableName: string | null;
    isActive: boolean;
  }
  
  const { data: colors } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=color"],
  });

  // Upload mutation
  const uploadMutation = useMutation({
    mutationFn: async (formData: FormData) => {
      const response = await fetch(`/api/catalog-products/${productId}/images`, {
        method: "POST",
        body: formData,
        credentials: "include",
      });
      if (!response.ok) throw new Error("Upload failed");
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${productId}/images`] });
      queryClient.invalidateQueries({ predicate: (query) => 
        Array.isArray(query.queryKey) && 
        typeof query.queryKey[0] === 'string' && 
        query.queryKey[0].startsWith('/api/catalog-products')
      });
      toast({ title: "Zdjęcie dodane" });
      if (fileInputRef.current) {
        fileInputRef.current.value = "";
      }
    },
    onError: () => {
      toast({ title: "Błąd uploadu", variant: "destructive" });
    },
  });

  // Delete mutation
  const deleteMutation = useMutation({
    mutationFn: async (imageId: number) => {
      await apiRequest("DELETE", `/api/catalog-products/${productId}/images/${imageId}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${productId}/images`] });
      queryClient.invalidateQueries({ predicate: (query) => 
        Array.isArray(query.queryKey) && 
        typeof query.queryKey[0] === 'string' && 
        query.queryKey[0].startsWith('/api/catalog-products')
      });
      toast({ title: "Zdjęcie usunięte" });
    },
    onError: () => {
      toast({ title: "Błąd usuwania", variant: "destructive" });
    },
  });

  // Update mutation
  const updateMutation = useMutation({
    mutationFn: async ({ imageId, data }: { imageId: number; data: any }) => {
      await apiRequest("PATCH", `/api/catalog-products/${productId}/images/${imageId}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${productId}/images`] });
      queryClient.invalidateQueries({ predicate: (query) => 
        Array.isArray(query.queryKey) && 
        typeof query.queryKey[0] === 'string' && 
        query.queryKey[0].startsWith('/api/catalog-products')
      });
      toast({ title: "Zdjęcie zaktualizowane" });
      setEditingImage(null);
    },
    onError: () => {
      toast({ title: "Błąd aktualizacji", variant: "destructive" });
    },
  });

  const uploadFiles = async (files: FileList | File[]) => {
    const fileArray = Array.from(files);
    if (fileArray.length === 0) return;

    setUploadingCount(fileArray.length);
    let successCount = 0;
    let failedCount = 0;

    try {
      for (const file of fileArray) {
        try {
          const formData = new FormData();
          formData.append("image", file);
          formData.append("imageType", "packshot");
          formData.append("altText", "");
          formData.append("isPrimary", "false");

          const response = await fetch(`/api/catalog-products/${productId}/images`, {
            method: "POST",
            body: formData,
            credentials: "include",
          });

          if (response.ok) {
            successCount++;
          } else {
            failedCount++;
          }
        } catch (error) {
          failedCount++;
        }
      }

      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${productId}/images`] });
      queryClient.invalidateQueries({ predicate: (query) => 
        Array.isArray(query.queryKey) && 
        typeof query.queryKey[0] === 'string' && 
        query.queryKey[0].startsWith('/api/catalog-products')
      });
      
      if (successCount > 0 && failedCount === 0) {
        toast({ title: `${successCount} zdjęć dodanych` });
      } else if (successCount > 0 && failedCount > 0) {
        toast({ 
          title: `${successCount} zdjęć dodanych, ${failedCount} nieudanych`,
          variant: "default"
        });
      } else {
        toast({ title: "Wszystkie uploady nieudane", variant: "destructive" });
      }

      if (fileInputRef.current) {
        fileInputRef.current.value = "";
      }
    } finally {
      setUploadingCount(0);
    }
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files) {
      uploadFiles(e.target.files);
    }
  };

  const handleFileDrop = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDraggingFiles(false);

    const files = e.dataTransfer.files;
    if (files.length > 0) {
      uploadFiles(files);
    }
  };

  const handleDragEnter = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDraggingFiles(true);
  };

  const handleDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    
    // Only set to false if we're leaving the dropzone entirely
    const rect = e.currentTarget.getBoundingClientRect();
    const x = e.clientX;
    const y = e.clientY;
    
    if (x <= rect.left || x >= rect.right || y <= rect.top || y >= rect.bottom) {
      setIsDraggingFiles(false);
    }
  };

  const handleDragOverFiles = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
  };

  const handleEdit = (image: ProductImage) => {
    setEditingImage(image);
    setEditAltText(image.altText || "");
    setEditImageType(image.imageType || "packshot");
    setEditTags(image.tags || []);
  };

  const handleSaveEdit = () => {
    if (!editingImage) return;
    updateMutation.mutate({
      imageId: editingImage.id,
      data: {
        altText: editAltText,
        imageType: editImageType,
        tags: editTags,
      },
    });
  };

  const handleSetPrimary = (imageId: number) => {
    updateMutation.mutate({
      imageId,
      data: { isPrimary: true },
    });
  };

  const handleDragStart = (image: ProductImage) => {
    setDraggedImage(image);
  };

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
  };

  const handleDrop = async (targetImage: ProductImage) => {
    if (!draggedImage || draggedImage.id === targetImage.id) {
      setDraggedImage(null);
      return;
    }

    const sortedImages = [...(images || [])].sort((a, b) => a.sortOrder - b.sortOrder);
    const draggedIndex = sortedImages.findIndex(img => img.id === draggedImage.id);
    const targetIndex = sortedImages.findIndex(img => img.id === targetImage.id);

    const reordered = [...sortedImages];
    const [removed] = reordered.splice(draggedIndex, 1);
    reordered.splice(targetIndex, 0, removed);

    try {
      // Update sort order and set first image as primary
      await Promise.all(
        reordered.map((img, index) =>
          apiRequest("PATCH", `/api/catalog-products/${productId}/images/${img.id}`, {
            sortOrder: index,
            isPrimary: index === 0, // First image is always primary
          })
        )
      );
      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${productId}/images`] });
      queryClient.invalidateQueries({ predicate: (query) => 
        Array.isArray(query.queryKey) && 
        typeof query.queryKey[0] === 'string' && 
        query.queryKey[0].startsWith('/api/catalog-products')
      });
      toast({ title: "Kolejność zaktualizowana" });
    } catch (error) {
      toast({ title: "Błąd zmiany kolejności", variant: "destructive" });
    }

    setDraggedImage(null);
  };

  if (isLoading) {
    return (
      <Card>
        <CardContent className="pt-6">
          <div className="flex justify-center py-12">
            <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle>Galeria zdjęć</CardTitle>
        <CardDescription>
          Zarządzanie zdjęciami produktu (packshot, rendery, wizualizacje)
        </CardDescription>
      </CardHeader>
      <CardContent className="space-y-4">
        {/* Drag & Drop Zone */}
        <div
          onDrop={handleFileDrop}
          onDragOver={handleDragOverFiles}
          onDragEnter={handleDragEnter}
          onDragLeave={handleDragLeave}
          className={`border-2 border-dashed rounded-lg p-8 text-center transition-colors ${
            isDraggingFiles
              ? "border-primary bg-primary/5"
              : "border-muted-foreground/25 hover:border-muted-foreground/50"
          }`}
          data-testid="dropzone"
        >
          <input
            ref={fileInputRef}
            type="file"
            accept="image/*"
            multiple
            className="hidden"
            onChange={handleFileChange}
            data-testid="input-image-upload"
          />
          
          {uploadingCount > 0 ? (
            <div className="space-y-2">
              <Loader2 className="h-8 w-8 animate-spin mx-auto text-primary" />
              <p className="text-sm font-medium">Wgrywanie {uploadingCount} zdjęć...</p>
            </div>
          ) : (
            <>
              <Upload className="h-12 w-12 mx-auto mb-4 text-muted-foreground" />
              <p className="text-lg font-medium mb-2">
                Przeciągnij i upuść zdjęcia tutaj
              </p>
              <p className="text-sm text-muted-foreground mb-4">
                lub
              </p>
              <Button
                type="button"
                variant="outline"
                onClick={() => fileInputRef.current?.click()}
                disabled={uploadingCount > 0}
                data-testid="button-upload-image"
              >
                <Upload className="mr-2 h-4 w-4" />
                Wybierz pliki
              </Button>
              <p className="text-xs text-muted-foreground mt-2">
                Obsługiwane formaty: JPEG, PNG, WebP (max 5MB każde)
              </p>
            </>
          )}
        </div>

        {/* Images grid */}
        {!images || images.length === 0 ? (
          <div className="text-center text-muted-foreground py-8">
            <p className="text-sm">Brak zdjęć. Użyj powyższej strefy aby dodać zdjęcia.</p>
          </div>
        ) : (
          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
            {[...images].sort((a, b) => a.sortOrder - b.sortOrder).map((image, index) => (
              <div
                key={image.id}
                draggable
                onDragStart={() => handleDragStart(image)}
                onDragOver={handleDragOver}
                onDrop={() => handleDrop(image)}
                className={`relative group rounded-lg overflow-hidden hover-elevate cursor-move ${
                  draggedImage?.id === image.id ? "opacity-50" : ""
                } ${index === 0 ? "border-4 border-primary shadow-lg" : "border"}`}
                data-testid={`image-card-${image.id}`}
              >
                <div className="aspect-square bg-muted relative">
                  {/* Image number badge */}
                  <div className="absolute top-2 left-2 bg-background/90 backdrop-blur-sm text-foreground px-2.5 py-1.5 rounded-md text-sm font-bold shadow-md z-10">
                    #{index + 1}
                  </div>
                  
                  <img
                    src={image.url}
                    alt={image.altText || "Zdjęcie produktu"}
                    className="w-full h-full object-cover"
                    data-testid={`img-product-${image.id}`}
                  />
                  {index === 0 && (
                    <div className="absolute top-2 right-2 bg-primary text-primary-foreground px-3 py-2 rounded-md text-sm font-semibold flex items-center gap-1.5 shadow-lg">
                      <Star className="h-4 w-4 fill-current" />
                      GŁÓWNE
                    </div>
                  )}
                  <div className="absolute bottom-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity flex gap-1">
                    <Button
                      size="icon"
                      variant="secondary"
                      onClick={() => handleEdit(image)}
                      data-testid={`button-edit-${image.id}`}
                    >
                      <Edit2 className="h-4 w-4" />
                    </Button>
                    <Button
                      size="icon"
                      variant="destructive"
                      onClick={() => deleteMutation.mutate(image.id)}
                      disabled={deleteMutation.isPending}
                      data-testid={`button-delete-${image.id}`}
                    >
                      <X className="h-4 w-4" />
                    </Button>
                  </div>
                </div>
                <div className="p-2 space-y-1">
                  <div className="text-xs font-medium capitalize">{image.imageType}</div>
                  {image.altText && (
                    <div className="text-xs text-muted-foreground truncate">{image.altText}</div>
                  )}
                </div>
              </div>
            ))}
          </div>
        )}

        {/* Edit dialog */}
        {editingImage && (
          <div className="fixed inset-0 bg-black/50 flex items-center justify-center p-4 z-50">
            <Card className="w-full max-w-md">
              <CardHeader>
                <CardTitle>Edytuj zdjęcie</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <label className="text-sm font-medium">Typ zdjęcia</label>
                  <Select value={editImageType} onValueChange={setEditImageType}>
                    <SelectTrigger data-testid="select-image-type">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="packshot">Packshot</SelectItem>
                      <SelectItem value="render">Render</SelectItem>
                      <SelectItem value="visualization">Wizualizacja</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <label className="text-sm font-medium">ALT Text (SEO)</label>
                  <Input
                    value={editAltText}
                    onChange={(e) => setEditAltText(e.target.value)}
                    placeholder="Opis zdjęcia dla wyszukiwarek"
                    data-testid="input-alt-text"
                  />
                </div>
                <div>
                  <label className="text-sm font-medium">Tagi</label>
                  <TagInput
                    tags={editTags}
                    onChange={setEditTags}
                    suggestions={colors?.filter(c => c.isActive).map(c => c.readableName || c.name) || []}
                    placeholder="Dodaj tag..."
                  />
                </div>
                <div className="flex gap-2">
                  <Button
                    onClick={handleSaveEdit}
                    disabled={updateMutation.isPending}
                    data-testid="button-save-edit"
                  >
                    {updateMutation.isPending ? (
                      <>
                        <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                        Zapisywanie...
                      </>
                    ) : (
                      "Zapisz"
                    )}
                  </Button>
                  <Button
                    variant="outline"
                    onClick={() => setEditingImage(null)}
                    data-testid="button-cancel-edit"
                  >
                    Anuluj
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>
        )}
      </CardContent>
    </Card>
  );
}

// Product Addons Component
function ProductAddons({ productId }: { productId: number }) {
  const { toast } = useToast();
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [selectedAddonId, setSelectedAddonId] = useState<number | null>(null);

  // Fetch assigned addons
  const { data: assignments, isLoading: loadingAssignments } = useQuery<any[]>({
    queryKey: [`/api/catalog-products/${productId}/addons`],
    enabled: !!productId,
  });

  // Fetch all available addons
  const { data: allAddons } = useQuery<any[]>({
    queryKey: ["/api/product-addons"],
  });

  // Assign addon mutation
  const assignMutation = useMutation({
    mutationFn: async (addonId: number) => {
      await apiRequest("POST", `/api/catalog-products/${productId}/addons`, {
        addonId,
        sortOrder: (assignments?.length || 0),
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${productId}/addons`] });
      toast({ title: "Dodatek przypisany" });
      setIsDialogOpen(false);
      setSelectedAddonId(null);
    },
    onError: () => {
      toast({ title: "Błąd przypisywania dodatku", variant: "destructive" });
    },
  });

  // Unassign addon mutation
  const unassignMutation = useMutation({
    mutationFn: async (assignmentId: number) => {
      await apiRequest("DELETE", `/api/catalog-products/${productId}/addons/${assignmentId}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${productId}/addons`] });
      toast({ title: "Dodatek usunięty" });
    },
    onError: () => {
      toast({ title: "Błąd usuwania dodatku", variant: "destructive" });
    },
  });

  const handleAssign = () => {
    if (selectedAddonId) {
      assignMutation.mutate(selectedAddonId);
    }
  };

  const handleUnassign = (assignmentId: number, addonName: string) => {
    if (window.confirm(`Czy na pewno chcesz usunąć dodatek "${addonName}" z tego produktu?`)) {
      unassignMutation.mutate(assignmentId);
    }
  };

  // Get available addons (not yet assigned)
  const assignedAddonIds = new Set(assignments?.map((a) => a.addonId) || []);
  const availableAddons = allAddons?.filter((addon) => !assignedAddonIds.has(addon.id)) || [];

  const ADDON_TYPE_LABELS: Record<string, string> = {
    fabric: "Tkanina",
    board: "Płyta",
    certificate: "Certyfikat",
    accessory: "Akcesorium",
    component: "Komponent",
  };

  if (loadingAssignments) {
    return (
      <Card>
        <CardContent className="pt-6">
          <div className="flex justify-center py-12">
            <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <div>
            <CardTitle>Dodatki produktu</CardTitle>
            <CardDescription>
              Przypisane tkaniny, płyty, certyfikaty i akcesoria
            </CardDescription>
          </div>
          <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
            <DialogTrigger asChild>
              <Button data-testid="button-add-addon-assignment">
                <Plus className="mr-2 h-4 w-4" />
                Przypisz dodatek
              </Button>
            </DialogTrigger>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>Przypisz dodatek</DialogTitle>
                <DialogDescription>
                  Wybierz dodatek do przypisania do tego produktu
                </DialogDescription>
              </DialogHeader>
              <div className="space-y-4">
                {availableAddons.length === 0 ? (
                  <div className="text-center text-muted-foreground py-8">
                    <p>Wszystkie dodatki są już przypisane</p>
                    <p className="text-sm mt-2">
                      Możesz utworzyć nowe dodatki w sekcji "Dodatki"
                    </p>
                  </div>
                ) : (
                  <>
                    <Select value={selectedAddonId?.toString()} onValueChange={(val) => setSelectedAddonId(parseInt(val))}>
                      <SelectTrigger data-testid="select-addon">
                        <SelectValue placeholder="Wybierz dodatek" />
                      </SelectTrigger>
                      <SelectContent>
                        {availableAddons.map((addon) => (
                          <SelectItem key={addon.id} value={addon.id.toString()}>
                            {ADDON_TYPE_LABELS[addon.addonType] || addon.addonType}: {addon.name}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <div className="flex justify-end gap-2">
                      <Button
                        variant="outline"
                        onClick={() => {
                          setIsDialogOpen(false);
                          setSelectedAddonId(null);
                        }}
                      >
                        Anuluj
                      </Button>
                      <Button
                        onClick={handleAssign}
                        disabled={!selectedAddonId || assignMutation.isPending}
                        data-testid="button-confirm-assign"
                      >
                        {assignMutation.isPending && (
                          <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                        )}
                        Przypisz
                      </Button>
                    </div>
                  </>
                )}
              </div>
            </DialogContent>
          </Dialog>
        </div>
      </CardHeader>
      <CardContent>
        {!assignments || assignments.length === 0 ? (
          <div className="text-center text-muted-foreground py-12">
            <Package className="h-12 w-12 mx-auto mb-4 text-muted-foreground" />
            <p>Brak przypisanych dodatków</p>
            <p className="text-sm mt-2">
              Kliknij "Przypisz dodatek" aby dodać pierwszy dodatek
            </p>
          </div>
        ) : (
          <div className="space-y-3">
            {assignments.map((assignment) => (
              <div
                key={assignment.id}
                className="flex items-center justify-between p-4 border rounded-lg hover-elevate"
                data-testid={`addon-assignment-${assignment.id}`}
              >
                <div className="flex items-center gap-4">
                  <Package className="h-5 w-5 text-muted-foreground" />
                  <div>
                    <div className="flex items-center gap-2">
                      <span className="font-medium">{assignment.name}</span>
                      <Badge variant="outline">
                        {ADDON_TYPE_LABELS[assignment.addonType] || assignment.addonType}
                      </Badge>
                    </div>
                    {assignment.description && (
                      <p className="text-sm text-muted-foreground mt-1">
                        {assignment.description}
                      </p>
                    )}
                    {assignment.code && (
                      <p className="text-xs text-muted-foreground mt-1">
                        Kod: {assignment.code}
                      </p>
                    )}
                  </div>
                </div>
                <Button
                  size="icon"
                  variant="ghost"
                  onClick={() => handleUnassign(assignment.id, assignment.name)}
                  disabled={unassignMutation.isPending}
                  data-testid={`button-unassign-${assignment.id}`}
                >
                  <Trash2 className="h-4 w-4" />
                </Button>
              </div>
            ))}
          </div>
        )}
      </CardContent>
    </Card>
  );
}
