import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Sparkles, Save, CheckCircle2, Info } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";

interface AISettings {
  id: number;
  systemInstructions: string;
  productMaterialInfo: string;
  edgeBandInfo: string;
  includeProductTable: boolean;
  productTableHeader: string;
  topBannerImageId: number | null;
  bottomBannerImageId: number | null;
  customInstructions: string;
  promptIntro: string;
  promptFeatures: string;
  promptSafety: string;
  promptCare: string;
  promptWarranty: string;
  createdAt: string;
  updatedAt: string;
}

interface ProductImage {
  id: number;
  url: string;
  altText: string;
  imageType: string;
  filename: string;
}

export default function AISettings() {
  const { toast } = useToast();
  
  const [systemInstructions, setSystemInstructions] = useState("");
  const [productMaterialInfo, setProductMaterialInfo] = useState("");
  const [edgeBandInfo, setEdgeBandInfo] = useState("");
  const [includeProductTable, setIncludeProductTable] = useState(true);
  const [productTableHeader, setProductTableHeader] = useState("Wymiary produktu");
  const [topBannerImageId, setTopBannerImageId] = useState<number | null>(null);
  const [bottomBannerImageId, setBottomBannerImageId] = useState<number | null>(null);
  const [customInstructions, setCustomInstructions] = useState("");
  const [promptIntro, setPromptIntro] = useState("");
  const [promptFeatures, setPromptFeatures] = useState("");
  const [promptSafety, setPromptSafety] = useState("");
  const [promptCare, setPromptCare] = useState("");
  const [promptWarranty, setPromptWarranty] = useState("");

  // Fetch AI settings
  const { data: settings, isLoading: settingsLoading } = useQuery<AISettings>({
    queryKey: ["/api/ai-settings"],
  });

  // Fetch available images for banner selection
  const { data: images, isLoading: imagesLoading } = useQuery<ProductImage[]>({
    queryKey: ["/api/product-images"],
  });

  // Load settings into state when fetched
  useEffect(() => {
    if (settings) {
      setSystemInstructions(settings.systemInstructions || "");
      setProductMaterialInfo(settings.productMaterialInfo || "");
      setEdgeBandInfo(settings.edgeBandInfo || "");
      setIncludeProductTable(settings.includeProductTable !== false);
      setProductTableHeader(settings.productTableHeader || "Wymiary produktu");
      setTopBannerImageId(settings.topBannerImageId);
      setBottomBannerImageId(settings.bottomBannerImageId);
      setCustomInstructions(settings.customInstructions || "");
      setPromptIntro(settings.promptIntro || "");
      setPromptFeatures(settings.promptFeatures || "");
      setPromptSafety(settings.promptSafety || "");
      setPromptCare(settings.promptCare || "");
      setPromptWarranty(settings.promptWarranty || "");
    }
  }, [settings]);

  const saveSettingsMutation = useMutation({
    mutationFn: async (data: {
      systemInstructions: string;
      productMaterialInfo: string;
      edgeBandInfo: string;
      includeProductTable: boolean;
      productTableHeader: string;
      topBannerImageId: number | null;
      bottomBannerImageId: number | null;
      customInstructions: string;
      promptIntro: string;
      promptFeatures: string;
      promptSafety: string;
      promptCare: string;
      promptWarranty: string;
    }) => {
      return await apiRequest("PUT", "/api/ai-settings", data);
    },
    onSuccess: () => {
      toast({
        title: "Ustawienia zapisane",
        description: "Konfiguracja promptów AI została zaktualizowana.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/ai-settings"] });
    },
    onError: (error) => {
      toast({
        title: "Błąd zapisu",
        description: "Nie udało się zapisać ustawień AI.",
        variant: "destructive",
      });
      console.error("Save settings error:", error);
    },
  });

  const handleSave = () => {
    saveSettingsMutation.mutate({
      systemInstructions,
      productMaterialInfo,
      edgeBandInfo,
      includeProductTable,
      productTableHeader,
      topBannerImageId,
      bottomBannerImageId,
      customInstructions,
      promptIntro,
      promptFeatures,
      promptSafety,
      promptCare,
      promptWarranty,
    });
  };

  if (settingsLoading) {
    return (
      <div className="container mx-auto p-6">
        <div className="animate-pulse space-y-4">
          <div className="h-8 bg-muted rounded w-1/3"></div>
          <div className="h-64 bg-muted rounded"></div>
        </div>
      </div>
    );
  }

  // Use all images for banner selection
  const bannerImages = images || [];

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold flex items-center gap-2" data-testid="text-page-title">
            <Sparkles className="w-8 h-8" />
            Ustawienia AI
          </h1>
          <p className="text-muted-foreground mt-1">
            Konfiguracja promptów i parametrów dla generowania opisów produktów
          </p>
        </div>
        <Button
          onClick={handleSave}
          disabled={saveSettingsMutation.isPending}
          data-testid="button-save-settings"
        >
          {saveSettingsMutation.isPending ? (
            <>
              <Save className="mr-2 h-4 w-4 animate-spin" />
              Zapisywanie...
            </>
          ) : (
            <>
              <Save className="mr-2 h-4 w-4" />
              Zapisz ustawienia
            </>
          )}
        </Button>
      </div>

      {/* System Instructions */}
      <Card>
        <CardHeader>
          <CardTitle>Instrukcje systemowe</CardTitle>
          <CardDescription>
            Główne instrukcje dla AI określające styl i ton generowanych opisów
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="system-instructions" data-testid="label-system-instructions">
              Prompt systemowy
            </Label>
            <Textarea
              id="system-instructions"
              value={systemInstructions}
              onChange={(e) => setSystemInstructions(e.target.value)}
              placeholder="Jesteś ekspertem w tworzeniu profesjonalnych opisów produktów meblarskich..."
              className="min-h-[120px]"
              data-testid="textarea-system-instructions"
            />
            <p className="text-sm text-muted-foreground">
              <Info className="inline w-3 h-3 mr-1" />
              Ten tekst będzie używany jako systemowa instrukcja dla AI przy generowaniu opisów
            </p>
          </div>
        </CardContent>
      </Card>

      {/* Product Parameters */}
      <Card>
        <CardHeader>
          <CardTitle>Parametry produktu</CardTitle>
          <CardDescription>
            Domyślne wartości materiałów i wykończenia wstawiane do promptu
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="material-info" data-testid="label-material-info">
              Informacja o materiale
            </Label>
            <Input
              id="material-info"
              value={productMaterialInfo}
              onChange={(e) => setProductMaterialInfo(e.target.value)}
              placeholder="np. płyta meblowa 18mm renomowanych polskich firm"
              data-testid="input-material-info"
            />
          </div>
          
          <div className="space-y-2">
            <Label htmlFor="edge-band-info" data-testid="label-edge-band-info">
              Informacja o obrzeżu
            </Label>
            <Input
              id="edge-band-info"
              value={edgeBandInfo}
              onChange={(e) => setEdgeBandInfo(e.target.value)}
              placeholder="np. obrzeże ABS 0.8mm"
              data-testid="input-edge-band-info"
            />
          </div>
        </CardContent>
      </Card>

      {/* Product Table Settings */}
      <Card>
        <CardHeader>
          <CardTitle>Tabela wymiarów produktu</CardTitle>
          <CardDescription>
            Konfiguracja automatycznego dodawania tabeli z wymiarami do opisu
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="flex items-center justify-between">
            <div className="space-y-0.5">
              <Label htmlFor="include-table" data-testid="label-include-table">
                Dołącz tabelę wymiarów
              </Label>
              <p className="text-sm text-muted-foreground">
                Automatycznie dodaj tabelę HTML z długością, szerokością i wysokością
              </p>
            </div>
            <Switch
              id="include-table"
              checked={includeProductTable}
              onCheckedChange={setIncludeProductTable}
              data-testid="switch-include-table"
            />
          </div>

          {includeProductTable && (
            <div className="space-y-2">
              <Label htmlFor="table-header" data-testid="label-table-header">
                Nagłówek tabeli
              </Label>
              <Input
                id="table-header"
                value={productTableHeader}
                onChange={(e) => setProductTableHeader(e.target.value)}
                placeholder="Wymiary produktu"
                data-testid="input-table-header"
              />
            </div>
          )}
        </CardContent>
      </Card>

      {/* Banner Images */}
      <Card>
        <CardHeader>
          <CardTitle>Banery obrazkowe</CardTitle>
          <CardDescription>
            Wybierz obrazy automatycznie wstawiane na początku i końcu opisu
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="top-banner" data-testid="label-top-banner">
              Górny baner
            </Label>
            <Select
              value={topBannerImageId ? String(topBannerImageId) : "none"}
              onValueChange={(value) => setTopBannerImageId(value === "none" ? null : parseInt(value))}
            >
              <SelectTrigger id="top-banner" data-testid="select-top-banner">
                <SelectValue placeholder="Wybierz baner górny" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="none">Brak baneru</SelectItem>
                {bannerImages.map((img) => (
                  <SelectItem key={img.id} value={String(img.id)}>
                    {img.altText || img.filename || `Obraz ${img.id}`}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div className="space-y-2">
            <Label htmlFor="bottom-banner" data-testid="label-bottom-banner">
              Dolny baner
            </Label>
            <Select
              value={bottomBannerImageId ? String(bottomBannerImageId) : "none"}
              onValueChange={(value) => setBottomBannerImageId(value === "none" ? null : parseInt(value))}
            >
              <SelectTrigger id="bottom-banner" data-testid="select-bottom-banner">
                <SelectValue placeholder="Wybierz baner dolny" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="none">Brak baneru</SelectItem>
                {bannerImages.map((img) => (
                  <SelectItem key={img.id} value={String(img.id)}>
                    {img.altText || img.filename || `Obraz ${img.id}`}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </CardContent>
      </Card>

      {/* AI Section Prompts */}
      <Card>
        <CardHeader>
          <CardTitle>Prompty AI dla sekcji opisu</CardTitle>
          <CardDescription>
            Bazowe prompty dla różnych sekcji opisu produktu. W matrycy produktu możesz dodać szczegóły specyficzne dla danego produktu.
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-6">
          <div className="space-y-2">
            <Label htmlFor="prompt-intro" data-testid="label-prompt-intro">
              Wprowadzenie {"{ai-intro}"}
            </Label>
            <Textarea
              id="prompt-intro"
              value={promptIntro}
              onChange={(e) => setPromptIntro(e.target.value)}
              placeholder="Napisz profesjonalne, przyciągające uwagę wprowadzenie..."
              className="min-h-[100px]"
              data-testid="textarea-prompt-intro"
            />
            <p className="text-sm text-muted-foreground">
              <Info className="inline w-3 h-3 mr-1" />
              Bazowy prompt dla sekcji wprowadzenia - w matrycy możesz dodać szczegóły produktu
            </p>
          </div>

          <div className="space-y-2">
            <Label htmlFor="prompt-features" data-testid="label-prompt-features">
              Cechy produktu {"{ai-features}"}
            </Label>
            <Textarea
              id="prompt-features"
              value={promptFeatures}
              onChange={(e) => setPromptFeatures(e.target.value)}
              placeholder="Opisz najważniejsze cechy i zalety produktu..."
              className="min-h-[100px]"
              data-testid="textarea-prompt-features"
            />
            <p className="text-sm text-muted-foreground">
              <Info className="inline w-3 h-3 mr-1" />
              Bazowy prompt dla sekcji cech - AI wygeneruje punktowane zalety
            </p>
          </div>

          <div className="space-y-2">
            <Label htmlFor="prompt-safety" data-testid="label-prompt-safety">
              Bezpieczeństwo {"{ai-safety}"}
            </Label>
            <Textarea
              id="prompt-safety"
              value={promptSafety}
              onChange={(e) => setPromptSafety(e.target.value)}
              placeholder="Przedstaw kwestie bezpieczeństwa użytkowania..."
              className="min-h-[100px]"
              data-testid="textarea-prompt-safety"
            />
            <p className="text-sm text-muted-foreground">
              <Info className="inline w-3 h-3 mr-1" />
              Bazowy prompt dla sekcji bezpieczeństwa
            </p>
          </div>

          <div className="space-y-2">
            <Label htmlFor="prompt-care" data-testid="label-prompt-care">
              Pielęgnacja {"{ai-care}"}
            </Label>
            <Textarea
              id="prompt-care"
              value={promptCare}
              onChange={(e) => setPromptCare(e.target.value)}
              placeholder="Podaj praktyczne wskazówki dotyczące pielęgnacji..."
              className="min-h-[100px]"
              data-testid="textarea-prompt-care"
            />
            <p className="text-sm text-muted-foreground">
              <Info className="inline w-3 h-3 mr-1" />
              Bazowy prompt dla sekcji pielęgnacji i czyszczenia
            </p>
          </div>

          <div className="space-y-2">
            <Label htmlFor="prompt-warranty" data-testid="label-prompt-warranty">
              Gwarancja {"{ai-warranty}"}
            </Label>
            <Textarea
              id="prompt-warranty"
              value={promptWarranty}
              onChange={(e) => setPromptWarranty(e.target.value)}
              placeholder="Opisz warunki gwarancji producenta..."
              className="min-h-[100px]"
              data-testid="textarea-prompt-warranty"
            />
            <p className="text-sm text-muted-foreground">
              <Info className="inline w-3 h-3 mr-1" />
              Bazowy prompt dla sekcji gwarancji
            </p>
          </div>

          <div className="bg-muted p-4 rounded-md">
            <p className="text-sm font-medium mb-2">Jak to działa?</p>
            <ul className="text-sm text-muted-foreground space-y-1">
              <li>• W szablonie użyj tagów {"{ai-intro}"} {"{ai-features}"} itd.</li>
              <li>• W matrycy produktu możesz dodać szczegóły specyficzne dla produktu</li>
              <li>• AI połączy bazowy prompt + szczegóły = finalny opis sekcji</li>
            </ul>
          </div>
        </CardContent>
      </Card>

      {/* Custom Instructions */}
      <Card>
        <CardHeader>
          <CardTitle>Dodatkowe instrukcje</CardTitle>
          <CardDescription>
            Niestandardowe wskazówki dla AI (np. jak interpretować wymiary)
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="custom-instructions" data-testid="label-custom-instructions">
              Własne instrukcje
            </Label>
            <Textarea
              id="custom-instructions"
              value={customInstructions}
              onChange={(e) => setCustomInstructions(e.target.value)}
              placeholder="Pamiętaj: DŁUGOŚĆ to wymiar poziomy (szerokość), SZEROKOŚĆ to wymiar głębokości..."
              className="min-h-[100px]"
              data-testid="textarea-custom-instructions"
            />
            <p className="text-sm text-muted-foreground">
              <Info className="inline w-3 h-3 mr-1" />
              Te instrukcje będą dodane na końcu prompta systemowego
            </p>
          </div>
        </CardContent>
      </Card>

      {/* Save Button (Bottom) */}
      <div className="flex justify-end">
        <Button
          onClick={handleSave}
          disabled={saveSettingsMutation.isPending}
          size="lg"
          data-testid="button-save-settings-bottom"
        >
          {saveSettingsMutation.isPending ? (
            <>
              <Save className="mr-2 h-4 w-4 animate-spin" />
              Zapisywanie...
            </>
          ) : (
            <>
              <CheckCircle2 className="mr-2 h-4 w-4" />
              Zapisz wszystkie ustawienia
            </>
          )}
        </Button>
      </div>
    </div>
  );
}
