import { useState, useCallback, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useLocation } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Checkbox } from "@/components/ui/checkbox";
import { useToast } from "@/hooks/use-toast";
import { Plus, Pencil, Trash2, Save, X, Image as ImageIcon, Upload, List, LayoutGrid, Check, XIcon, ChevronDown, ChevronRight, Copy } from "lucide-react";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { CSVImportDialog } from "@/components/csv-import-dialog";
import { AccessoryGroupComboboxWithAdd } from "@/components/accessory-group-combobox-with-add";
import { useDropzone } from "react-dropzone";

type AccessoryGroup = {
  id: number;
  name: string;
  code: string;
  category: string | null;
  description: string | null;
  displayOrder: number;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
};

type Accessory = {
  id: number;
  groupId: number | null;
  name: string;
  code: string;
  alpmaCode: string | null;
  description: string | null;
  imageUrl: string | null;
  displayOrder: number;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
  groupName?: string;
  groupCode?: string;
};

// CSV Template for accessories
const CSV_TEMPLATE = `group_code,name,code,description,image_url,display_order
fabric,Tkanina Przykładowa,TKN-001,Opis tkaniny,https://example.com/image.jpg,1
board,Płyta Przykładowa,PLY-001,Opis płyty,,2`;

export default function AccessoriesPage() {
  const { toast } = useToast();
  const [, navigate] = useLocation();
  const [selectedGroupId, setSelectedGroupId] = useState<number | null>(null);
  const [isGroupDialogOpen, setIsGroupDialogOpen] = useState(false);
  const [isAccessoryDialogOpen, setIsAccessoryDialogOpen] = useState(false);
  const [editingGroup, setEditingGroup] = useState<AccessoryGroup | null>(null);
  const [editingAccessory, setEditingAccessory] = useState<Accessory | null>(null);
  
  // Bulk import state
  const [bulkImportGroupId, setBulkImportGroupId] = useState<string>("");
  const [bulkImportFiles, setBulkImportFiles] = useState<File[]>([]);
  
  // Group filter state
  const [groupFilter, setGroupFilter] = useState<string>("");
  
  // View mode state (compact vs normal) - default to compact
  const [isCompactView, setIsCompactView] = useState<boolean>(true);
  
  // Category collapse state and "show all" mode
  const [collapsedCategories, setCollapsedCategories] = useState<Set<string>>(new Set());
  const [showAllGroups, setShowAllGroups] = useState<boolean>(false);
  
  // Bulk selection state
  const [selectedAccessoryIds, setSelectedAccessoryIds] = useState<Set<number>>(new Set());
  
  // Bulk edit state
  const [isBulkEditDialogOpen, setIsBulkEditDialogOpen] = useState(false);
  const [bulkEditIsActive, setBulkEditIsActive] = useState<boolean | null>(null);
  const [bulkEditDescription, setBulkEditDescription] = useState<string>("");
  
  // Show inactive accessories state
  const [showInactive, setShowInactive] = useState(false);
  
  // Duplicate accessory dialog state
  const [isDuplicateDialogOpen, setIsDuplicateDialogOpen] = useState(false);
  const [duplicatingAccessory, setDuplicatingAccessory] = useState<Accessory | null>(null);
  const [duplicateTargetGroupId, setDuplicateTargetGroupId] = useState<string>("");
  
  // Duplicate group dialog state
  const [isDuplicateGroupDialogOpen, setIsDuplicateGroupDialogOpen] = useState(false);
  const [duplicatingGroup, setDuplicatingGroup] = useState<AccessoryGroup | null>(null);
  const [duplicateGroupName, setDuplicateGroupName] = useState<string>("");
  
  // Clear selection when group filter changes
  useEffect(() => {
    setSelectedAccessoryIds(new Set());
  }, [selectedGroupId]);

  // Fetch groups
  const { data: groups = [], isLoading: groupsLoading } = useQuery<AccessoryGroup[]>({
    queryKey: ["/api/accessory-groups"],
  });

  // Fetch accessories (with showInactive parameter)
  const { data: accessories = [], isLoading: accessoriesLoading } = useQuery<Accessory[]>({
    queryKey: ["/api/accessories", { showInactive }],
  });

  // Filter and sort groups
  const filteredAndSortedGroups = groups
    .filter((g) => 
      groupFilter === "" || 
      g.name.toLowerCase().includes(groupFilter.toLowerCase()) ||
      g.code.toLowerCase().includes(groupFilter.toLowerCase())
    )
    .sort((a, b) => a.name.localeCompare(b.name, 'pl'));
  
  // Group groups by category
  const groupedByCategory = filteredAndSortedGroups.reduce((acc, group) => {
    const category = group.category || "Bez kategorii";
    if (!acc[category]) {
      acc[category] = [];
    }
    acc[category].push(group);
    return acc;
  }, {} as Record<string, AccessoryGroup[]>);
  
  const categories = Object.keys(groupedByCategory).sort();

  // Filter accessories by selected group
  const filteredAccessories = selectedGroupId
    ? accessories.filter((a) => a.groupId === selectedGroupId)
    : accessories;
  
  // Toggle category collapse
  const toggleCategory = (category: string) => {
    setCollapsedCategories(prev => {
      const newSet = new Set(prev);
      if (newSet.has(category)) {
        newSet.delete(category);
      } else {
        newSet.add(category);
      }
      return newSet;
    });
  };
  
  // Toggle accessory selection
  const toggleAccessorySelection = (id: number) => {
    setSelectedAccessoryIds(prev => {
      const newSet = new Set(prev);
      if (newSet.has(id)) {
        newSet.delete(id);
      } else {
        newSet.add(id);
      }
      return newSet;
    });
  };
  
  // Select all / deselect all accessories
  const toggleSelectAll = () => {
    if (selectedAccessoryIds.size === filteredAccessories.length) {
      setSelectedAccessoryIds(new Set());
    } else {
      setSelectedAccessoryIds(new Set(filteredAccessories.map(a => a.id)));
    }
  };

  // Create/Update Group Mutation
  const groupMutation = useMutation({
    mutationFn: async (data: Partial<AccessoryGroup>) => {
      if (editingGroup) {
        return apiRequest("PUT", `/api/accessory-groups/${editingGroup.id}`, data);
      } else {
        return apiRequest("POST", "/api/accessory-groups", data);
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessory-groups"] });
      setIsGroupDialogOpen(false);
      setEditingGroup(null);
      toast({
        title: "Sukces",
        description: editingGroup ? "Grupa zaktualizowana" : "Grupa utworzona",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zapisać grupy",
        variant: "destructive",
      });
    },
  });

  // Delete Group Mutation
  const deleteGroupMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/accessory-groups/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessory-groups"] });
      toast({ title: "Sukces", description: "Grupa usunięta" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się usunąć grupy",
        variant: "destructive",
      });
    },
  });

  // Duplicate Group Mutation
  const duplicateGroupMutation = useMutation({
    mutationFn: async ({ id, name }: { id: number; name: string }) => {
      return apiRequest("POST", `/api/accessory-groups/${id}/duplicate`, { name });
    },
    onSuccess: (data: any) => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessory-groups"] });
      queryClient.invalidateQueries({ queryKey: ["/api/accessories"] });
      setIsDuplicateGroupDialogOpen(false);
      setDuplicatingGroup(null);
      setDuplicateGroupName("");
      toast({
        title: "Sukces",
        description: `Grupa zduplikowana z ${data.copiedAccessoriesCount} akcesoriami`,
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zduplikować grupy",
        variant: "destructive",
      });
    },
  });

  // Create/Update Accessory Mutation
  const accessoryMutation = useMutation({
    mutationFn: async (data: Partial<Accessory>) => {
      if (editingAccessory) {
        return apiRequest("PUT", `/api/accessories/${editingAccessory.id}`, data);
      } else {
        return apiRequest("POST", "/api/accessories", data);
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessories"] });
      setIsAccessoryDialogOpen(false);
      setEditingAccessory(null);
      toast({
        title: "Sukces",
        description: editingAccessory ? "Akcesorium zaktualizowane" : "Akcesorium utworzone",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zapisać akcesorium",
        variant: "destructive",
      });
    },
  });

  // Delete Accessory Mutation
  const deleteAccessoryMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/accessories/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessories"] });
      toast({ title: "Sukces", description: "Akcesorium usunięte" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się usunąć akcesorium",
        variant: "destructive",
      });
    },
  });

  // Toggle Active Status Mutation
  const toggleActiveMutation = useMutation({
    mutationFn: async ({ id, isActive }: { id: number; isActive: boolean }) => {
      return apiRequest("PUT", `/api/accessories/${id}`, { isActive });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessories"] });
      toast({ title: "Sukces", description: "Status zmieniony" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zmienić statusu",
        variant: "destructive",
      });
    },
  });
  
  // Duplicate Accessory Mutation
  const duplicateAccessoryMutation = useMutation({
    mutationFn: async ({ id, targetGroupId }: { id: number; targetGroupId: number }) => {
      const response = await apiRequest("POST", `/api/accessories/${id}/duplicate`, { targetGroupId });
      return response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessories"] });
      setIsDuplicateDialogOpen(false);
      setDuplicatingAccessory(null);
      setDuplicateTargetGroupId("");
      toast({ 
        title: "Sukces", 
        description: `Akcesorium zduplikowane do grupy` 
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zduplikować akcesorium",
        variant: "destructive",
      });
    },
  });
  
  // Bulk Delete Accessories Mutation
  const bulkDeleteMutation = useMutation({
    mutationFn: async (ids: number[]) => {
      const response = await apiRequest("POST", "/api/accessories/bulk-delete", { accessoryIds: ids });
      return response.json();
    },
    onSuccess: () => {
      // Invalidate all accessories queries regardless of parameters
      queryClient.invalidateQueries({ queryKey: ["/api/accessories"], refetchType: "all" });
      setSelectedAccessoryIds(new Set());
      toast({ 
        title: "Sukces", 
        description: `Usunięto akcesoria` 
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się usunąć akcesoriów",
        variant: "destructive",
      });
    },
  });

  // Bulk Edit Accessories Mutation
  const bulkEditMutation = useMutation({
    mutationFn: async (data: { accessoryIds: number[], updates: { isActive?: boolean, description?: string } }) => {
      const response = await apiRequest("POST", "/api/accessories/bulk-edit", data);
      return response.json();
    },
    onSuccess: () => {
      // Invalidate all accessories queries regardless of parameters
      queryClient.invalidateQueries({ queryKey: ["/api/accessories"], refetchType: "all" });
      setSelectedAccessoryIds(new Set());
      setIsBulkEditDialogOpen(false);
      setBulkEditIsActive(null);
      setBulkEditDescription("");
      toast({ 
        title: "Sukces", 
        description: `Zaktualizowano akcesoria` 
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować akcesoriów",
        variant: "destructive",
      });
    },
  });

  const handleGroupSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    const formData = new FormData(e.currentTarget);
    groupMutation.mutate({
      name: formData.get("name") as string,
      code: formData.get("code") as string,
      category: formData.get("category") as string || null,
      description: formData.get("description") as string || null,
      displayOrder: parseInt(formData.get("displayOrder") as string) || 0,
    });
  };

  const handleBulkEditSubmit = () => {
    // Validate that accessories are selected
    if (selectedAccessoryIds.size === 0) {
      toast({
        title: "Błąd",
        description: "Nie zaznaczono żadnych akcesoriów",
        variant: "destructive",
      });
      return;
    }

    const updates: { isActive?: boolean, description?: string } = {};
    
    if (bulkEditIsActive !== null) {
      updates.isActive = bulkEditIsActive;
    }
    
    if (bulkEditDescription.trim()) {
      updates.description = bulkEditDescription.trim();
    }
    
    if (Object.keys(updates).length === 0) {
      toast({
        title: "Błąd",
        description: "Wybierz przynajmniej jedno pole do edycji",
        variant: "destructive",
      });
      return;
    }
    
    bulkEditMutation.mutate({
      accessoryIds: Array.from(selectedAccessoryIds),
      updates
    });
  };

  const handleAccessorySubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    const formData = new FormData(e.currentTarget);
    accessoryMutation.mutate({
      groupId: formData.get("groupId") ? parseInt(formData.get("groupId") as string) : null,
      name: formData.get("name") as string,
      code: formData.get("code") as string,
      alpmaCode: formData.get("alpmaCode") as string || null,
      description: formData.get("description") as string || null,
      imageUrl: formData.get("imageUrl") as string || null,
      displayOrder: parseInt(formData.get("displayOrder") as string) || 0,
    });
  };

  // Bulk import mutation
  const bulkImportMutation = useMutation({
    mutationFn: async () => {
      const formData = new FormData();
      bulkImportFiles.forEach((file) => {
        formData.append('images', file);
      });
      if (bulkImportGroupId && bulkImportGroupId !== "none") {
        formData.append('groupId', bulkImportGroupId);
      }

      const response = await fetch('/api/accessories/bulk-import', {
        method: 'POST',
        body: formData,
        credentials: 'include',
      });

      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(errorText || response.statusText);
      }

      return await response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessories"] });
      setBulkImportFiles([]);
      setBulkImportGroupId("");
      toast({
        title: "Sukces",
        description: `Zaimportowano ${data.created} akcesoriów${data.errors > 0 ? `, ${data.errors} błędów` : ''}`,
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaimportować akcesoriów",
        variant: "destructive",
      });
    },
  });

  // Drag & drop for bulk import
  const onDrop = useCallback((acceptedFiles: File[]) => {
    setBulkImportFiles(prev => [...prev, ...acceptedFiles]);
  }, []);

  const { getRootProps, getInputProps, isDragActive } = useDropzone({
    onDrop,
    accept: {
      'image/*': ['.png', '.jpg', '.jpeg', '.gif', '.webp']
    },
    multiple: true
  });

  const handleBulkImport = () => {
    if (bulkImportFiles.length === 0) {
      toast({
        title: "Błąd",
        description: "Wybierz przynajmniej jedno zdjęcie",
        variant: "destructive",
      });
      return;
    }
    bulkImportMutation.mutate();
  };

  const removeBulkFile = (index: number) => {
    setBulkImportFiles(prev => prev.filter((_, i) => i !== index));
  };

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold">Akcesoria</h1>
          <p className="text-sm text-muted-foreground">
            Zarządzaj grupami i akcesoriami produktów
          </p>
        </div>
      </div>

      {/* Bulk Import Section */}
      <Card>
        <CardHeader>
          <CardTitle className="text-base">Bulk Import Zdjęć</CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="bulk-group">Grupa (opcjonalnie)</Label>
              <Select value={bulkImportGroupId} onValueChange={setBulkImportGroupId}>
                <SelectTrigger id="bulk-group" data-testid="select-bulk-group">
                  <SelectValue placeholder="Wybierz grupę (lub zostaw puste)" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="none">Brak grupy</SelectItem>
                  {groups.map((group) => (
                    <SelectItem key={group.id} value={group.id.toString()}>
                      {group.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label>Przeciągnij zdjęcia</Label>
              <div
                {...getRootProps()}
                className={`border-2 border-dashed p-6 text-center cursor-pointer hover-elevate ${
                  isDragActive ? 'border-primary bg-primary/10' : 'border-border'
                }`}
                data-testid="dropzone-bulk-import"
              >
                <input {...getInputProps()} />
                <Upload className="h-8 w-8 mx-auto mb-2 text-muted-foreground" />
                {isDragActive ? (
                  <p className="text-sm">Upuść zdjęcia tutaj...</p>
                ) : (
                  <p className="text-sm text-muted-foreground">
                    Przeciągnij zdjęcia lub kliknij aby wybrać
                  </p>
                )}
              </div>
            </div>
          </div>

          {bulkImportFiles.length > 0 && (
            <div className="space-y-2">
              <div className="flex items-center justify-between">
                <Label>Wybrane pliki ({bulkImportFiles.length})</Label>
                <Button
                  size="sm"
                  variant="outline"
                  onClick={() => setBulkImportFiles([])}
                  data-testid="button-clear-bulk-files"
                >
                  <X className="h-4 w-4 mr-1" />
                  Wyczyść wszystko
                </Button>
              </div>
              <div className="grid grid-cols-2 md:grid-cols-4 gap-2">
                {bulkImportFiles.map((file, index) => (
                  <div
                    key={index}
                    className="relative border p-2 group hover-elevate"
                  >
                    <div className="aspect-square bg-muted flex items-center justify-center mb-1">
                      <ImageIcon className="h-8 w-8 text-muted-foreground" />
                    </div>
                    <p className="text-xs truncate" title={file.name}>{file.name}</p>
                    <Button
                      size="icon"
                      variant="destructive"
                      className="absolute top-1 right-1 h-6 w-6 opacity-0 group-hover:opacity-100"
                      onClick={() => removeBulkFile(index)}
                      data-testid={`button-remove-bulk-file-${index}`}
                    >
                      <X className="h-3 w-3" />
                    </Button>
                  </div>
                ))}
              </div>
              <div className="flex gap-2">
                <Button
                  onClick={handleBulkImport}
                  disabled={bulkImportMutation.isPending}
                  data-testid="button-bulk-import"
                >
                  {bulkImportMutation.isPending ? 'Importowanie...' : `Importuj ${bulkImportFiles.length} zdjęć`}
                </Button>
                <Button
                  variant="outline"
                  onClick={() => document.getElementById('bulk-file-input')?.click()}
                  disabled={bulkImportMutation.isPending}
                  data-testid="button-add-more-files"
                >
                  <Plus className="h-4 w-4 mr-1" />
                  Dodaj więcej zdjęć
                </Button>
                <input
                  id="bulk-file-input"
                  type="file"
                  multiple
                  accept="image/*"
                  className="hidden"
                  onChange={(e) => {
                    const files = Array.from(e.target.files || []);
                    setBulkImportFiles(prev => [...prev, ...files]);
                    e.target.value = ''; // Reset input
                  }}
                />
              </div>
            </div>
          )}
        </CardContent>
      </Card>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
        {/* Groups Section */}
        <div className="md:col-span-1">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between gap-1.5 space-y-0 pb-1 pt-2 px-3">
              <CardTitle className="text-sm font-medium">Grupy akcesoriów</CardTitle>
              <Button
                size="sm"
                onClick={() => {
                  setEditingGroup(null);
                  setIsGroupDialogOpen(true);
                }}
                data-testid="button-add-group"
              >
                <Plus className="h-4 w-4 mr-1" />
                Dodaj
              </Button>
            </CardHeader>
            <CardContent className="p-2">
              {groupsLoading ? (
                <div className="text-sm text-muted-foreground">Ładowanie...</div>
              ) : groups.length === 0 ? (
                <div className="text-sm text-muted-foreground">Brak grup</div>
              ) : (
                <div className="space-y-1">
                  {/* Compact filter input and Show All button */}
                  <div className="flex gap-1 mb-1">
                    <Input
                      placeholder="Szukaj grupy..."
                      value={groupFilter}
                      onChange={(e) => setGroupFilter(e.target.value)}
                      className="h-8 text-sm flex-1"
                      data-testid="input-filter-groups"
                    />
                    <Button
                      size="sm"
                      variant={showAllGroups ? "default" : "outline"}
                      onClick={() => setShowAllGroups(!showAllGroups)}
                      className="h-8 text-xs"
                      data-testid="button-show-all-groups"
                    >
                      {showAllGroups ? "Zwiń" : "Wszystkie"}
                    </Button>
                  </div>
                  
                  {filteredAndSortedGroups.length === 0 && groupFilter !== "" ? (
                    <div className="text-sm text-muted-foreground text-center py-2">
                      Nie znaleziono grup
                    </div>
                  ) : (
                    <div className="space-y-2">
                      {categories.map((category) => {
                        const groupsInCategory = groupedByCategory[category];
                        const isCollapsed = !showAllGroups && collapsedCategories.has(category);
                        
                        return (
                          <div key={category} className="space-y-1">
                            {/* Category header */}
                            <div
                              className="flex items-center gap-1 py-1 px-2 bg-muted hover-elevate active-elevate-2 cursor-pointer"
                              onClick={() => toggleCategory(category)}
                              data-testid={`category-header-${category}`}
                            >
                              {isCollapsed ? (
                                <ChevronRight className="h-3 w-3" />
                              ) : (
                                <ChevronDown className="h-3 w-3" />
                              )}
                              <span className="text-xs font-semibold">{category}</span>
                              <span className="text-xs text-muted-foreground ml-auto">
                                ({groupsInCategory.length})
                              </span>
                            </div>
                            
                            {/* Groups in category */}
                            {!isCollapsed && groupsInCategory.map((group) => (
                              <div
                                key={group.id}
                                className={`p-2 border ml-2 hover-elevate active-elevate-2 cursor-pointer ${
                                  selectedGroupId === group.id ? "bg-accent" : ""
                                }`}
                                onClick={() => setSelectedGroupId(group.id)}
                                data-testid={`group-item-${group.id}`}
                              >
                                <div className="flex items-start justify-between">
                                  <div className="flex-1">
                                    <div className="font-medium text-sm">{group.name}</div>
                                    <div className="text-xs text-muted-foreground">
                                      Kod: {group.code}
                                    </div>
                                  </div>
                                  <div className="flex gap-1">
                                    <Button
                                      size="icon"
                                      variant="ghost"
                                      onClick={(e) => {
                                        e.stopPropagation();
                                        setEditingGroup(group);
                                        setIsGroupDialogOpen(true);
                                      }}
                                      data-testid={`button-edit-group-${group.id}`}
                                    >
                                      <Pencil className="h-3 w-3" />
                                    </Button>
                                    <Button
                                      size="icon"
                                      variant="ghost"
                                      onClick={(e) => {
                                        e.stopPropagation();
                                        setDuplicatingGroup(group);
                                        setDuplicateGroupName(`${group.name} (kopia)`);
                                        setIsDuplicateGroupDialogOpen(true);
                                      }}
                                      data-testid={`button-duplicate-group-${group.id}`}
                                    >
                                      <Copy className="h-3 w-3" />
                                    </Button>
                                    <Button
                                      size="icon"
                                      variant="ghost"
                                      onClick={(e) => {
                                        e.stopPropagation();
                                        if (confirm("Czy na pewno usunąć tę grupę?")) {
                                          deleteGroupMutation.mutate(group.id);
                                        }
                                      }}
                                      data-testid={`button-delete-group-${group.id}`}
                                    >
                                      <Trash2 className="h-3 w-3" />
                                    </Button>
                                  </div>
                                </div>
                              </div>
                            ))}
                          </div>
                        );
                      })}
                    </div>
                  )}
                </div>
              )}
            </CardContent>
          </Card>
        </div>

        {/* Accessories Section */}
        <div className="md:col-span-2">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between gap-1.5 space-y-0 pb-1 pt-2 px-3">
              <div className="flex flex-col gap-1">
                <CardTitle className="text-sm font-medium">
                  {selectedGroupId
                    ? `Akcesoria w grupie: ${groups.find((g) => g.id === selectedGroupId)?.name || ""}`
                    : "Wszystkie akcesoria"}
                </CardTitle>
                <div className="flex items-center gap-2">
                  <Checkbox
                    id="show-inactive"
                    checked={showInactive}
                    onCheckedChange={(checked) => setShowInactive(checked as boolean)}
                    data-testid="checkbox-show-inactive"
                  />
                  <label htmlFor="show-inactive" className="text-xs text-muted-foreground cursor-pointer">
                    Pokaż nieaktywne
                  </label>
                </div>
              </div>
              <div className="flex gap-1.5">
                <Button
                  size="sm"
                  variant={isCompactView ? "default" : "outline"}
                  onClick={() => setIsCompactView(!isCompactView)}
                  data-testid="button-toggle-view"
                >
                  {isCompactView ? <List className="h-4 w-4" /> : <LayoutGrid className="h-4 w-4" />}
                </Button>
                <CSVImportDialog
                  endpoint="/api/accessories/import-csv"
                  templateCsv={CSV_TEMPLATE}
                  templateFilename="akcesoria_szablon.csv"
                  invalidateQueryKey={["/api/accessories"]}
                  title="Import akcesoriów z CSV"
                  description="Importuj wiele akcesoriów jednocześnie z pliku CSV. Pola: group_code, name, code, description, image_url, display_order"
                />
                <Button
                  size="sm"
                  onClick={() => navigate("/accessories/new/edit")}
                  data-testid="button-add-accessory"
                >
                  <Plus className="h-4 w-4 mr-1" />
                  Dodaj
                </Button>
              </div>
            </CardHeader>
            {/* Bulk Actions Bar */}
            {filteredAccessories.length > 0 && (
              <div className="px-3 pb-2 flex items-center gap-2 border-b">
                <div className="flex items-center gap-2">
                  <Checkbox
                    checked={selectedAccessoryIds.size === filteredAccessories.length && filteredAccessories.length > 0}
                    onCheckedChange={toggleSelectAll}
                    data-testid="checkbox-select-all"
                  />
                  <span className="text-xs text-muted-foreground">
                    {selectedAccessoryIds.size > 0 
                      ? `Zaznaczono: ${selectedAccessoryIds.size}` 
                      : "Zaznacz wszystkie"}
                  </span>
                </div>
                {selectedAccessoryIds.size > 0 && (
                  <div className="flex gap-1 ml-auto">
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => setIsBulkEditDialogOpen(true)}
                      data-testid="button-bulk-edit"
                    >
                      <Pencil className="h-4 w-4 mr-1" />
                      Edytuj zaznaczone
                    </Button>
                    <Button
                      size="sm"
                      variant="destructive"
                      onClick={() => {
                        if (confirm(`Czy na pewno usunąć ${selectedAccessoryIds.size} akcesoriów?`)) {
                          bulkDeleteMutation.mutate(Array.from(selectedAccessoryIds));
                        }
                      }}
                      disabled={bulkDeleteMutation.isPending}
                      data-testid="button-bulk-delete"
                    >
                      <Trash2 className="h-4 w-4 mr-1" />
                      Usuń zaznaczone
                    </Button>
                  </div>
                )}
              </div>
            )}
            <CardContent className="p-2">
              {accessoriesLoading ? (
                <div className="text-sm text-muted-foreground">Ładowanie...</div>
              ) : filteredAccessories.length === 0 ? (
                <div className="text-sm text-muted-foreground">Brak akcesoriów</div>
              ) : isCompactView ? (
                // Compact View - 30px height per item
                <div className="space-y-0.5">
                  {filteredAccessories.map((accessory) => (
                    <div
                      key={accessory.id}
                      className="flex items-center gap-1.5 h-[30px] px-1.5 border hover-elevate text-xs"
                      data-testid={`accessory-item-${accessory.id}`}
                    >
                      <Checkbox
                        checked={selectedAccessoryIds.has(accessory.id)}
                        onCheckedChange={() => toggleAccessorySelection(accessory.id)}
                        data-testid={`checkbox-accessory-${accessory.id}`}
                      />
                      {accessory.imageUrl ? (
                        <img
                          src={accessory.imageUrl}
                          alt={accessory.name}
                          className="w-6 h-6 object-cover border"
                        />
                      ) : (
                        <div className="w-6 h-6 border flex items-center justify-center bg-muted flex-shrink-0">
                          <ImageIcon className="h-3 w-3 text-muted-foreground" />
                        </div>
                      )}
                      <div className="flex-1 min-w-0 truncate font-medium">
                        {accessory.name}
                      </div>
                      <div className="text-muted-foreground truncate text-[10px]" style={{ maxWidth: "80px" }}>
                        {accessory.code}
                      </div>
                      {accessory.alpmaCode && (
                        <div className="text-primary truncate text-[10px] font-medium" style={{ maxWidth: "70px" }}>
                          {accessory.alpmaCode}
                        </div>
                      )}
                      <Button
                        size="icon"
                        variant={accessory.isActive ? "default" : "outline"}
                        className="h-6 w-6"
                        onClick={() => toggleActiveMutation.mutate({ id: accessory.id, isActive: !accessory.isActive })}
                        data-testid={`button-toggle-active-${accessory.id}`}
                      >
                        {accessory.isActive ? <Check className="h-3 w-3" /> : <XIcon className="h-3 w-3" />}
                      </Button>
                      <Button
                        size="icon"
                        variant="ghost"
                        className="h-6 w-6"
                        onClick={() => {
                          setDuplicatingAccessory(accessory);
                          setIsDuplicateDialogOpen(true);
                        }}
                        data-testid={`button-duplicate-accessory-${accessory.id}`}
                      >
                        <Copy className="h-3 w-3" />
                      </Button>
                      <Button
                        size="icon"
                        variant="ghost"
                        className="h-6 w-6"
                        onClick={() => navigate(`/accessories/${accessory.id}/edit`)}
                        data-testid={`button-edit-accessory-${accessory.id}`}
                      >
                        <Pencil className="h-3 w-3" />
                      </Button>
                      <Button
                        size="icon"
                        variant="ghost"
                        className="h-6 w-6"
                        onClick={() => {
                          if (confirm("Czy na pewno usunąć to akcesorium?")) {
                            deleteAccessoryMutation.mutate(accessory.id);
                          }
                        }}
                        data-testid={`button-delete-accessory-${accessory.id}`}
                      >
                        <Trash2 className="h-3 w-3" />
                      </Button>
                    </div>
                  ))}
                </div>
              ) : (
                // Normal View - Grid with cards
                <div className="grid grid-cols-1 md:grid-cols-2 gap-2">
                  {filteredAccessories.map((accessory) => (
                    <div
                      key={accessory.id}
                      className="p-2 border hover-elevate"
                      data-testid={`accessory-item-${accessory.id}`}
                    >
                      <div className="flex gap-2">
                        <div className="flex items-start pt-1">
                          <Checkbox
                            checked={selectedAccessoryIds.has(accessory.id)}
                            onCheckedChange={() => toggleAccessorySelection(accessory.id)}
                            data-testid={`checkbox-accessory-${accessory.id}`}
                          />
                        </div>
                        {accessory.imageUrl ? (
                          <img
                            src={accessory.imageUrl}
                            alt={accessory.name}
                            className="w-12 h-12 object-cover border"
                          />
                        ) : (
                          <div className="w-12 h-12 border flex items-center justify-center bg-muted">
                            <ImageIcon className="h-5 w-5 text-muted-foreground" />
                          </div>
                        )}
                        <div className="flex-1 min-w-0">
                          <div className="font-medium text-sm">{accessory.name}</div>
                          <div className="text-xs text-muted-foreground">
                            Kod: {accessory.code}
                          </div>
                          {accessory.alpmaCode && (
                            <div className="text-xs text-primary font-medium">
                              Kod Alpma: {accessory.alpmaCode}
                            </div>
                          )}
                          {accessory.groupName && (
                            <div className="text-xs text-muted-foreground">
                              Grupa: {accessory.groupName}
                            </div>
                          )}
                        </div>
                      </div>
                      <div className="flex gap-1 mt-2">
                        <Button
                          size="sm"
                          variant={accessory.isActive ? "default" : "outline"}
                          onClick={() => toggleActiveMutation.mutate({ id: accessory.id, isActive: !accessory.isActive })}
                          data-testid={`button-toggle-active-${accessory.id}`}
                        >
                          {accessory.isActive ? "Aktywny" : "Nieaktywny"}
                        </Button>
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => {
                            setDuplicatingAccessory(accessory);
                            setIsDuplicateDialogOpen(true);
                          }}
                          data-testid={`button-duplicate-accessory-${accessory.id}`}
                        >
                          <Copy className="h-3 w-3 mr-1" />
                          Duplikuj
                        </Button>
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => navigate(`/accessories/${accessory.id}/edit`)}
                          data-testid={`button-edit-accessory-${accessory.id}`}
                        >
                          <Pencil className="h-3 w-3 mr-1" />
                          Edytuj
                        </Button>
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => {
                            if (confirm("Czy na pewno usunąć to akcesorium?")) {
                              deleteAccessoryMutation.mutate(accessory.id);
                            }
                          }}
                          data-testid={`button-delete-accessory-${accessory.id}`}
                        >
                          <Trash2 className="h-3 w-3 mr-1" />
                          Usuń
                        </Button>
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      </div>

      {/* Group Dialog */}
      <Dialog open={isGroupDialogOpen} onOpenChange={setIsGroupDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>
              {editingGroup ? "Edytuj grupę" : "Nowa grupa"}
            </DialogTitle>
            <DialogDescription>
              Grupy organizują akcesoria w kategorie (np. Okucia, Płyty, Nóżki)
            </DialogDescription>
          </DialogHeader>
          <form onSubmit={handleGroupSubmit}>
            <div className="space-y-4">
              <div>
                <Label htmlFor="name">Nazwa grupy</Label>
                <Input
                  id="name"
                  name="name"
                  defaultValue={editingGroup?.name || ""}
                  required
                  data-testid="input-group-name"
                />
              </div>
              <div>
                <Label htmlFor="code">Kod grupy</Label>
                <Input
                  id="code"
                  name="code"
                  defaultValue={editingGroup?.code || ""}
                  required
                  placeholder="np. hardware, boards, legs"
                  data-testid="input-group-code"
                />
              </div>
              <div>
                <Label htmlFor="category">Kategoria (opcjonalnie)</Label>
                <Input
                  id="category"
                  name="category"
                  defaultValue={editingGroup?.category || ""}
                  placeholder="np. Tkaniny, Okucia, Płyty meblowe"
                  data-testid="input-group-category"
                />
              </div>
              <div>
                <Label htmlFor="description">Opis (opcjonalnie)</Label>
                <Textarea
                  id="description"
                  name="description"
                  defaultValue={editingGroup?.description || ""}
                  rows={3}
                  data-testid="textarea-group-description"
                />
              </div>
              <div>
                <Label htmlFor="displayOrder">Kolejność wyświetlania</Label>
                <Input
                  id="displayOrder"
                  name="displayOrder"
                  type="number"
                  defaultValue={editingGroup?.displayOrder || 0}
                  data-testid="input-group-order"
                />
              </div>
            </div>
            <DialogFooter className="mt-4">
              <Button
                type="button"
                variant="outline"
                onClick={() => setIsGroupDialogOpen(false)}
                data-testid="button-cancel-group"
              >
                <X className="h-4 w-4 mr-1" />
                Anuluj
              </Button>
              <Button type="submit" disabled={groupMutation.isPending} data-testid="button-save-group">
                <Save className="h-4 w-4 mr-1" />
                {groupMutation.isPending ? "Zapisywanie..." : "Zapisz"}
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>

      {/* Accessory Dialog */}
      <Dialog open={isAccessoryDialogOpen} onOpenChange={setIsAccessoryDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>
              {editingAccessory ? "Edytuj akcesorium" : "Nowe akcesorium"}
            </DialogTitle>
            <DialogDescription>
              Dodaj szczegóły akcesorium (nazwa, opis, zdjęcie)
            </DialogDescription>
          </DialogHeader>
          <form onSubmit={handleAccessorySubmit}>
            <div className="space-y-4">
              <div>
                <Label htmlFor="accessoryGroupId">Grupa</Label>
                <Select name="groupId" defaultValue={editingAccessory?.groupId?.toString() || ""}>
                  <SelectTrigger data-testid="select-accessory-group">
                    <SelectValue placeholder="Wybierz grupę" />
                  </SelectTrigger>
                  <SelectContent>
                    {groups.map((group) => (
                      <SelectItem key={group.id} value={group.id.toString()}>
                        {group.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label htmlFor="accessoryName">Nazwa akcesorium</Label>
                <Input
                  id="accessoryName"
                  name="name"
                  defaultValue={editingAccessory?.name || ""}
                  required
                  data-testid="input-accessory-name"
                />
              </div>
              <div>
                <Label htmlFor="accessoryCode">Kod akcesorium</Label>
                <Input
                  id="accessoryCode"
                  name="code"
                  defaultValue={editingAccessory?.code || ""}
                  required
                  placeholder="np. silent_hinges, drawer_slides"
                  data-testid="input-accessory-code"
                />
              </div>
              <div>
                <Label htmlFor="accessoryAlpmaCode">Kod Alpma (opcjonalnie)</Label>
                <Input
                  id="accessoryAlpmaCode"
                  name="alpmaCode"
                  defaultValue={editingAccessory?.alpmaCode || ""}
                  placeholder="Własny kod produktu"
                  data-testid="input-accessory-alpma-code"
                />
              </div>
              <div>
                <Label htmlFor="accessoryDescription">Opis (opcjonalnie)</Label>
                <Textarea
                  id="accessoryDescription"
                  name="description"
                  defaultValue={editingAccessory?.description || ""}
                  rows={3}
                  data-testid="textarea-accessory-description"
                />
              </div>
              <div>
                <Label htmlFor="imageUrl">URL zdjęcia (opcjonalnie)</Label>
                <Input
                  id="imageUrl"
                  name="imageUrl"
                  type="url"
                  defaultValue={editingAccessory?.imageUrl || ""}
                  placeholder="https://example.com/image.jpg"
                  data-testid="input-accessory-image"
                />
              </div>
              <div>
                <Label htmlFor="accessoryDisplayOrder">Kolejność wyświetlania</Label>
                <Input
                  id="accessoryDisplayOrder"
                  name="displayOrder"
                  type="number"
                  defaultValue={editingAccessory?.displayOrder || 0}
                  data-testid="input-accessory-order"
                />
              </div>
            </div>
            <DialogFooter className="mt-4">
              <Button
                type="button"
                variant="outline"
                onClick={() => setIsAccessoryDialogOpen(false)}
                data-testid="button-cancel-accessory"
              >
                <X className="h-4 w-4 mr-1" />
                Anuluj
              </Button>
              <Button
                type="submit"
                disabled={accessoryMutation.isPending}
                data-testid="button-save-accessory"
              >
                <Save className="h-4 w-4 mr-1" />
                {accessoryMutation.isPending ? "Zapisywanie..." : "Zapisz"}
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>

      {/* Bulk Edit Dialog */}
      <Dialog open={isBulkEditDialogOpen} onOpenChange={setIsBulkEditDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Edytuj zaznaczone akcesoria ({selectedAccessoryIds.size})</DialogTitle>
            <DialogDescription>
              Wprowadź zmiany, które chcesz zastosować do zaznaczonych akcesoriów. Puste pola zostaną pominięte.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label>Status aktywności</Label>
              <div className="flex gap-2 mt-2">
                <Button
                  type="button"
                  size="sm"
                  variant={bulkEditIsActive === true ? "default" : "outline"}
                  onClick={() => setBulkEditIsActive(bulkEditIsActive === true ? null : true)}
                  data-testid="button-set-active"
                >
                  Aktywne
                </Button>
                <Button
                  type="button"
                  size="sm"
                  variant={bulkEditIsActive === false ? "default" : "outline"}
                  onClick={() => setBulkEditIsActive(bulkEditIsActive === false ? null : false)}
                  data-testid="button-set-inactive"
                >
                  Nieaktywne
                </Button>
                {bulkEditIsActive !== null && (
                  <Button
                    type="button"
                    size="sm"
                    variant="ghost"
                    onClick={() => setBulkEditIsActive(null)}
                    data-testid="button-clear-active"
                  >
                    <X className="h-4 w-4" />
                  </Button>
                )}
              </div>
            </div>
            <div>
              <Label htmlFor="bulkEditDescription">Opis</Label>
              <Textarea
                id="bulkEditDescription"
                value={bulkEditDescription}
                onChange={(e) => setBulkEditDescription(e.target.value)}
                rows={3}
                placeholder="Pozostaw puste aby nie zmieniać"
                data-testid="textarea-bulk-edit-description"
              />
            </div>
          </div>
          <DialogFooter className="mt-4">
            <Button
              type="button"
              variant="outline"
              onClick={() => {
                setIsBulkEditDialogOpen(false);
                setBulkEditIsActive(null);
                setBulkEditDescription("");
              }}
              data-testid="button-cancel-bulk-edit"
            >
              <X className="h-4 w-4 mr-1" />
              Anuluj
            </Button>
            <Button
              type="button"
              onClick={handleBulkEditSubmit}
              disabled={bulkEditMutation.isPending}
              data-testid="button-save-bulk-edit"
            >
              <Save className="h-4 w-4 mr-1" />
              {bulkEditMutation.isPending ? "Zapisywanie..." : "Zapisz"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Duplicate Accessory Dialog */}
      <Dialog open={isDuplicateDialogOpen} onOpenChange={setIsDuplicateDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Duplikuj akcesorium</DialogTitle>
            <DialogDescription>
              {duplicatingAccessory ? `Duplikuj "${duplicatingAccessory.name}" do wybranej grupy` : "Wybierz grupę docelową"}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="duplicateTargetGroup">Grupa docelowa</Label>
              <AccessoryGroupComboboxWithAdd
                groups={groups}
                value={duplicateTargetGroupId}
                onChange={setDuplicateTargetGroupId}
                placeholder="Wybierz grupę docelową"
                searchPlaceholder="Szukaj grupy..."
                emptyText="Nie znaleziono grup."
                testId="select-duplicate-target-group"
              />
              {duplicatingAccessory && duplicateTargetGroupId === duplicatingAccessory.groupId?.toString() && (
                <p className="text-xs text-muted-foreground mt-2">
                  Duplikujesz do tej samej grupy - zostanie utworzona kopia
                </p>
              )}
            </div>
          </div>
          <DialogFooter className="mt-4">
            <Button
              type="button"
              variant="outline"
              onClick={() => {
                setIsDuplicateDialogOpen(false);
                setDuplicatingAccessory(null);
                setDuplicateTargetGroupId("");
              }}
              data-testid="button-cancel-duplicate"
            >
              <X className="h-4 w-4 mr-1" />
              Anuluj
            </Button>
            <Button
              type="button"
              onClick={() => {
                if (duplicatingAccessory && duplicateTargetGroupId) {
                  duplicateAccessoryMutation.mutate({
                    id: duplicatingAccessory.id,
                    targetGroupId: parseInt(duplicateTargetGroupId)
                  });
                }
              }}
              disabled={!duplicateTargetGroupId || duplicateAccessoryMutation.isPending}
              data-testid="button-confirm-duplicate"
            >
              <Copy className="h-4 w-4 mr-1" />
              {duplicateAccessoryMutation.isPending ? "Duplikowanie..." : "Duplikuj"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Duplicate Group Dialog */}
      <Dialog open={isDuplicateGroupDialogOpen} onOpenChange={setIsDuplicateGroupDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Duplikuj grupę akcesoriów</DialogTitle>
            <DialogDescription>
              {duplicatingGroup ? `Duplikuj grupę "${duplicatingGroup.name}" wraz ze wszystkimi akcesoriami` : "Wprowadź nazwę nowej grupy"}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="duplicateGroupName">Nazwa nowej grupy</Label>
              <Input
                id="duplicateGroupName"
                value={duplicateGroupName}
                onChange={(e) => setDuplicateGroupName(e.target.value)}
                placeholder="Wprowadź nazwę grupy"
                data-testid="input-duplicate-group-name"
              />
            </div>
          </div>
          <DialogFooter className="mt-4">
            <Button
              type="button"
              variant="outline"
              onClick={() => {
                setIsDuplicateGroupDialogOpen(false);
                setDuplicatingGroup(null);
                setDuplicateGroupName("");
              }}
              data-testid="button-cancel-duplicate-group"
            >
              <X className="h-4 w-4 mr-1" />
              Anuluj
            </Button>
            <Button
              type="button"
              onClick={() => {
                if (duplicatingGroup && duplicateGroupName.trim()) {
                  duplicateGroupMutation.mutate({
                    id: duplicatingGroup.id,
                    name: duplicateGroupName.trim()
                  });
                }
              }}
              disabled={!duplicateGroupName.trim() || duplicateGroupMutation.isPending}
              data-testid="button-confirm-duplicate-group"
            >
              <Copy className="h-4 w-4 mr-1" />
              {duplicateGroupMutation.isPending ? "Duplikowanie..." : "Duplikuj"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
