/**
 * Template Renderer
 * 
 * Renders HTML templates with variable substitution and basic control flow.
 * Supports:
 * - Simple variables: {{color}}, {{material}}
 * - Loops: {{#each addons}}<li>{{name}}</li>{{/each}}
 * - Conditionals: {{#if color}}Kolor: {{color}}{{/if}}
 * - Nested properties: {{product.title}}, {{addon.price}}
 * - Accessory tags: {{akcesorium-tkaniny_wotan}}, {{okucia-zawiasy}}
 */

interface RenderContext {
  [key: string]: any;
}

/**
 * Render accessory-specific tags
 * {{akcesorium-GROUP_CODE}} => HTML structure with accessories from that group
 * {{akcesorium-GROUP_CODE:grid3}} => HTML structure with 3-column grid layout
 * 
 * Expected context structure:
 * accessories: {
 *   groups: [
 *     {
 *       groupCode: "tkaniny_wotan",
 *       groupName: "Tkaniny Wotan",
 *       items: [
 *         { name: "Velvet granatowy", description: "...", imageUrl: "...", code: "VEL-GRA" }
 *       ]
 *     }
 *   ]
 * }
 */
function renderAccessoryTags(template: string, context: RenderContext): string {
  // Pattern: {{akcesorium-GROUP_CODE}} or {{akcesorium-GROUP_CODE:grid3}}
  // Supports optional :gridN parameter for layout control
  const accessoryTagRegex = /\{\{(?:akcesorium|okucia|akcesoria)-([a-zA-Z0-9_-]+)(?::grid(\d+))?\}\}/g;
  
  return template.replace(accessoryTagRegex, (match, groupCode, gridCols) => {
    if (!context.accessories || !context.accessories.groups) {
      return ''; // No accessories available
    }
    
    // Find the group by code
    const group = context.accessories.groups.find(
      (g: any) => g.groupCode === groupCode || g.code === groupCode
    );
    
    if (!group || !group.items || group.items.length === 0) {
      return ''; // Group not found or empty
    }
    
    // Parse grid columns (default to 1 if not specified)
    const columns = gridCols ? parseInt(gridCols) : 1;
    const gridClass = columns > 1 ? `accessory-items-grid accessory-items-grid-${columns}` : 'accessory-items';
    
    // Render accessories from this group
    // Layout: Left side image, right side title + description (or grid layout if gridN specified)
    return `
      <div class="accessory-group" data-group="${groupCode}">
        <h3 class="accessory-group-title">${group.groupName || group.name || groupCode}</h3>
        <div class="${gridClass}">
          ${group.items.map((item: any) => `
            <div class="accessory-item">
              ${item.imageUrl ? `
                <div class="accessory-image">
                  <img src="${item.imageUrl}" alt="${item.name}" loading="lazy" />
                </div>
              ` : ''}
              <div class="accessory-content">
                <h4 class="accessory-name">${item.name}${item.code ? ` <span class="accessory-code">(${item.code})</span>` : ''}</h4>
                ${item.description ? `<p class="accessory-description">${item.description}</p>` : ''}
              </div>
            </div>
          `).join('')}
        </div>
      </div>
    `.trim();
  });
}

/**
 * Get nested property value from object using dot notation
 * Example: getNestedValue({product: {title: "Test"}}, "product.title") => "Test"
 */
function getNestedValue(obj: any, path: string): any {
  if (!path) return obj;
  
  const keys = path.split('.');
  let result = obj;
  
  for (const key of keys) {
    if (result === null || result === undefined) return '';
    result = result[key];
  }
  
  return result !== null && result !== undefined ? result : '';
}

/**
 * Render a simple variable placeholder
 * {{color}} => "biały"
 * {{@index}} => "0" (supports @ prefix for loop helpers)
 */
function renderVariable(template: string, context: RenderContext): string {
  return template.replace(/\{\{([@\w]+(?:\.[\w]+)*)\}\}/g, (match, varName) => {
    const value = getNestedValue(context, varName.trim());
    return String(value);
  });
}

/**
 * Render conditional blocks
 * {{#if color}}Kolor: {{color}}{{/if}}
 * 
 * Supports nested loops and conditionals by recursively calling renderTemplate
 */
function renderConditionals(template: string, context: RenderContext): string {
  const ifRegex = /\{\{#if\s+([@\w]+(?:\.[\w]+)*)\}\}([\s\S]*?)\{\{\/if\}\}/g;
  
  return template.replace(ifRegex, (match, varName, content) => {
    const value = getNestedValue(context, varName.trim());
    
    // Check if value is truthy
    if (value && (typeof value !== 'object' || (Array.isArray(value) && value.length > 0))) {
      // Recursively render to support nested loops/conditionals
      return renderTemplate(content, context);
    }
    
    return '';
  });
}

/**
 * Render loop blocks
 * {{#each addons}}<li>{{name}} - {{price}} PLN</li>{{/each}}
 * 
 * Supports nested loops and conditionals by recursively calling renderTemplate
 */
function renderLoops(template: string, context: RenderContext): string {
  const eachRegex = /\{\{#each\s+([@\w]+(?:\.[\w]+)*)\}\}([\s\S]*?)\{\{\/each\}\}/g;
  
  return template.replace(eachRegex, (match, varName, content) => {
    const items = getNestedValue(context, varName.trim());
    
    if (!Array.isArray(items) || items.length === 0) {
      return '';
    }
    
    return items.map((item, index) => {
      // Create context with item properties + special @index variable
      const itemContext = {
        ...context,
        ...item,
        '@index': index,
        '@first': index === 0,
        '@last': index === items.length - 1,
      };
      
      // Recursively render to support nested loops/conditionals
      return renderTemplate(content, itemContext);
    }).join('');
  });
}

/**
 * Main template rendering function
 * Processes template with variables, conditionals, and loops
 */
export function renderTemplate(htmlContent: string, variables: RenderContext): string {
  if (!htmlContent) return '';
  if (!variables) return htmlContent;
  
  let result = htmlContent;
  
  // Process in order: accessory tags -> loops -> conditionals -> variables
  // This allows variables inside loops and conditionals to be rendered
  result = renderAccessoryTags(result, variables);
  result = renderLoops(result, variables);
  result = renderConditionals(result, variables);
  result = renderVariable(result, variables);
  
  return result;
}

/**
 * Helper to create sample data for template preview
 */
export function createSampleData(): RenderContext {
  return {
    product: {
      alpmaCode: 'VB80x36-BIA',
      sku: 'PROD-001',
      title: 'Szafka na buty VB80x36 biała',
      color: 'biały',
      dimensions: {
        width: 80,
        depth: 36,
        height: 120
      },
      price: 1299.99,
      productType: 'szafka_na_buty'
    },
    color: 'biały',
    material: 'drewno',
    dimensions: '120x80x75 cm',
    price: 1299.99,
    ai_description: 'Elegancka szafka na buty w kolorze białym to idealny wybór dla Twojego przedpokoju. Wykonana z wysokiej jakości materiałów, zapewnia trwałość i funkcjonalność na lata.',
    accessories: {
      groups: [
        {
          groupCode: 'tkaniny_wotan',
          groupName: 'Tkaniny Wotan',
          code: 'tkaniny_wotan',
          name: 'Tkaniny Wotan',
          items: [
            { 
              name: 'Velvet granatowy', 
              code: 'VEL-GRA',
              description: 'Miękka, aksamitna tkanina w intensywnym kolorze granatowym. Odporna na ścieranie.',
              imageUrl: 'https://placehold.co/150x150/1e3a8a/white?text=Velvet'
            },
            { 
              name: 'Velvet szary', 
              code: 'VEL-SZA',
              description: 'Elegancka szara tkanina aksamitna. Łatwa w czyszczeniu.',
              imageUrl: 'https://placehold.co/150x150/6b7280/white?text=Velvet'
            }
          ]
        },
        {
          groupCode: 'okucia-zawiasy',
          groupName: 'Zawiasy',
          code: 'okucia-zawiasy',
          name: 'Zawiasy',
          items: [
            { 
              name: 'Zawias Silent System', 
              code: 'SILENT-1',
              description: 'Ciche domykanie drzwi. Wytrzymałość 100,000 otwarć.',
              imageUrl: 'https://placehold.co/150x150/4b5563/white?text=Zawias'
            }
          ]
        }
      ]
    },
    sizeTable: {
      headers: ['Szerokość', 'Głębokość', 'Wysokość'],
      rows: [
        { variant: 'Mały', values: ['60cm', '36cm', '120cm'] },
        { variant: 'Średni', values: ['80cm', '36cm', '120cm'] },
        { variant: 'Duży', values: ['100cm', '36cm', '120cm'] }
      ]
    },
    addons: [
      { name: 'Płyta laminowana', price: 150, type: 'board' },
      { name: 'Certyfikat FSC', price: 0, type: 'certificate' },
      { name: 'Tkanina wodoodporna', price: 200, type: 'fabric' },
    ],
    features: [
      'Wysoka jakość wykonania',
      'Ekologiczne materiały',
      'Gwarancja 5 lat',
    ],
  };
}
