import { useState, useEffect } from "react";

interface DashboardSettings {
  searchQuery: string;
  paymentFilter: string;
  fulfillmentFilter: string;
  invoiceFilter: string;
  paymentTypeFilter: string;
  sourceFilter: string;
  dateRangeType: string;
  customDateFrom: string | null;
  customDateTo: string | null;
  currentPage: number;
  itemsPerPage: number;
  sortBy: string;
  sortOrder: "ASC" | "DESC";
  chartDays: number;
}

const DEFAULT_SETTINGS: DashboardSettings = {
  searchQuery: "",
  paymentFilter: "all",
  fulfillmentFilter: "all",
  invoiceFilter: "all",
  paymentTypeFilter: "all",
  sourceFilter: "all",
  dateRangeType: "all",
  customDateFrom: null,
  customDateTo: null,
  currentPage: 1,
  itemsPerPage: 25,
  sortBy: "order_number",
  sortOrder: "DESC",
  chartDays: 30,
};

const STORAGE_KEY = "dashboard-settings";

export function useDashboardSettings() {
  const [settings, setSettings] = useState<DashboardSettings>(() => {
    try {
      const stored = localStorage.getItem(STORAGE_KEY);
      if (stored) {
        const parsed = JSON.parse(stored);
        return { ...DEFAULT_SETTINGS, ...parsed };
      }
    } catch (error) {
      console.error("Error loading dashboard settings:", error);
    }
    return DEFAULT_SETTINGS;
  });

  useEffect(() => {
    try {
      localStorage.setItem(STORAGE_KEY, JSON.stringify(settings));
    } catch (error) {
      console.error("Error saving dashboard settings:", error);
    }
  }, [settings]);

  const updateSettings = (updates: Partial<DashboardSettings>) => {
    setSettings((prev) => ({ ...prev, ...updates }));
  };

  const resetSettings = () => {
    setSettings(DEFAULT_SETTINGS);
  };

  return {
    settings,
    updateSettings,
    resetSettings,
  };
}
