import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Separator } from "@/components/ui/separator";
import { Copy, Info } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "@/components/ui/tooltip";

interface TagItem {
  tag: string;
  description: string;
  category: string;
}

interface AccessoryGroup {
  code: string;
  name: string;
  category: string | null;
}

interface AvailableTagsPanelProps {
  onTagInsert?: (tag: string) => void;
  onShowExamples?: () => void;
}

export function AvailableTagsPanel({ onTagInsert, onShowExamples }: AvailableTagsPanelProps) {
  const { toast } = useToast();
  
  const { data: accessoryGroups = [] } = useQuery<AccessoryGroup[]>({
    queryKey: ["/api/accessory-groups/codes"],
  });

  const productTags: TagItem[] = [
    { tag: "{{title}}", description: "Pełny tytuł produktu", category: "Produkt" },
    { tag: "{{sku}}", description: "SKU produktu", category: "Produkt" },
    { tag: "{{dimensions}}", description: "Wymiary (np. 50×30×124 cm)", category: "Produkt" },
    { tag: "{{length}}", description: "Długość", category: "Wymiary" },
    { tag: "{{width}}", description: "Szerokość", category: "Wymiary" },
    { tag: "{{height}}", description: "Wysokość", category: "Wymiary" },
    { tag: "{{color}}", description: "Kolor", category: "Produkt" },
    { tag: "{{price}}", description: "Cena", category: "Produkt" },
    { tag: "{{basePrice}}", description: "Cena bazowa", category: "Produkt" },
  ];

  const structureTags: TagItem[] = [
    { tag: "{{#each accessories}}...{{/each}}", description: "Pętla przez wszystkie akcesoria", category: "Pętle" },
    { tag: "{{#size_table}}...{{/size_table}}", description: "Tabela wymiarów produktu", category: "Tabele" },
    { tag: "{{#if condition}}...{{/if}}", description: "Warunek", category: "Logika" },
    { tag: "{{accessories.length}}", description: "Liczba akcesoriów", category: "Akcesoria" },
  ];

  const imageTags: TagItem[] = [
    { tag: "{img1 w=100}", description: "Pierwsze zdjęcie (szerokość 100%)", category: "100%" },
    { tag: "{img1 w=75}", description: "Pierwsze zdjęcie (szerokość 75%)", category: "75%" },
    { tag: "{img1 w=50}", description: "Pierwsze zdjęcie (szerokość 50%)", category: "50%" },
    { tag: "{img1 w=25}", description: "Pierwsze zdjęcie (szerokość 25%)", category: "25%" },
    { tag: "{img2 w=100}", description: "Drugie zdjęcie (szerokość 100%)", category: "100%" },
    { tag: "{img2 w=75}", description: "Drugie zdjęcie (szerokość 75%)", category: "75%" },
    { tag: "{img2 w=50}", description: "Drugie zdjęcie (szerokość 50%)", category: "50%" },
    { tag: "{img2 w=25}", description: "Drugie zdjęcie (szerokość 25%)", category: "25%" },
    { tag: "{img3 w=100}", description: "Trzecie zdjęcie (szerokość 100%)", category: "100%" },
    { tag: "{img3 w=75}", description: "Trzecie zdjęcie (szerokość 75%)", category: "75%" },
    { tag: "{img3 w=50}", description: "Trzecie zdjęcie (szerokość 50%)", category: "50%" },
    { tag: "{img3 w=25}", description: "Trzecie zdjęcie (szerokość 25%)", category: "25%" },
    { tag: "{img4 w=100}", description: "Czwarte zdjęcie (szerokość 100%)", category: "100%" },
    { tag: "{img4 w=75}", description: "Czwarte zdjęcie (szerokość 75%)", category: "75%" },
    { tag: "{img4 w=50}", description: "Czwarte zdjęcie (szerokość 50%)", category: "50%" },
    { tag: "{img4 w=25}", description: "Czwarte zdjęcie (szerokość 25%)", category: "25%" },
    { tag: "{img5 w=100}", description: "Piąte zdjęcie (szerokość 100%)", category: "100%" },
    { tag: "{img5 w=75}", description: "Piąte zdjęcie (szerokość 75%)", category: "75%" },
    { tag: "{img5 w=50}", description: "Piąte zdjęcie (szerokość 50%)", category: "50%" },
    { tag: "{img5 w=25}", description: "Piąte zdjęcie (szerokość 25%)", category: "25%" },
  ];

  const accessoryLoopTags: TagItem[] = [
    { tag: "{{name}}", description: "Nazwa akcesorium (w pętli)", category: "W pętli" },
    { tag: "{{code}}", description: "Kod akcesorium (w pętli)", category: "W pętli" },
    { tag: "{{description}}", description: "Opis akcesorium (w pętli)", category: "W pętli" },
    { tag: "{{imageUrl}}", description: "URL zdjęcia (w pętli)", category: "W pętli" },
    { tag: "{{groupName}}", description: "Nazwa grupy (w pętli)", category: "W pętli" },
  ];

  const accessoryDirectTags: TagItem[] = accessoryGroups.map(group => ({
    tag: `{{akcesorium-${group.code}}}`,
    description: group.name,
    category: group.category || "Bez kategorii",
  }));

  const handleCopyTag = (tag: string) => {
    if (onTagInsert) {
      onTagInsert(tag);
      toast({
        title: "Tag wstawiony",
        description: `Wstawiono: ${tag}`,
      });
    } else {
      navigator.clipboard.writeText(tag);
      toast({
        title: "Skopiowano",
        description: `Tag ${tag} został skopiowany do schowka`,
      });
    }
  };

  const TagButton = ({ item }: { item: TagItem }) => (
    <TooltipProvider>
      <Tooltip>
        <TooltipTrigger asChild>
          <Button
            variant="outline"
            size="sm"
            className="justify-between w-full hover-elevate active-elevate-2"
            onClick={() => handleCopyTag(item.tag)}
            data-testid={`button-copy-tag-${item.tag.replace(/[{}]/g, '')}`}
          >
            <code className="text-xs">{item.tag}</code>
            <Copy className="h-3 w-3 ml-2" />
          </Button>
        </TooltipTrigger>
        <TooltipContent>
          <p>{item.description}</p>
        </TooltipContent>
      </Tooltip>
    </TooltipProvider>
  );

  const renderTagSection = (title: string, tags: TagItem[], showCategory: boolean = false) => {
    if (tags.length === 0) return null;

    const groupedTags = showCategory
      ? tags.reduce((acc, tag) => {
          const cat = tag.category;
          if (!acc[cat]) acc[cat] = [];
          acc[cat].push(tag);
          return acc;
        }, {} as Record<string, TagItem[]>)
      : { [title]: tags };

    return (
      <div className="space-y-3">
        <div className="flex items-center gap-2">
          <h3 className="font-medium text-sm">{title}</h3>
          <Badge variant="secondary">{tags.length}</Badge>
        </div>
        {Object.entries(groupedTags).map(([category, categoryTags]) => (
          <div key={category} className="space-y-2">
            {showCategory && <p className="text-xs text-muted-foreground">{category}</p>}
            <div className="grid gap-1.5">
              {categoryTags.map((item, idx) => (
                <TagButton key={idx} item={item} />
              ))}
            </div>
          </div>
        ))}
      </div>
    );
  };

  return (
    <Card className="h-full flex flex-col" data-testid="card-available-tags">
      <CardHeader className="space-y-1 pb-3">
        <div className="flex items-center justify-between">
          <CardTitle className="text-base">Dostępne Tagi</CardTitle>
          <Button
            variant="ghost"
            size="sm"
            onClick={onShowExamples}
            data-testid="button-show-examples"
          >
            <Info className="h-4 w-4 mr-2" />
            Przykłady
          </Button>
        </div>
        <CardDescription className="text-xs space-y-1">
          <p>Kliknij tag aby wstawić do edytora</p>
          <p className="text-muted-foreground/80 italic">
            📸 Obrazki: {`{img1 w=100}`} (szerokość: 100/75/50/25%)
          </p>
          <p className="text-muted-foreground/80 italic">
            🔧 Akcesoria bez parametru: {`{{akcesorium-CODE}}`} → HTML lista
          </p>
          <p className="text-muted-foreground/80 italic">
            🖼️ Gridy stałe: {`{{akcesorium-CODE:grid3}}`} → JPG (grid3/grid4/grid2-2)
          </p>
          <p className="text-muted-foreground/80 italic">
            ♾️ Gridy nieskończone: {`{{akcesorium-CODE:grid3-}}`} → JPG (grid2-/grid3-/grid4-)
          </p>
        </CardDescription>
      </CardHeader>
      <CardContent className="flex-1 overflow-hidden p-0">
        <ScrollArea className="h-full px-6">
          <div className="space-y-4 pb-6">
            {renderTagSection("Zmienne Produktu", productTags)}
            <Separator />
            {renderTagSection("Zdjęcia Produktu", imageTags, true)}
            <Separator />
            {renderTagSection("Struktury i Pętle", structureTags)}
            <Separator />
            {renderTagSection("Dane w Pętli {{#each accessories}}", accessoryLoopTags)}
            <Separator />
            {renderTagSection("Tagi Akcesoriów", accessoryDirectTags, true)}
          </div>
        </ScrollArea>
      </CardContent>
    </Card>
  );
}
