import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Loader2, FileText } from "lucide-react";
import { Badge } from "@/components/ui/badge";

interface Template {
  id: number;
  name: string;
  categoryId: number | null;
  categoryName: string | null;
  templateType: string;
  htmlContent: string;
  contentDoc: any;
  isGlobal: boolean;
  isActive: boolean;
}

interface TemplatePickerDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  onSelect: (template: Template) => void;
}

export function TemplatePickerDialog({
  open,
  onOpenChange,
  onSelect,
}: TemplatePickerDialogProps) {
  const [selectedTemplate, setSelectedTemplate] = useState<Template | null>(null);

  // Fetch active templates
  const { data: templates, isLoading } = useQuery<Template[]>({
    queryKey: ["/api/description-templates"],
    enabled: open,
  });

  const activeTemplates = templates?.filter((t) => t.isActive) || [];

  const handleSelect = () => {
    if (selectedTemplate) {
      onSelect(selectedTemplate);
      onOpenChange(false);
      setSelectedTemplate(null);
    }
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-2xl">
        <DialogHeader>
          <DialogTitle>Wybierz szablon</DialogTitle>
          <DialogDescription>
            Wybierz szablon do skopiowania treści do opisu produktu
          </DialogDescription>
        </DialogHeader>

        {isLoading ? (
          <div className="flex items-center justify-center py-12">
            <Loader2 className="h-8 w-8 animate-spin" />
          </div>
        ) : activeTemplates.length === 0 ? (
          <div className="text-center text-muted-foreground py-12">
            Brak aktywnych szablonów
          </div>
        ) : (
          <>
            <ScrollArea className="h-[400px] w-full">
              <div className="space-y-2 p-4">
                {activeTemplates.map((template) => (
                  <div
                    key={template.id}
                    className={`p-4 border rounded-md cursor-pointer transition-all ${
                      selectedTemplate?.id === template.id
                        ? "border-primary bg-accent"
                        : "hover:bg-accent/50"
                    }`}
                    onClick={() => setSelectedTemplate(template)}
                    data-testid={`template-option-${template.id}`}
                  >
                    <div className="flex items-start gap-3">
                      <FileText className="h-5 w-5 text-muted-foreground mt-1" />
                      <div className="flex-1 min-w-0">
                        <h4 className="font-medium truncate">{template.name}</h4>
                        <div className="flex items-center gap-2 mt-1">
                          {template.categoryName && (
                            <Badge variant="outline" className="text-xs">
                              {template.categoryName}
                            </Badge>
                          )}
                          <Badge variant="secondary" className="text-xs">
                            {template.templateType}
                          </Badge>
                          {template.isGlobal && (
                            <Badge variant="default" className="text-xs">
                              Globalny
                            </Badge>
                          )}
                        </div>
                        <div
                          className="text-xs text-muted-foreground mt-2 line-clamp-2"
                          dangerouslySetInnerHTML={{
                            __html: template.htmlContent.substring(0, 150),
                          }}
                        />
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </ScrollArea>

            <DialogFooter>
              <Button
                type="button"
                variant="outline"
                onClick={() => onOpenChange(false)}
                data-testid="button-cancel-template-picker"
              >
                Anuluj
              </Button>
              <Button
                type="button"
                onClick={handleSelect}
                disabled={!selectedTemplate}
                data-testid="button-use-template"
              >
                Użyj szablonu
              </Button>
            </DialogFooter>
          </>
        )}
      </DialogContent>
    </Dialog>
  );
}
