import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Loader2, Check } from "lucide-react";

interface ProductImage {
  id: number;
  productId: number;
  url: string;
  filename: string;
  altText: string | null;
  imageType: string | null;
  sortOrder: number;
}

interface ImageGalleryPickerProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  onSelect: (imageUrl: string) => void;
}

export function ImageGalleryPicker({ open, onOpenChange, onSelect }: ImageGalleryPickerProps) {
  const [selectedUrl, setSelectedUrl] = useState<string>("");

  // Fetch all product images
  const { data: allImages, isLoading } = useQuery<ProductImage[]>({
    queryKey: ["/api/all-product-images"],
    queryFn: async () => {
      const response = await fetch("/api/all-product-images");
      if (!response.ok) throw new Error("Failed to fetch images");
      return await response.json();
    },
    enabled: open,
  });

  const handleSelect = () => {
    if (selectedUrl) {
      onSelect(selectedUrl);
      onOpenChange(false);
      setSelectedUrl("");
    }
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-4xl max-h-[80vh]">
        <DialogHeader>
          <DialogTitle>Wybierz zdjęcie z galerii</DialogTitle>
          <DialogDescription>
            Wybierz zdjęcie z galerii wszystkich produktów
          </DialogDescription>
        </DialogHeader>

        {isLoading ? (
          <div className="flex items-center justify-center py-12">
            <Loader2 className="h-8 w-8 animate-spin" />
          </div>
        ) : allImages && allImages.length === 0 ? (
          <div className="text-center text-muted-foreground py-12">
            Brak zdjęć w galerii
          </div>
        ) : (
          <>
            <ScrollArea className="h-[400px] w-full">
              <div className="grid grid-cols-4 gap-4 p-4">
                {allImages?.map((image) => (
                  <div
                    key={image.id}
                    className={`relative cursor-pointer rounded-md overflow-hidden border-2 transition-all ${
                      selectedUrl === image.url
                        ? "border-primary ring-2 ring-primary"
                        : "border-transparent hover:border-muted-foreground"
                    }`}
                    onClick={() => setSelectedUrl(image.url)}
                    data-testid={`image-option-${image.id}`}
                  >
                    <img
                      src={image.url}
                      alt={image.altText || image.filename}
                      className="w-full h-32 object-cover"
                    />
                    {selectedUrl === image.url && (
                      <div className="absolute top-2 right-2 bg-primary rounded-full p-1">
                        <Check className="h-4 w-4 text-primary-foreground" />
                      </div>
                    )}
                    {image.altText && (
                      <div className="absolute bottom-0 left-0 right-0 bg-black/70 text-white text-xs p-1 truncate">
                        {image.altText}
                      </div>
                    )}
                  </div>
                ))}
              </div>
            </ScrollArea>

            <div className="flex justify-end gap-2 pt-4">
              <Button
                type="button"
                variant="outline"
                onClick={() => onOpenChange(false)}
                data-testid="button-cancel-image-picker"
              >
                Anuluj
              </Button>
              <Button
                type="button"
                onClick={handleSelect}
                disabled={!selectedUrl}
                data-testid="button-select-image"
              >
                Wybierz zdjęcie
              </Button>
            </div>
          </>
        )}
      </DialogContent>
    </Dialog>
  );
}
