import { useEffect, useRef, useState } from 'react';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Loader2 } from 'lucide-react';

interface LogEntry {
  timestamp: string;
  message: string;
  type: 'info' | 'success' | 'error' | 'warning';
}

interface GenerationLogsDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  sessionId: string;
  title?: string;
  description?: string;
}

export function GenerationLogsDialog({
  open,
  onOpenChange,
  sessionId,
  title = 'Logi generowania',
  description = 'Podgląd na żywo procesu generowania produktów',
}: GenerationLogsDialogProps) {
  const [logs, setLogs] = useState<LogEntry[]>([]);
  const [isConnected, setIsConnected] = useState(false);
  const scrollRef = useRef<HTMLDivElement>(null);
  const eventSourceRef = useRef<EventSource | null>(null);

  // Auto-scroll to bottom when new logs arrive
  useEffect(() => {
    if (scrollRef.current) {
      scrollRef.current.scrollTop = scrollRef.current.scrollHeight;
    }
  }, [logs]);

  // Setup SSE connection
  useEffect(() => {
    if (!open || !sessionId) return;

    console.log('📡 Connecting to SSE stream for session:', sessionId);
    
    // Create EventSource for SSE
    const eventSource = new EventSource(`/api/generation-logs/stream/${sessionId}`);
    eventSourceRef.current = eventSource;

    eventSource.onopen = () => {
      console.log('✅ SSE connection established');
      setIsConnected(true);
    };

    eventSource.onmessage = (event) => {
      try {
        const logData: LogEntry = JSON.parse(event.data);
        setLogs((prev) => [...prev, logData]);
      } catch (error) {
        console.error('Error parsing log data:', error);
      }
    };

    eventSource.onerror = (error) => {
      console.error('❌ SSE connection error:', error);
      setIsConnected(false);
      eventSource.close();
    };

    // Cleanup on unmount or when dialog closes
    return () => {
      console.log('🔌 Closing SSE connection');
      eventSource.close();
      setIsConnected(false);
    };
  }, [open, sessionId]);

  // Clear logs when dialog closes
  useEffect(() => {
    if (!open) {
      setLogs([]);
    }
  }, [open]);

  const getLogColor = (type: LogEntry['type']) => {
    switch (type) {
      case 'success':
        return 'text-green-600 dark:text-green-400';
      case 'error':
        return 'text-red-600 dark:text-red-400';
      case 'warning':
        return 'text-yellow-600 dark:text-yellow-400';
      default:
        return 'text-foreground/80';
    }
  };

  const getLogIcon = (type: LogEntry['type']) => {
    switch (type) {
      case 'success':
        return '✅';
      case 'error':
        return '❌';
      case 'warning':
        return '⚠️';
      default:
        return '📝';
    }
  };

  const formatTimestamp = (timestamp: string) => {
    const date = new Date(timestamp);
    return date.toLocaleTimeString('pl-PL', {
      hour: '2-digit',
      minute: '2-digit',
      second: '2-digit',
    });
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-4xl max-h-[50vh] flex flex-col" data-testid="dialog-generation-logs" style={{ zIndex: 40 }}>
        <DialogHeader className="flex-shrink-0">
          <div className="flex items-center justify-between">
            <div>
              <DialogTitle className="text-xl font-bold" data-testid="text-logs-title">
                {title}
              </DialogTitle>
              <DialogDescription data-testid="text-logs-description">
                {description}
              </DialogDescription>
            </div>
            <div className="flex items-center gap-2">
              {isConnected ? (
                <div className="flex items-center gap-2 text-sm text-green-600 dark:text-green-400" data-testid="status-connected">
                  <Loader2 className="h-4 w-4 animate-spin" />
                  <span>Połączono</span>
                </div>
              ) : (
                <div className="text-sm text-muted-foreground" data-testid="status-disconnected">
                  Rozłączono
                </div>
              )}
            </div>
          </div>
        </DialogHeader>

        <div 
          ref={scrollRef}
          className="flex-1 min-h-0 overflow-y-auto bg-muted/30 rounded-lg border p-4 space-y-2" 
          data-testid="scroll-logs-content"
        >
          {logs.length === 0 ? (
            <div className="text-center text-muted-foreground py-8" data-testid="text-no-logs">
              Oczekiwanie na logi...
            </div>
          ) : (
            logs.map((log, index) => (
              <div
                key={index}
                className="flex items-start gap-2 text-sm font-mono"
                data-testid={`log-entry-${index}`}
              >
                <span className="text-muted-foreground flex-shrink-0">
                  {formatTimestamp(log.timestamp)}
                </span>
                <span className="flex-shrink-0">{getLogIcon(log.type)}</span>
                <span className={getLogColor(log.type)}>{log.message}</span>
              </div>
            ))
          )}
        </div>

        <div className="flex justify-end gap-2">
          <Button
            variant="outline"
            onClick={() => setLogs([])}
            data-testid="button-clear-logs"
          >
            Wyczyść logi
          </Button>
          <Button
            onClick={() => onOpenChange(false)}
            data-testid="button-close-dialog"
          >
            Zamknij
          </Button>
        </div>
      </DialogContent>
    </Dialog>
  );
}
