import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Server, CheckCircle2, AlertCircle, Loader2 } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";

export function FileStorageSettings() {
  const { toast } = useToast();
  const [provider, setProvider] = useState<"local" | "sftp">("local");
  const [host, setHost] = useState("");
  const [port, setPort] = useState("22");
  const [username, setUsername] = useState("");
  const [baseUrl, setBaseUrl] = useState("");
  const [pathPrefix, setPathPrefix] = useState("/OMS");
  const [isActive, setIsActive] = useState(false);

  // Fetch current settings
  const { data: settings, isLoading } = useQuery<{
    provider: "local" | "sftp";
    host?: string;
    port?: number;
    username?: string;
    base_url?: string;
    path_prefix?: string;
    is_active?: boolean;
    connection_status?: string;
    last_error?: string;
  }>({
    queryKey: ["/api/file-storage/settings"],
  });

  // Populate form with settings
  useEffect(() => {
    if (settings) {
      setProvider(settings.provider || "local");
      setHost(settings.host || "");
      setPort(String(settings.port || 22));
      setUsername(settings.username || "");
      setBaseUrl(settings.base_url || "");
      setPathPrefix(settings.path_prefix || "/OMS");
      setIsActive(settings.is_active || false);
    }
  }, [settings]);

  // Save settings mutation
  const saveMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/file-storage/settings", {
        provider,
        host,
        port: parseInt(port),
        username,
        baseUrl,
        pathPrefix,
        isActive,
      });
    },
    onSuccess: () => {
      toast({
        title: "Ustawienia zapisane",
        description: "Konfiguracja serwera plików została zaktualizowana.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/file-storage/settings"] });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd zapisu",
        description: error.message || "Nie udało się zapisać ustawień",
        variant: "destructive",
      });
    },
  });

  // Test connection mutation
  const testMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/file-storage/test", {
        host,
        port: parseInt(port),
        username,
        pathPrefix,
      });
    },
    onSuccess: (data: any) => {
      toast({
        title: "Połączenie udane",
        description: data.message || "Udało się połączyć z serwerem plików",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/file-storage/settings"] });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd połączenia",
        description: error.details || error.message || "Nie udało się połączyć z serwerem",
        variant: "destructive",
      });
    },
  });

  const handleSave = () => {
    saveMutation.mutate();
  };

  const handleTest = () => {
    if (!host || !username) {
      toast({
        title: "Brak danych",
        description: "Uzupełnij Host i Username przed testem połączenia",
        variant: "destructive",
      });
      return;
    }
    testMutation.mutate();
  };

  // Migrate files mutation
  const migrateMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/file-storage/migrate", {});
    },
    onSuccess: (data: any) => {
      toast({
        title: "Migracja zakończona",
        description: `Przeniesiono ${data.results.success} z ${data.results.total} plików. Niepowodzeń: ${data.results.failed}`,
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd migracji",
        description: error.details || error.message || "Nie udało się zmigrować plików",
        variant: "destructive",
      });
    },
  });

  const handleMigrate = () => {
    if (!settings?.is_active || settings?.provider !== 'sftp') {
      toast({
        title: "SFTP nie aktywny",
        description: "Najpierw zapisz i aktywuj konfigurację SFTP",
        variant: "destructive",
      });
      return;
    }
    
    if (!confirm("Czy na pewno chcesz zmigrować ~1000 plików na SFTP? To może zająć kilka minut.")) {
      return;
    }
    
    migrateMutation.mutate();
  };

  // Migrate database URLs mutation (for matrices and catalog products)
  const migrateDbUrlsMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/file-storage/migrate-database-urls", {});
    },
    onSuccess: (data: any) => {
      const { matrices, catalogImages } = data.results;
      toast({
        title: "Migracja URL-i zakończona",
        description: `Matryc: ${matrices.updated}/${matrices.total}, Zdjęcia produktów: ${catalogImages.updated}/${catalogImages.total}`,
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd migracji URL-i",
        description: error.details || error.message || "Nie udało się zmigrować URL-i",
        variant: "destructive",
      });
    },
  });

  const handleMigrateDbUrls = () => {
    if (!settings?.is_active || settings?.provider !== 'sftp') {
      toast({
        title: "SFTP nie aktywny",
        description: "Najpierw zapisz i aktywuj konfigurację SFTP",
        variant: "destructive",
      });
      return;
    }
    
    if (!confirm("Czy na pewno chcesz zmigrować URL-e zdjęć w matrycach i katalogowych produktach na SFTP?")) {
      return;
    }
    
    migrateDbUrlsMutation.mutate();
  };

  if (isLoading) {
    return (
      <Card>
        <CardContent className="p-6">
          <div className="flex items-center justify-center">
            <Loader2 className="h-6 w-6 animate-spin" />
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Server className="h-5 w-5" />
          Serwer plików
        </CardTitle>
        <CardDescription>
          Konfiguracja zewnętrznego serwera plików (zdjęcia, PDF, CSV)
        </CardDescription>
      </CardHeader>
      <CardContent className="space-y-6">
        {/* Connection Status */}
        {settings?.connection_status && (
          <div className={`p-3 rounded-md flex items-center gap-2 ${
            settings.connection_status === 'connected' 
              ? 'bg-green-50 dark:bg-green-950 text-green-700 dark:text-green-400'
              : 'bg-red-50 dark:bg-red-950 text-red-700 dark:text-red-400'
          }`}>
            {settings.connection_status === 'connected' ? (
              <CheckCircle2 className="h-4 w-4" />
            ) : (
              <AlertCircle className="h-4 w-4" />
            )}
            <span className="text-sm font-medium">
              {settings.connection_status === 'connected' 
                ? 'Połączono z serwerem plików' 
                : `Błąd połączenia: ${settings.last_error || 'Unknown error'}`}
            </span>
          </div>
        )}

        {/* Provider Selection */}
        <div className="space-y-2">
          <Label htmlFor="provider">Typ storage</Label>
          <Select value={provider} onValueChange={(v) => setProvider(v as "local" | "sftp")}>
            <SelectTrigger id="provider" data-testid="select-provider">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="local">Lokalny (uploads/)</SelectItem>
              <SelectItem value="sftp">SFTP (zewnętrzny serwer)</SelectItem>
            </SelectContent>
          </Select>
          <p className="text-xs text-muted-foreground">
            Local: pliki w katalogu aplikacji (gubione przy redeploy). SFTP: trwały serwer zewnętrzny.
          </p>
        </div>

        {/* SFTP Configuration */}
        {provider === "sftp" && (
          <>
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="host">Host</Label>
                <Input
                  id="host"
                  value={host}
                  onChange={(e) => setHost(e.target.value)}
                  placeholder="files.alpsys.pl"
                  data-testid="input-host"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="port">Port</Label>
                <Input
                  id="port"
                  type="number"
                  value={port}
                  onChange={(e) => setPort(e.target.value)}
                  placeholder="22"
                  data-testid="input-port"
                />
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="username">Username</Label>
              <Input
                id="username"
                value={username}
                onChange={(e) => setUsername(e.target.value)}
                placeholder="oms-sftp"
                data-testid="input-username"
              />
              <p className="text-xs text-muted-foreground">
                Hasło jest przechowywane jako FILE_STORAGE_PASSWORD w Replit Secrets
              </p>
            </div>

            <div className="space-y-2">
              <Label htmlFor="baseUrl">Base URL</Label>
              <Input
                id="baseUrl"
                value={baseUrl}
                onChange={(e) => setBaseUrl(e.target.value)}
                placeholder="https://files.alpsys.pl"
                data-testid="input-base-url"
              />
              <p className="text-xs text-muted-foreground">
                URL publiczny do dostępu do plików (bez trailing slash)
              </p>
            </div>

            <div className="space-y-2">
              <Label htmlFor="pathPrefix">Path Prefix</Label>
              <Input
                id="pathPrefix"
                value={pathPrefix}
                onChange={(e) => setPathPrefix(e.target.value)}
                placeholder="/OMS"
                data-testid="input-path-prefix"
              />
              <p className="text-xs text-muted-foreground">
                Katalog główny na serwerze (np. /OMS)
              </p>
            </div>

            <div className="flex items-center justify-between p-3 border rounded-md">
              <div className="space-y-0.5">
                <Label htmlFor="test-connection">Test połączenia</Label>
                <p className="text-sm text-muted-foreground">
                  Sprawdź czy można połączyć się z serwerem
                </p>
              </div>
              <Button
                onClick={handleTest}
                disabled={testMutation.isPending}
                variant="outline"
                data-testid="button-test-connection"
              >
                {testMutation.isPending ? (
                  <>
                    <Loader2 className="h-4 w-4 animate-spin" />
                    Testowanie...
                  </>
                ) : (
                  "Test"
                )}
              </Button>
            </div>
          </>
        )}

        {/* Active Switch */}
        <div className="flex items-center justify-between p-3 border rounded-md">
          <div className="space-y-0.5">
            <Label htmlFor="is-active">Aktywuj konfigurację</Label>
            <p className="text-sm text-muted-foreground">
              Użyj tej konfiguracji dla wszystkich nowych plików
            </p>
          </div>
          <Switch
            id="is-active"
            checked={isActive}
            onCheckedChange={setIsActive}
            data-testid="switch-is-active"
          />
        </div>

        {/* Migration Buttons */}
        {provider === "sftp" && settings?.is_active && (
          <div className="p-4 border-2 border-dashed rounded-md bg-muted/20 space-y-4">
            {/* Migrate Database URLs */}
            <div className="space-y-2">
              <Label>Migracja URL-i w bazie danych</Label>
              <p className="text-sm text-muted-foreground">
                Zaktualizuj URL-e zdjęć w matrycach i katalogowych produktach na SFTP
              </p>
              <Button
                onClick={handleMigrateDbUrls}
                disabled={migrateDbUrlsMutation.isPending}
                variant="default"
                data-testid="button-migrate-db-urls"
                className="w-full"
              >
                {migrateDbUrlsMutation.isPending ? (
                  <>
                    <Loader2 className="h-4 w-4 animate-spin mr-2" />
                    Migracja URL-i w toku...
                  </>
                ) : (
                  "Migruj URL-e (Matryc + Katalog)"
                )}
              </Button>
            </div>

            {/* Migrate Physical Files */}
            <div className="space-y-2">
              <Label>Migracja fizycznych plików</Label>
              <p className="text-sm text-muted-foreground">
                Przenieś ~1000 plików z lokalnego storage na SFTP server
              </p>
              <Button
                onClick={handleMigrate}
                disabled={migrateMutation.isPending}
                variant="secondary"
                data-testid="button-migrate-files"
                className="w-full"
              >
                {migrateMutation.isPending ? (
                  <>
                    <Loader2 className="h-4 w-4 animate-spin mr-2" />
                    Migracja w toku... ({migrateMutation.isPending ? 'to może zająć kilka minut' : ''})
                  </>
                ) : (
                  "Migruj pliki na SFTP"
                )}
              </Button>
            </div>
          </div>
        )}

        {/* Save Button */}
        <Button
          onClick={handleSave}
          disabled={saveMutation.isPending}
          data-testid="button-save-file-storage"
          className="w-full"
        >
          {saveMutation.isPending ? (
            <>
              <Loader2 className="h-4 w-4 animate-spin" />
              Zapisywanie...
            </>
          ) : (
            "Zapisz konfigurację"
          )}
        </Button>
      </CardContent>
    </Card>
  );
}
