import { useState } from "react";
import { Check, ChevronsUpDown } from "lucide-react";
import { cn } from "@/lib/utils";
import { Button } from "@/components/ui/button";
import {
  Command,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
} from "@/components/ui/command";
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/ui/popover";

interface DictionaryItem {
  id: number;
  code: string;
  name: string;
  readableName: string | null;
  isActive: boolean;
}

interface DictionaryComboboxProps {
  items: DictionaryItem[] | undefined;
  value: string;
  onChange: (value: string) => void;
  placeholder?: string;
  searchPlaceholder?: string;
  emptyText?: string;
  testId?: string;
  valueField?: 'name' | 'code';
  displayField?: 'name' | 'readableName';
}

export function DictionaryCombobox({
  items,
  value,
  onChange,
  placeholder = "Wybierz...",
  searchPlaceholder = "Szukaj...",
  emptyText = "Nie znaleziono.",
  testId,
  valueField = 'name',
  displayField = 'readableName',
}: DictionaryComboboxProps) {
  const [open, setOpen] = useState(false);

  const getDisplayName = (item: DictionaryItem) => {
    if (displayField === 'name') {
      return item.name;
    }
    return item.readableName || item.name;
  };

  const activeItems = (items?.filter(item => item.isActive) || [])
    .sort((a, b) => {
      const nameA = getDisplayName(a).toLowerCase();
      const nameB = getDisplayName(b).toLowerCase();
      return nameA.localeCompare(nameB);
    });
  const selectedItem = activeItems.find(item => item[valueField] === value);

  return (
    <Popover open={open} onOpenChange={setOpen}>
      <PopoverTrigger asChild>
        <Button
          variant="outline"
          role="combobox"
          aria-expanded={open}
          className="w-full h-8 justify-between text-sm"
          data-testid={testId}
        >
          {selectedItem ? getDisplayName(selectedItem) : placeholder}
          <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
        </Button>
      </PopoverTrigger>
      <PopoverContent className="w-[--radix-popover-trigger-width] p-0" align="start">
        <Command>
          <CommandInput placeholder={searchPlaceholder} />
          <CommandList>
            <CommandEmpty>{emptyText}</CommandEmpty>
            <CommandGroup>
              {activeItems.map((item) => {
                const itemValue = item[valueField];
                const displayName = getDisplayName(item);
                return (
                  <CommandItem
                    key={item.id}
                    value={displayName}
                    onSelect={() => {
                      onChange(itemValue === value ? "" : itemValue);
                      setOpen(false);
                    }}
                    data-testid={`option-${item.name}`}
                  >
                    <Check
                      className={cn(
                        "mr-2 h-4 w-4",
                        value === itemValue ? "opacity-100" : "opacity-0"
                      )}
                    />
                    {displayName}
                  </CommandItem>
                );
              })}
            </CommandGroup>
          </CommandList>
        </Command>
      </PopoverContent>
    </Popover>
  );
}
