import { useState, useEffect } from "react";
import { Settings, Package, CalendarRange, LogOut, Users, Key, Webhook, BookOpen, Server, History, GitCompare, Image, FolderTree, FileText, Sparkles, BookMarked, Grid3x3, TrendingDown, AlertCircle, Activity, Clock, Wrench, ChevronDown, ShoppingCart, Box, ExternalLink, Boxes, LayoutGrid, FileStack, CheckCircle2, RefreshCw, DollarSign, Store, Warehouse, FileBox, Cylinder } from "lucide-react";
import { Link, useLocation } from "wouter";
import { useQuery } from "@tanstack/react-query";
import {
  Sidebar,
  SidebarContent,
  SidebarGroup,
  SidebarGroupContent,
  SidebarGroupLabel,
  SidebarMenu,
  SidebarMenuButton,
  SidebarMenuItem,
  SidebarMenuSub,
  SidebarMenuSubButton,
  SidebarMenuSubItem,
  SidebarHeader,
  SidebarFooter,
} from "@/components/ui/sidebar";
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from "@/components/ui/collapsible";
import { useAuth } from "@/hooks/use-auth";
import { Button } from "@/components/ui/button";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import { canAccessPage, type UserRole } from "@shared/permissions";
import type { Permission } from "@shared/permissions";

// Define menu item types
interface MenuItem {
  id: string;
  title: string;
  url: string;
  testId: string;
  permission: Permission;
  disabled?: boolean;
  icon?: any; // Optional icon for menu item
  iconColor?: string; // Optional color for icon
}

interface StandardMenuItem {
  title: string;
  url: string;
  icon: any;
  testId: string;
  permission: Permission;
  external?: boolean;
}

interface MenuSection {
  id: string;
  title: string;
  icon: any;
  defaultOpen: boolean;
  items: MenuItem[];
}

// Define menu structure with nested items
const menuStructure: {
  collapsibleSections: MenuSection[];
  standardItems: StandardMenuItem[];
} = {
  collapsibleSections: [
    {
      id: "produkty",
      title: "Produkty",
      icon: Box,
      defaultOpen: true,
      items: [
        {
          id: "catalog-products",
          title: "Katalog produktów",
          url: "/catalog-products",
          testId: "link-catalog-products",
          permission: "view_orders" as Permission,
          icon: Package,
          iconColor: "text-green-600/70",
        },
        {
          id: "catalog-sets",
          title: "Katalog zestawów",
          url: "/catalog-sets",
          testId: "link-catalog-sets",
          permission: "view_orders" as Permission,
          icon: Boxes,
          iconColor: "text-green-600/70",
        },
        {
          id: "product-matrices",
          title: "Matryce produktów",
          url: "/product-matrices",
          testId: "link-product-matrices",
          permission: "view_orders" as Permission,
          icon: Grid3x3,
          iconColor: "text-blue-600/70",
        },
        {
          id: "set-matrices",
          title: "Matryce zestawów",
          url: "/set-matrices",
          testId: "link-set-matrices",
          permission: "view_orders" as Permission,
          icon: LayoutGrid,
          iconColor: "text-blue-600/70",
        },
        {
          id: "templates",
          title: "Szablony opisów",
          url: "/templates",
          testId: "link-templates",
          permission: "view_orders" as Permission,
          icon: FileText,
        },
        {
          id: "dictionaries",
          title: "Słowniki",
          url: "/dictionaries",
          testId: "link-dictionaries",
          permission: "view_orders" as Permission,
          icon: BookOpen,
        },
        {
          id: "accessories",
          title: "Akcesoria",
          url: "/accessories",
          testId: "link-accessories",
          permission: "view_orders" as Permission,
          icon: Wrench,
        },
      ],
    },
    {
      id: "magazyn",
      title: "Magazyn",
      icon: Warehouse,
      defaultOpen: true,
      items: [
        {
          id: "okucia",
          title: "Okucia",
          url: "/warehouse/okucia",
          testId: "link-warehouse-okucia",
          permission: "view_orders" as Permission,
          icon: Wrench,
          iconColor: "text-gray-600/70",
        },
        {
          id: "plyty",
          title: "Płyty meblowe",
          url: "/warehouse/plyty",
          testId: "link-warehouse-plyty",
          permission: "view_orders" as Permission,
          icon: FileBox,
          iconColor: "text-amber-600/70",
        },
        {
          id: "sruby",
          title: "Śruby",
          url: "/warehouse/sruby",
          testId: "link-warehouse-sruby",
          permission: "view_orders" as Permission,
          icon: Cylinder,
          iconColor: "text-slate-600/70",
        },
        {
          id: "tkaniny",
          title: "Tkaniny",
          url: "/warehouse/tkaniny",
          testId: "link-warehouse-tkaniny",
          permission: "view_orders" as Permission,
          icon: Package,
          iconColor: "text-purple-600/70",
        },
        {
          id: "pianki",
          title: "Pianki",
          url: "/warehouse/pianki",
          testId: "link-warehouse-pianki",
          permission: "view_orders" as Permission,
          icon: Box,
          iconColor: "text-blue-600/70",
        },
      ],
    },
    {
      id: "sprzedaz",
      title: "Sprzedaż",
      icon: ShoppingCart,
      defaultOpen: true,
      items: [
        {
          id: "orders",
          title: "Zamówienia",
          url: "/",
          testId: "link-orders",
          permission: "view_orders" as Permission,
          icon: FileStack,
          iconColor: "text-orange-600",
        },
        {
          id: "order-changes",
          title: "Historia zamówień",
          url: "/order-changes",
          testId: "link-order-changes",
          permission: "view_recent_updates" as Permission,
          icon: History,
          iconColor: "text-purple-600",
        },
        {
          id: "order-matching",
          title: "Weryfikacja zamówień",
          url: "/order-matching",
          testId: "link-order-matching",
          permission: "view_orders" as Permission,
          icon: CheckCircle2,
          iconColor: "text-emerald-600",
        },
        {
          id: "recent-updates",
          title: "Ostatnie aktualizacje",
          url: "/recent-updates",
          testId: "link-recent-updates",
          permission: "view_orders" as Permission,
          icon: Clock,
          iconColor: "text-sky-600",
        },
        {
          id: "fees",
          title: "Opłaty",
          url: "/fees/summary",
          testId: "link-fees",
          permission: "view_orders" as Permission,
          icon: DollarSign,
          iconColor: "text-amber-600",
        },
        {
          id: "products-marketplace",
          title: "Marketplace",
          url: "/products",
          testId: "link-products",
          permission: "view_orders" as Permission,
          icon: Store,
          iconColor: "text-indigo-600",
        },
        {
          id: "date-sync",
          title: "Synchronizacja wg dat",
          url: "/date-sync",
          testId: "link-date-sync",
          permission: "manage_sync" as Permission,
          icon: CalendarRange,
          iconColor: "text-cyan-600",
        },
        {
          id: "sync-health",
          title: "Monitor synchronizacji",
          url: "/sync-health",
          testId: "link-sync-health",
          permission: "manage_settings" as Permission,
          icon: Activity,
          iconColor: "text-rose-600",
        },
      ],
    },
  ],
  standardItems: [
    {
      title: "Konfiguracja",
      url: "/settings",
      icon: Settings,
      testId: "link-settings",
      permission: "manage_settings" as Permission,
    },
    {
      title: "Użytkownicy",
      url: "/users",
      icon: Users,
      testId: "link-users",
      permission: "manage_users" as Permission,
    },
    {
      title: "Logi błędów",
      url: "/error-logs",
      icon: AlertCircle,
      testId: "link-error-logs",
      permission: "manage_users" as Permission,
    },
    {
      title: "Tokeny API",
      url: "/api-tokens",
      icon: Key,
      testId: "link-api-tokens",
      permission: "manage_api_tokens" as Permission,
    },
    {
      title: "Webhooks",
      url: "/webhooks",
      icon: Webhook,
      testId: "link-webhooks",
      permission: "manage_webhooks" as Permission,
    },
    {
      title: "Integracja Odoo",
      url: "/odoo",
      icon: Server,
      testId: "link-odoo",
      permission: "manage_settings" as Permission,
    },
    {
      title: "Ustawienia AI",
      url: "/ai-settings",
      icon: Sparkles,
      testId: "link-ai-settings",
      permission: "manage_settings" as Permission,
    },
    {
      title: "Dokumentacja API",
      url: "/api-docs",
      icon: BookOpen,
      testId: "link-api-docs",
      permission: "manage_api_tokens" as Permission,
      external: true,
    },
  ],
};

const STORAGE_KEY = "sidebar-menu-state";

interface StatsResponse {
  products: {
    catalogProducts: number;
    catalogSets: number;
    productMatrices: number;
    setMatrices: number;
    accessories: number;
  };
  warehouse: {
    categories: Record<string, number>;
    groups: Array<{
      id: number;
      code: string;
      name: string;
      category: string;
      count: number;
    }>;
  };
}

export function AppSidebar() {
  const [location] = useLocation();
  const { user, logoutMutation } = useAuth();

  // Fetch stats counts for menu badges
  const { data: stats } = useQuery<StatsResponse>({
    queryKey: ['/api/stats/counts'],
    refetchInterval: 30000, // Refresh every 30 seconds
  });

  // Load initial state from localStorage
  const getInitialState = () => {
    try {
      const saved = localStorage.getItem(STORAGE_KEY);
      if (saved) {
        return JSON.parse(saved);
      }
    } catch (e) {
      console.error("Failed to load sidebar state:", e);
    }
    // Default state - all sections open
    return menuStructure.collapsibleSections.reduce((acc, section) => {
      acc[section.id] = section.defaultOpen;
      return acc;
    }, {} as Record<string, boolean>);
  };

  const [openSections, setOpenSections] = useState<Record<string, boolean>>(getInitialState);

  // Save state to localStorage whenever it changes
  useEffect(() => {
    try {
      localStorage.setItem(STORAGE_KEY, JSON.stringify(openSections));
    } catch (e) {
      console.error("Failed to save sidebar state:", e);
    }
  }, [openSections]);

  const toggleSection = (sectionId: string) => {
    setOpenSections((prev) => ({
      ...prev,
      [sectionId]: !prev[sectionId],
    }));
  };

  const handleLogout = () => {
    logoutMutation.mutate();
  };

  const getUserInitials = () => {
    if (user?.firstName && user?.lastName) {
      return `${user.firstName[0]}${user.lastName[0]}`.toUpperCase();
    }
    if (user?.username) {
      return user.username.slice(0, 2).toUpperCase();
    }
    return "U";
  };

  // Check if a path is active (including nested routes)
  const isPathActive = (url: string) => {
    if (url === "/") {
      return location === "/";
    }
    return location.startsWith(url);
  };

  // Helper: Get count for specific menu item
  const getItemCount = (itemId: string): number | null => {
    if (!stats) return null;

    // Produkty
    if (itemId === 'catalog-products') return stats.products.catalogProducts;
    if (itemId === 'catalog-sets') return stats.products.catalogSets;
    if (itemId === 'product-matrices') return stats.products.productMatrices;
    if (itemId === 'set-matrices') return stats.products.setMatrices;
    if (itemId === 'accessories') return stats.products.accessories;

    // Magazyn - kategorias
    if (itemId === 'okucia') return stats.warehouse.categories['okucia'] || 0;
    if (itemId === 'plyty') return stats.warehouse.categories['plyty_meblowe'] || 0;
    if (itemId === 'sruby') return stats.warehouse.categories['sruby'] || 0;
    if (itemId === 'tkaniny') return stats.warehouse.categories['tkaniny'] || 0;
    if (itemId === 'pianki') return stats.warehouse.categories['pianki'] || 0;

    return null;
  };

  // Helper: Get total count for section
  const getSectionCount = (sectionId: string): number | null => {
    if (!stats) return null;

    if (sectionId === 'produkty') {
      return (
        stats.products.catalogProducts +
        stats.products.catalogSets +
        stats.products.productMatrices +
        stats.products.setMatrices +
        stats.products.accessories
      );
    }

    if (sectionId === 'magazyn') {
      return Object.values(stats.warehouse.categories).reduce((sum, count) => sum + count, 0);
    }

    return null;
  };

  return (
    <Sidebar>
      <SidebarHeader className="p-4">
        <div className="flex items-center gap-3">
          <img 
            src="/alpma-logo.png" 
            alt="Alpma Logo" 
            className="h-8 w-8 object-contain"
          />
          <div>
            <h1 className="font-semibold text-sm">Alpma OMS</h1>
            <p className="text-xs text-muted-foreground">Zarządzanie zamówieniami</p>
          </div>
        </div>
      </SidebarHeader>
      <SidebarContent>
        <SidebarGroup>
          <SidebarGroupLabel>Menu</SidebarGroupLabel>
          <SidebarGroupContent>
            <SidebarMenu>
              {/* Collapsible sections */}
              {menuStructure.collapsibleSections.map((section) => {
                const visibleItems = section.items.filter(
                  (item) => user?.role && canAccessPage(user.role as UserRole, item.url)
                );

                if (visibleItems.length === 0) return null;

                return (
                  <Collapsible
                    key={section.id}
                    open={openSections[section.id]}
                    onOpenChange={() => toggleSection(section.id)}
                  >
                    <SidebarMenuItem>
                      <CollapsibleTrigger asChild>
                        <SidebarMenuButton data-testid={`button-toggle-${section.id}`}>
                          <section.icon className="h-4 w-4" />
                          <span className="flex items-center gap-1.5">
                            {section.title}
                            {getSectionCount(section.id) !== null && (
                              <span className="text-xs text-muted-foreground">
                                ({getSectionCount(section.id)})
                              </span>
                            )}
                          </span>
                          <ChevronDown
                            className={`ml-auto h-4 w-4 transition-transform ${
                              openSections[section.id] ? "rotate-180" : ""
                            }`}
                          />
                        </SidebarMenuButton>
                      </CollapsibleTrigger>
                      <CollapsibleContent>
                        <SidebarMenuSub className="bg-sidebar-accent/30 mx-2 px-2 py-1">
                          {visibleItems.map((item) => (
                            <SidebarMenuSubItem key={item.id}>
                              <div className="relative group/menuitem">
                                <SidebarMenuSubButton
                                  asChild={!item.disabled}
                                  isActive={isPathActive(item.url)}
                                  data-testid={item.testId}
                                >
                                  {item.disabled ? (
                                    <span className="opacity-50 cursor-not-allowed flex items-center gap-2">
                                      {item.icon && (() => {
                                        const Icon = item.icon;
                                        const iconStyle = item.iconColor ? {
                                          color: item.iconColor.includes('green') ? 'rgb(22 163 74 / 0.7)' :
                                                 item.iconColor.includes('blue') ? 'rgb(37 99 235 / 0.7)' :
                                                 item.iconColor.includes('orange') ? 'rgb(234 88 12)' :
                                                 item.iconColor.includes('purple') ? 'rgb(147 51 234)' :
                                                 item.iconColor.includes('emerald') ? 'rgb(5 150 105)' :
                                                 item.iconColor.includes('sky') ? 'rgb(2 132 199)' :
                                                 item.iconColor.includes('amber') ? 'rgb(217 119 6)' :
                                                 item.iconColor.includes('indigo') ? 'rgb(79 70 229)' :
                                                 item.iconColor.includes('cyan') ? 'rgb(8 145 178)' :
                                                 item.iconColor.includes('rose') ? 'rgb(225 29 72)' :
                                                 undefined
                                        } : undefined;
                                        return <Icon className="h-4 w-4 shrink-0" style={iconStyle} />;
                                      })()}
                                      <span className="flex items-center gap-1.5">
                                        {item.title}
                                        {getItemCount(item.id) !== null && (
                                          <span className="text-xs text-muted-foreground">
                                            ({getItemCount(item.id)})
                                          </span>
                                        )}
                                      </span>
                                    </span>
                                  ) : (
                                    <Link href={item.url} className="flex items-center gap-2">
                                      {item.icon && (() => {
                                        const Icon = item.icon;
                                        const iconStyle = item.iconColor ? {
                                          color: item.iconColor.includes('green') ? 'rgb(22 163 74 / 0.7)' :
                                                 item.iconColor.includes('blue') ? 'rgb(37 99 235 / 0.7)' :
                                                 item.iconColor.includes('orange') ? 'rgb(234 88 12)' :
                                                 item.iconColor.includes('purple') ? 'rgb(147 51 234)' :
                                                 item.iconColor.includes('emerald') ? 'rgb(5 150 105)' :
                                                 item.iconColor.includes('sky') ? 'rgb(2 132 199)' :
                                                 item.iconColor.includes('amber') ? 'rgb(217 119 6)' :
                                                 item.iconColor.includes('indigo') ? 'rgb(79 70 229)' :
                                                 item.iconColor.includes('cyan') ? 'rgb(8 145 178)' :
                                                 item.iconColor.includes('rose') ? 'rgb(225 29 72)' :
                                                 undefined
                                        } : undefined;
                                        return <Icon className="h-4 w-4 shrink-0" style={iconStyle} />;
                                      })()}
                                      <span className="flex items-center gap-1.5">
                                        {item.title}
                                        {getItemCount(item.id) !== null && (
                                          <span className="text-xs text-muted-foreground">
                                            ({getItemCount(item.id)})
                                          </span>
                                        )}
                                      </span>
                                    </Link>
                                  )}
                                </SidebarMenuSubButton>
                                {!item.disabled && (
                                  <button
                                    onClick={(e) => {
                                      e.preventDefault();
                                      e.stopPropagation();
                                      window.open(item.url, '_blank');
                                    }}
                                    className="absolute right-2 top-1/2 -translate-y-1/2 opacity-0 group-hover/menuitem:opacity-100 transition-opacity p-1 bg-primary/10 hover:bg-primary/20 rounded"
                                    data-testid={`button-external-${item.testId}`}
                                  >
                                    <ExternalLink className="h-3 w-3 text-muted-foreground" />
                                  </button>
                                )}
                              </div>
                            </SidebarMenuSubItem>
                          ))}
                        </SidebarMenuSub>
                      </CollapsibleContent>
                    </SidebarMenuItem>
                  </Collapsible>
                );
              })}

              {/* Standard menu items */}
              {menuStructure.standardItems
                .filter(
                  (item) => user?.role && canAccessPage(user.role as UserRole, item.url)
                )
                .map((item) => (
                  <SidebarMenuItem key={item.title}>
                    <div className="relative group/menuitem">
                      <SidebarMenuButton
                        asChild
                        isActive={isPathActive(item.url)}
                        data-testid={item.testId}
                      >
                        {(item as any).external ? (
                          <a href={item.url}>
                            <item.icon className="h-4 w-4" />
                            <span>{item.title}</span>
                          </a>
                        ) : (
                          <Link href={item.url}>
                            <item.icon className="h-4 w-4" />
                            <span>{item.title}</span>
                          </Link>
                        )}
                      </SidebarMenuButton>
                      <button
                        onClick={(e) => {
                          e.preventDefault();
                          e.stopPropagation();
                          window.open(item.url, '_blank');
                        }}
                        className="absolute right-2 top-1/2 -translate-y-1/2 opacity-0 group-hover/menuitem:opacity-100 transition-opacity p-1 bg-primary/10 hover:bg-primary/20 rounded"
                        data-testid={`button-external-${item.testId}`}
                      >
                        <ExternalLink className="h-3 w-3 text-muted-foreground" />
                      </button>
                    </div>
                  </SidebarMenuItem>
                ))}
            </SidebarMenu>
          </SidebarGroupContent>
        </SidebarGroup>
      </SidebarContent>
      <SidebarFooter className="p-4 border-t">
        {user && (
          <div className="space-y-3">
            <div className="flex items-center gap-3">
              <Avatar className="h-8 w-8">
                <AvatarFallback className="bg-primary/10 text-primary">
                  {getUserInitials()}
                </AvatarFallback>
              </Avatar>
              <div className="flex-1 min-w-0">
                <p className="text-sm font-medium truncate" data-testid="text-username">
                  {user.firstName && user.lastName 
                    ? `${user.firstName} ${user.lastName}`
                    : user.username
                  }
                </p>
                <p className="text-xs text-muted-foreground truncate" data-testid="text-email">
                  {user.email}
                </p>
              </div>
            </div>
            <Button
              variant="outline"
              size="sm"
              className="w-full"
              onClick={handleLogout}
              disabled={logoutMutation.isPending}
              data-testid="button-logout"
            >
              <LogOut className="h-4 w-4 mr-2" />
              {logoutMutation.isPending ? "Wylogowywanie..." : "Wyloguj się"}
            </Button>
          </div>
        )}
        <p className="text-xs text-muted-foreground text-center mt-3">
          © 2025 Alpma OMS
        </p>
      </SidebarFooter>
    </Sidebar>
  );
}
