import { useQuery } from "@tanstack/react-query";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { Button } from "@/components/ui/button";
import { AlertCircle, RefreshCw, ChevronDown, ChevronUp } from "lucide-react";
import { Link } from "wouter";
import { useState } from "react";

export function AllegroConnectionAlert() {
  const [isExpanded, setIsExpanded] = useState(false);
  const { data: status, refetch } = useQuery<{
    connected: boolean;
    requiresReauth: boolean;
    error: string | null;
    lastErrorAt: string | null;
  }>({
    queryKey: ['/api/allegro/connection-status'],
    refetchInterval: 60000, // Check every minute
    retry: false,
  });

  if (!status?.requiresReauth) {
    return null;
  }

  return (
    <>
      {/* Mobile: Compact collapsible alert */}
      <div className="md:hidden mb-2">
        <button
          onClick={() => setIsExpanded(!isExpanded)}
          className="w-full flex items-center gap-2 bg-destructive/10 border border-destructive/30 rounded-md px-3 py-2 text-sm text-destructive hover:bg-destructive/15 transition-colors"
          data-testid="button-mobile-alert-toggle"
        >
          <AlertCircle className="h-4 w-4 flex-shrink-0" />
          <span className="flex-1 text-left font-medium truncate">
            Wymagane połączenie z Allegro
          </span>
          {isExpanded ? (
            <ChevronUp className="h-4 w-4 flex-shrink-0" />
          ) : (
            <ChevronDown className="h-4 w-4 flex-shrink-0" />
          )}
        </button>
        
        {isExpanded && (
          <div className="mt-2 p-3 bg-destructive/5 border border-destructive/20 rounded-md text-sm space-y-2">
            <p className="text-destructive">
              {status.error || 'Token dostępu wygasł.'}
            </p>
            <div className="flex gap-2">
              <Link href="/settings" className="flex-1">
                <Button 
                  variant="outline" 
                  size="sm"
                  className="w-full bg-background hover:bg-background/80 text-xs"
                  data-testid="button-goto-settings-mobile"
                >
                  Ustawienia
                </Button>
              </Link>
              <Button 
                variant="outline" 
                size="sm"
                className="bg-background hover:bg-background/80 text-xs"
                onClick={() => refetch()}
                data-testid="button-refresh-status-mobile"
              >
                <RefreshCw className="h-3 w-3" />
              </Button>
            </div>
          </div>
        )}
      </div>

      {/* Desktop: Full alert */}
      <Alert variant="destructive" className="hidden md:flex mb-4" data-testid="alert-allegro-connection">
        <AlertCircle className="h-4 w-4" />
        <AlertTitle>Wymagane ponowne połączenie z Allegro</AlertTitle>
        <AlertDescription className="mt-2">
          <div className="flex flex-col gap-2">
            <p>
              {status.error || 'Token dostępu wygasł. Połącz się ponownie z Allegro aby kontynuować synchronizację zamówień.'}
            </p>
            <div className="flex gap-2 mt-2">
              <Link href="/settings">
                <Button 
                  variant="outline" 
                  size="sm"
                  className="bg-background hover:bg-background/80"
                  data-testid="button-goto-settings"
                >
                  Przejdź do ustawień
                </Button>
              </Link>
              <Button 
                variant="outline" 
                size="sm"
                className="bg-background hover:bg-background/80"
                onClick={() => refetch()}
                data-testid="button-refresh-status"
              >
                <RefreshCw className="h-4 w-4 mr-2" />
                Odśwież status
              </Button>
            </div>
          </div>
        </AlertDescription>
      </Alert>
    </>
  );
}
