import { useQuery } from "@tanstack/react-query";
import { Wallet, Clock, Lock } from "lucide-react";

interface AllegroBalanceData {
  available: { [currency: string]: number };
  pending: { [currency: string]: number };
  holds: { [currency: string]: number };
  lastOperation: any;
}

export function AllegroBalanceWidget() {
  const { data, isLoading, error } = useQuery<AllegroBalanceData>({
    queryKey: ["/api/allegro/balance"],
    refetchInterval: 5 * 60 * 1000,
  });

  if (error) {
    return null;
  }

  if (isLoading) {
    return (
      <div className="flex items-center gap-4 py-2 px-4 bg-muted/30 rounded-md border text-sm">
        <div className="h-4 w-32 bg-muted rounded animate-pulse" />
      </div>
    );
  }

  const availablePLN = data?.available?.PLN || 0;
  const pendingPLN = data?.pending?.PLN || 0;
  const holdsPLN = data?.holds?.PLN || 0;

  const formatAmount = (amount: number) => {
    return new Intl.NumberFormat('pl-PL', {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(amount);
  };

  return (
    <div className="flex items-center gap-4 py-2 px-4 bg-muted/30 rounded-md border text-sm">
      <div className="flex items-center gap-2">
        <a 
          href="https://salescenter.allegro.com/funds-and-operations-history" 
          target="_blank" 
          rel="noopener noreferrer"
          className="flex items-center gap-1.5 text-green-600 dark:text-green-400 hover:underline"
          data-testid="allegro-balance-available"
        >
          <Wallet className="h-4 w-4" />
          <span className="font-medium">
            {formatAmount(availablePLN)} PLN
          </span>
        </a>
        <span className="text-[10px] text-muted-foreground uppercase tracking-wide">Dostępne</span>
      </div>

      <div className="h-4 w-px bg-border" />

      <div className="flex items-center gap-2">
        <div className="flex items-center gap-1.5 text-orange-600 dark:text-orange-400">
          <Clock className="h-4 w-4" />
          <span className="font-medium" data-testid="allegro-balance-pending">
            {formatAmount(pendingPLN)} PLN
          </span>
        </div>
        <span className="text-[10px] text-muted-foreground uppercase tracking-wide">Oczekujące</span>
      </div>

      <div className="h-4 w-px bg-border" />

      <div className="flex items-center gap-2">
        <div className="flex items-center gap-1.5 text-red-600 dark:text-red-400">
          <Lock className="h-4 w-4" />
          <span className="font-medium" data-testid="allegro-balance-holds">
            {formatAmount(holdsPLN)} PLN
          </span>
        </div>
        <span className="text-[10px] text-muted-foreground uppercase tracking-wide">Zatrzymane</span>
      </div>
    </div>
  );
}
