import { useState, useEffect } from "react";
import { useMutation, useQuery } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { Loader2, Sparkles, CheckCircle2 } from "lucide-react";
import { Badge } from "@/components/ui/badge";

interface AIGenerationDialogProps {
  productId: number;
  onGenerated: (description: string) => void;
}

interface AIGenerationRequest {
  id: number;
  productId: number;
  templateId: number | null;
  status: string;
  response: string | null;
  costMetadata: {
    model?: string;
    tokens?: { prompt: number; completion: number; total: number };
    cost?: number;
    templateName?: string;
    tone?: string;
    language?: string;
  } | null;
  createdAt: string;
}

export function AIGenerationDialog({ productId, onGenerated }: AIGenerationDialogProps) {
  const [open, setOpen] = useState(false);
  const [selectedTemplateId, setSelectedTemplateId] = useState<string | null>(null);
  const [tone, setTone] = useState<string>("professional");
  const [language, setLanguage] = useState<string>("pl");
  const [generatedDescription, setGeneratedDescription] = useState<string>("");
  const { toast } = useToast();

  // Pobierz dane produktu (aby sprawdzić matrixId)
  const { data: product } = useQuery<{ matrixId?: number | null }>({
    queryKey: [`/api/catalog-products/${productId}`],
  });

  // Pobierz dane matrycy jeśli produkt ma matrixId
  const { data: matrix } = useQuery<{ templateId?: number | null }>({
    queryKey: [`/api/product-matrices/${product?.matrixId}`],
    enabled: !!product?.matrixId,
  });

  // Automatycznie preselektuj szablon z matrycy
  useEffect(() => {
    if (matrix?.templateId && !selectedTemplateId) {
      setSelectedTemplateId(matrix.templateId.toString());
    }
  }, [matrix?.templateId]);

  const { data: templates } = useQuery<Array<{ id: number; name: string; categoryName?: string }>>({
    queryKey: ["/api/description-templates"],
  });

  const generateMutation = useMutation({
    mutationFn: async () => {
      const payload: any = {
        tone,
        language,
        maxLength: 500,
      };
      
      if (selectedTemplateId) {
        payload.templateId = parseInt(selectedTemplateId);
      }
      
      const response = await apiRequest(
        "POST",
        `/api/catalog-products/${productId}/ai-generate`,
        payload
      );
      
      return await response.json();
    },
    onSuccess: (data) => {
      if (data.response) {
        setGeneratedDescription(data.response);
        toast({
          title: "Opis wygenerowany!",
          description: `Koszt: $${data.costMetadata?.cost?.toFixed(4) || '0.00'}`,
        });
      } else {
        toast({
          title: "Błąd",
          description: "Nie udało się wygenerować opisu",
          variant: "destructive",
        });
      }
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd generacji",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const handleGenerate = () => {
    setGeneratedDescription("");
    generateMutation.mutate();
  };

  const handleAccept = () => {
    onGenerated(generatedDescription);
    setOpen(false);
    setGeneratedDescription("");
    toast({
      title: "Opis zapisany",
      description: "Wygenerowany opis został dodany do formularza",
    });
  };

  return (
    <Dialog open={open} onOpenChange={setOpen}>
      <DialogTrigger asChild>
        <Button
          type="button"
          variant="outline"
          size="sm"
          data-testid="button-ai-generate"
        >
          <Sparkles className="h-4 w-4 mr-2" />
          Generuj opis AI
        </Button>
      </DialogTrigger>
      <DialogContent className="max-w-3xl max-h-[80vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle>Generowanie opisu AI</DialogTitle>
          <DialogDescription>
            Wybierz opcje generowania opisu produktu przez sztuczną inteligencję
          </DialogDescription>
        </DialogHeader>

        <div className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="template-select">Szablon (opcjonalny)</Label>
            <Select value={selectedTemplateId || "none"} onValueChange={(value) => setSelectedTemplateId(value === "none" ? null : value)}>
              <SelectTrigger id="template-select" data-testid="select-template">
                <SelectValue placeholder="Wybierz szablon..." />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="none">Bez szablonu</SelectItem>
                {templates?.map((template) => (
                  <SelectItem key={template.id} value={template.id.toString()}>
                    {template.name}
                    {template.categoryName && (
                      <Badge variant="secondary" className="ml-2">
                        {template.categoryName}
                      </Badge>
                    )}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
            {matrix?.templateId && selectedTemplateId === matrix.templateId.toString() && (
              <p className="text-xs text-muted-foreground">
                ✓ Szablon automatycznie wybrany z matrycy produktowej
              </p>
            )}
          </div>

          <div className="grid grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="tone-select">Ton wypowiedzi</Label>
              <Select value={tone} onValueChange={setTone}>
                <SelectTrigger id="tone-select" data-testid="select-tone">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="professional">Profesjonalny</SelectItem>
                  <SelectItem value="friendly">Przyjazny</SelectItem>
                  <SelectItem value="casual">Swobodny</SelectItem>
                  <SelectItem value="formal">Formalny</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="language-select">Język</Label>
              <Select value={language} onValueChange={setLanguage}>
                <SelectTrigger id="language-select" data-testid="select-language">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="pl">Polski</SelectItem>
                  <SelectItem value="en">Angielski</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>

          <Button
            onClick={handleGenerate}
            disabled={generateMutation.isPending}
            className="w-full"
            data-testid="button-generate-ai"
          >
            {generateMutation.isPending ? (
              <>
                <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                Generowanie...
              </>
            ) : (
              <>
                <Sparkles className="h-4 w-4 mr-2" />
                Wygeneruj opis
              </>
            )}
          </Button>

          {generatedDescription && (
            <div className="space-y-2">
              <div className="flex items-center justify-between">
                <Label>Wygenerowany opis</Label>
                <CheckCircle2 className="h-5 w-5 text-green-500" />
              </div>
              <Textarea
                value={generatedDescription}
                onChange={(e) => setGeneratedDescription(e.target.value)}
                rows={12}
                className="font-mono text-sm"
                data-testid="textarea-generated-description"
              />
              <div className="flex gap-2 justify-end">
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => setGeneratedDescription("")}
                  data-testid="button-cancel-generated"
                >
                  Anuluj
                </Button>
                <Button
                  type="button"
                  onClick={handleAccept}
                  data-testid="button-accept-generated"
                >
                  Zastosuj opis
                </Button>
              </div>
            </div>
          )}
        </div>
      </DialogContent>
    </Dialog>
  );
}
