import { useState, useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import { Link, useRoute } from "wouter";
import { Button } from "@/components/ui/button";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Separator } from "@/components/ui/separator";
import { Badge } from "@/components/ui/badge";
import { ChevronDown, ChevronRight, Plus, Loader2, ChevronsDownUp, ChevronsUpDown, Pencil, Copy, MoreVertical, Trash2 } from "lucide-react";
import { cn } from "@/lib/utils";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";

type AccessoryGroup = {
  id: number;
  name: string;
  code: string;
  category: string | null;
  description: string | null;
  displayOrder: number;
  isActive: boolean;
};

interface AccessoriesSubmenuProps {
  onAddNew?: () => void;
  onEditGroup?: (group: AccessoryGroup) => void;
  onDuplicateGroup?: (group: AccessoryGroup) => void;
  onDeleteGroup?: (group: AccessoryGroup) => void;
}

interface StatsResponse {
  products: {
    catalogProducts: number;
    catalogSets: number;
    productMatrices: number;
    setMatrices: number;
    accessories: number;
  };
  warehouse: {
    categories: Record<string, number>;
    groups: Array<{
      id: number;
      code: string;
      name: string;
      category: string;
      count: number;
    }>;
  };
  accessories: {
    groups: Array<{
      id: number;
      code: string;
      name: string;
      category: string;
      count: number;
    }>;
  };
}

const CATEGORY_LABELS: Record<string, string> = {
  "Okucia": "Okucia",
  "Płyty": "Płyty meblowe",
  "Tkaniny": "Tkaniny",
};

const STORAGE_KEY_COLLAPSED = 'accessories-collapsed-categories';
const STORAGE_KEY_SHOW_ALL = 'accessories-show-all';

export function AccessoriesSubmenu({ onAddNew, onEditGroup, onDuplicateGroup, onDeleteGroup }: AccessoriesSubmenuProps) {
  const [, params] = useRoute("/accessories/:groupCode");
  const currentGroupCode = params?.groupCode;

  // Load collapsed state from localStorage
  const [collapsedCategories, setCollapsedCategories] = useState<Set<string>>(() => {
    try {
      const saved = localStorage.getItem(STORAGE_KEY_COLLAPSED);
      if (saved) {
        return new Set(JSON.parse(saved));
      }
    } catch (e) {
      console.error('Failed to load collapsed categories:', e);
    }
    // Default: all categories collapsed
    return new Set(Object.keys(CATEGORY_LABELS));
  });

  // Load show all state from localStorage
  const [showAllGroups, setShowAllGroups] = useState<boolean>(() => {
    try {
      const saved = localStorage.getItem(STORAGE_KEY_SHOW_ALL);
      return saved ? JSON.parse(saved) : false;
    } catch (e) {
      console.error('Failed to load show all state:', e);
      return false;
    }
  });

  // Save collapsed state to localStorage
  useEffect(() => {
    try {
      localStorage.setItem(STORAGE_KEY_COLLAPSED, JSON.stringify(Array.from(collapsedCategories)));
    } catch (e) {
      console.error('Failed to save collapsed categories:', e);
    }
  }, [collapsedCategories]);

  // Save show all state to localStorage
  useEffect(() => {
    try {
      localStorage.setItem(STORAGE_KEY_SHOW_ALL, JSON.stringify(showAllGroups));
    } catch (e) {
      console.error('Failed to save show all state:', e);
    }
  }, [showAllGroups]);

  const { data: groups = [], isLoading } = useQuery<AccessoryGroup[]>({
    queryKey: ["/api/accessory-groups"],
  });

  // Fetch stats for counts
  const { data: stats } = useQuery<StatsResponse>({
    queryKey: ['/api/stats/counts'],
    refetchInterval: 30000,
  });

  // Get count for specific group
  const getGroupCount = (groupCode: string): number | null => {
    if (!stats?.accessories.groups) return null;
    const group = stats.accessories.groups.find(g => g.code === groupCode);
    return group?.count ?? null;
  };

  // Group by category
  const groupedByCategory = groups.reduce((acc, group) => {
    const category = group.category || "Bez kategorii";
    if (!acc[category]) {
      acc[category] = [];
    }
    acc[category].push(group);
    return acc;
  }, {} as Record<string, AccessoryGroup[]>);

  // Sort categories
  const sortedCategories = Object.keys(groupedByCategory).sort((a, b) => {
    if (a === "Bez kategorii") return 1;
    if (b === "Bez kategorii") return -1;
    return a.localeCompare(b);
  });

  const toggleCategory = (category: string) => {
    setCollapsedCategories((prev) => {
      const newSet = new Set(prev);
      if (newSet.has(category)) {
        newSet.delete(category);
      } else {
        newSet.add(category);
      }
      return newSet;
    });
  };

  const toggleExpandAll = () => {
    setShowAllGroups((prev) => {
      const newValue = !prev;
      if (newValue) {
        // Expand all
        setCollapsedCategories(new Set());
      } else {
        // Collapse all
        setCollapsedCategories(new Set(sortedCategories));
      }
      return newValue;
    });
  };

  // Auto-expand category containing current group
  useEffect(() => {
    if (currentGroupCode && groups.length > 0) {
      const currentGroup = groups.find(g => g.code === currentGroupCode);
      if (currentGroup?.category) {
        setCollapsedCategories(prev => {
          const newSet = new Set(prev);
          newSet.delete(currentGroup.category!);
          return newSet;
        });
      }
    }
  }, [currentGroupCode, groups]);

  if (isLoading) {
    return (
      <div className="flex items-center justify-center p-8">
        <Loader2 className="h-6 w-6 animate-spin" />
      </div>
    );
  }

  return (
    <div className="flex flex-col h-full w-64 max-w-64">
      <div className="px-2 py-1.5 space-y-0.5">
        <div className="flex items-center justify-between gap-2">
          <h2 className="text-sm font-semibold">Akcesoria</h2>
          <div className="flex items-center gap-1">
            <Button
              size="sm"
              variant="ghost"
              onClick={toggleExpandAll}
              data-testid="button-toggle-expand-all"
              title={showAllGroups ? "Zwiń wszystkie" : "Rozwiń wszystkie"}
              className="h-5 w-5 p-0"
            >
              {showAllGroups ? (
                <ChevronsDownUp className="h-3 w-3" />
              ) : (
                <ChevronsUpDown className="h-3 w-3" />
              )}
            </Button>
            {onAddNew && (
              <Button
                size="sm"
                variant="outline"
                onClick={onAddNew}
                data-testid="button-add-group"
                className="h-5 w-5 p-0"
              >
                <Plus className="h-3 w-3" />
              </Button>
            )}
          </div>
        </div>
      </div>

      <Separator />

      <ScrollArea className="flex-1">
        <div className="p-1 space-y-2">
          {sortedCategories.map((category) => {
            const categoryGroups = groupedByCategory[category].sort(
              (a, b) => a.displayOrder - b.displayOrder || a.name.localeCompare(b.name)
            );
            const isCollapsed = collapsedCategories.has(category);
            const label = CATEGORY_LABELS[category] || category;

            return (
              <div key={category} className="py-0">
                <Button
                  variant="ghost"
                  className="w-full justify-start px-2 py-1 !h-7 !min-h-0 font-semibold text-sm bg-muted/50 hover:bg-muted"
                  onClick={() => toggleCategory(category)}
                  data-testid={`button-category-${category}`}
                >
                  {isCollapsed ? (
                    <ChevronRight className="h-3 w-3 mr-1 shrink-0" />
                  ) : (
                    <ChevronDown className="h-3 w-3 mr-1 shrink-0" />
                  )}
                  <span className="truncate">{label}</span>
                  <Badge variant="outline" className="ml-auto shrink-0 h-4 text-[10px] px-1">
                    {categoryGroups.length}
                  </Badge>
                </Button>

                {!isCollapsed && (
                  <div className="ml-3 space-y-1 mt-1">
                    {categoryGroups.map((group) => (
                      <div key={group.id} className="flex items-center gap-1 group/item min-w-0">
                        <Link href={`/accessories/${group.code}`} className="flex-1 min-w-0">
                          <Button
                            variant="ghost"
                            className={cn(
                              "w-full justify-start px-2 py-0 !h-6 !min-h-0 text-sm font-normal",
                              currentGroupCode === group.code && "bg-accent"
                            )}
                            data-testid={`link-group-${group.code}`}
                          >
                            <span className="truncate flex items-center gap-1.5">
                              {group.name}
                              {getGroupCount(group.code) !== null && (
                                <span className="text-[10px] text-muted-foreground">
                                  ({getGroupCount(group.code)})
                                </span>
                              )}
                            </span>
                          </Button>
                        </Link>
                        {(onEditGroup || onDuplicateGroup || onDeleteGroup) && (
                          <DropdownMenu>
                            <DropdownMenuTrigger asChild>
                              <Button
                                variant="ghost"
                                size="icon"
                                className="h-6 w-6 shrink-0"
                                onClick={(e) => {
                                  e.preventDefault();
                                  e.stopPropagation();
                                }}
                                data-testid={`button-group-menu-${group.code}`}
                              >
                                <MoreVertical className="h-3.5 w-3.5" />
                              </Button>
                            </DropdownMenuTrigger>
                            <DropdownMenuContent align="end">
                              {onEditGroup && (
                                <DropdownMenuItem
                                  onClick={(e) => {
                                    e.stopPropagation();
                                    onEditGroup(group);
                                  }}
                                  data-testid={`menu-edit-${group.code}`}
                                >
                                  <Pencil className="h-3 w-3 mr-2" />
                                  Edytuj grupę
                                </DropdownMenuItem>
                              )}
                              {onDuplicateGroup && (
                                <DropdownMenuItem
                                  onClick={(e) => {
                                    e.stopPropagation();
                                    onDuplicateGroup(group);
                                  }}
                                  data-testid={`menu-duplicate-${group.code}`}
                                >
                                  <Copy className="h-3 w-3 mr-2" />
                                  Duplikuj z zawartością
                                </DropdownMenuItem>
                              )}
                              {onDeleteGroup && (
                                <DropdownMenuItem
                                  onClick={(e) => {
                                    e.stopPropagation();
                                    onDeleteGroup(group);
                                  }}
                                  data-testid={`menu-delete-${group.code}`}
                                  className="text-destructive focus:text-destructive"
                                >
                                  <Trash2 className="h-3 w-3 mr-2" />
                                  Usuń z zawartością
                                </DropdownMenuItem>
                              )}
                            </DropdownMenuContent>
                          </DropdownMenu>
                        )}
                      </div>
                    ))}
                  </div>
                )}
              </div>
            );
          })}
        </div>
      </ScrollArea>
    </div>
  );
}
