# Alpma OMS - External API Documentation

## Overview

Alpma OMS provides a RESTful API for external integrations (e.g., Odoo ERP) using a hybrid Pull + Push architecture:
- **Pull (REST API)**: Query orders and products on-demand
- **Push (Webhooks)**: Receive real-time notifications for order events

## Authentication

All API requests require Bearer token authentication.

### Request Format
```http
GET /api/external/orders HTTP/1.1
Host: your-domain.replit.app
Authorization: Bearer your_api_token_here
```

### Getting an API Token

API tokens must be created by a system administrator through the admin panel:
1. Login as admin
2. Navigate to API Tokens management
3. Create new token (token displayed only once)
4. Store token securely

## REST API Endpoints

### 1. List Orders

Get paginated list of orders with optional filtering.

**Endpoint:** `GET /api/external/orders`

**Query Parameters:**
- `page` (integer, default: 1) - Page number
- `limit` (integer, default: 50) - Items per page
- `source` (string, optional) - Filter by platform: `allegro` or `shoper`
- `payment_status` (string, optional) - Filter by payment status
- `fulfillment_status` (string, optional) - Filter by fulfillment status
- `date_from` (string, optional) - Filter orders from date (ISO 8601)
- `date_to` (string, optional) - Filter orders to date (ISO 8601)

**Example Request:**
```bash
curl -X GET "https://your-domain.replit.app/api/external/orders?page=1&limit=10&source=allegro" \
  -H "Authorization: Bearer test_token_78da7ce9ef884fe70bdae4e2343ca7ea"
```

**Example Response:**
```json
{
  "success": true,
  "data": [
    {
      "order_number": "1215",
      "order_code": "AL-1215",
      "source_order_id": "5ce2e000-a4df-11f0-961e-6d069f850ff8",
      "source": "ALLEGRO",
      "order_date": "2025-10-09T07:13:54.111Z",
      "buyer_first_name": "Paweł",
      "buyer_last_name": "Szczęch",
      "buyer_email": "example@allegromail.pl",
      "buyer_phone": "+48 668 623 764",
      "buyer_login": "pablosz5",
      "delivery_method": "Allegro Kurier DPD",
      "delivery_amount": "0.00",
      "delivery_address": {
        "city": "Szynwałd",
        "street": "Szynwałd 191A",
        "zipCode": "33-158",
        "countryCode": "PL"
      },
      "total_amount": "884.10",
      "currency": "PLN",
      "payment_status": "PENDING",
      "payment_type": "ONLINE",
      "payment_amount": "884.10",
      "has_returns": false,
      "refund_amount": "0",
      "created_at": "2025-10-09T08:02:38.940Z",
      "updated_at": "2025-10-09T08:12:12.161Z"
    }
  ],
  "pagination": {
    "page": 1,
    "limit": 10,
    "total": 1215,
    "totalPages": 122
  }
}
```

### 2. Get Order Details

Retrieve detailed information about a specific order.

**Endpoint:** `GET /api/external/orders/:orderCode`

**Path Parameters:**
- `orderCode` (string) - Order code (e.g., AL-1215, SH-456)

**Example Request:**
```bash
curl -X GET "https://your-domain.replit.app/api/external/orders/AL-1215" \
  -H "Authorization: Bearer test_token_78da7ce9ef884fe70bdae4e2343ca7ea"
```

**Example Response:**
```json
{
  "success": true,
  "data": {
    "order_number": "1215",
    "order_code": "AL-1215",
    "source_order_id": "5ce2e000-a4df-11f0-961e-6d069f850ff8",
    "source": "ALLEGRO",
    "order_date": "2025-10-09T07:13:54.111Z",
    "buyer_first_name": "Paweł",
    "buyer_last_name": "Szczęch",
    "buyer_email": "example@allegromail.pl",
    "buyer_phone": "+48 668 623 764",
    "delivery_method": "Allegro Kurier DPD",
    "delivery_address": {
      "city": "Szynwałd",
      "street": "Szynwałd 191A",
      "zipCode": "33-158",
      "countryCode": "PL"
    },
    "total_amount": "884.10",
    "currency": "PLN",
    "payment_status": "PENDING",
    "shipments": [],
    "has_returns": false,
    "refund_amount": "0"
  }
}
```

### 3. List Products

Get list of all unique products from the catalog.

**Endpoint:** `GET /api/external/products`

**Example Request:**
```bash
curl -X GET "https://your-domain.replit.app/api/external/products" \
  -H "Authorization: Bearer test_token_78da7ce9ef884fe70bdae4e2343ca7ea"
```

**Example Response:**
```json
{
  "success": true,
  "data": [
    {
      "id": 181,
      "product_id": "3580",
      "external_id": "5905806238587",
      "name": "Szafka na buty VB100x36D1N1 Dąb Estana",
      "description": null,
      "image_url": "5905806238587_1.jpg",
      "source": "SHOPER",
      "category": null,
      "created_at": "2025-10-06T18:33:10.090Z",
      "updated_at": "2025-10-09T08:01:35.784Z"
    }
  ],
  "total": 650
}
```

## Webhooks

Webhooks allow your system to receive real-time notifications when order events occur.

### Webhook Events

- `order.created` - New order created
- `order.updated` - Order information updated
- `order.paid` - Order payment confirmed
- `order.shipped` - Order shipped with tracking number

### Webhook Payload Format

All webhooks send POST requests with JSON payload:

```json
{
  "event": "order.created",
  "timestamp": "2025-10-09T08:30:00.000Z",
  "data": {
    "order_number": "1216",
    "order_code": "AL-1216",
    "source": "ALLEGRO",
    "buyer_email": "customer@example.com",
    "total_amount": "299.00",
    "currency": "PLN",
    "payment_status": "PAID"
  }
}
```

### Webhook Security

Each webhook request includes an `X-Webhook-Signature` header with HMAC-SHA256 signature:

```
X-Webhook-Signature: a1b2c3d4e5f6...
```

**Verifying Webhook Signature (Node.js example):**
```javascript
const crypto = require('crypto');

function verifyWebhookSignature(payload, signature, secret) {
  const expectedSignature = crypto
    .createHmac('sha256', secret)
    .update(JSON.stringify(payload))
    .digest('hex');
  
  return signature === expectedSignature;
}

// Express.js webhook handler
app.post('/webhook/alpma-orders', (req, res) => {
  const signature = req.headers['x-webhook-signature'];
  const isValid = verifyWebhookSignature(req.body, signature, YOUR_WEBHOOK_SECRET);
  
  if (!isValid) {
    return res.status(401).json({ error: 'Invalid signature' });
  }
  
  // Process webhook event
  console.log('Event:', req.body.event);
  console.log('Order:', req.body.data);
  
  res.json({ received: true });
});
```

### Retry Logic

If webhook delivery fails, the system automatically retries with exponential backoff:
- **Attempt 1**: Immediate
- **Attempt 2**: After 2 seconds
- **Attempt 3**: After 4 seconds
- **Attempt 4**: After 8 seconds

Configurable retry attempts (default: 3 retries).

## Error Handling

### HTTP Status Codes

- `200 OK` - Request successful
- `400 Bad Request` - Invalid request parameters
- `401 Unauthorized` - Missing or invalid API token
- `404 Not Found` - Resource not found
- `500 Internal Server Error` - Server error

### Error Response Format

```json
{
  "success": false,
  "error": "Error message description"
}
```

## Rate Limiting

**Current limits:**
- 1000 requests per hour per API token
- Webhook delivery timeout: 10 seconds

## Integration Examples

### Odoo Integration (Python)

```python
import requests
import hmac
import hashlib
from datetime import datetime, timedelta

# Configuration
API_URL = "https://your-domain.replit.app/api/external"
API_TOKEN = "your_api_token_here"
WEBHOOK_SECRET = "your_webhook_secret"

# Fetch orders
def fetch_orders(date_from=None, limit=50):
    headers = {
        "Authorization": f"Bearer {API_TOKEN}"
    }
    
    params = {
        "limit": limit,
        "source": "allegro"
    }
    
    if date_from:
        params["date_from"] = date_from.isoformat()
    
    response = requests.get(f"{API_URL}/orders", headers=headers, params=params)
    return response.json()

# Sync recent orders
def sync_recent_orders():
    yesterday = datetime.now() - timedelta(days=1)
    data = fetch_orders(date_from=yesterday, limit=100)
    
    for order in data.get("data", []):
        # Import to Odoo
        print(f"Importing order {order['order_code']}")
        # ... Odoo API calls here

# Verify webhook signature
def verify_webhook(payload, signature):
    expected_signature = hmac.new(
        WEBHOOK_SECRET.encode(),
        payload.encode(),
        hashlib.sha256
    ).hexdigest()
    
    return hmac.compare_digest(signature, expected_signature)
```

### Node.js Integration

```javascript
const axios = require('axios');

const API_URL = 'https://your-domain.replit.app/api/external';
const API_TOKEN = 'your_api_token_here';

// Fetch orders
async function fetchOrders(page = 1, limit = 50) {
  try {
    const response = await axios.get(`${API_URL}/orders`, {
      headers: {
        'Authorization': `Bearer ${API_TOKEN}`
      },
      params: { page, limit, source: 'allegro' }
    });
    
    return response.data;
  } catch (error) {
    console.error('API Error:', error.message);
    throw error;
  }
}

// Get order details
async function getOrderDetails(orderCode) {
  const response = await axios.get(`${API_URL}/orders/${orderCode}`, {
    headers: {
      'Authorization': `Bearer ${API_TOKEN}`
    }
  });
  
  return response.data;
}

// Usage
(async () => {
  const orders = await fetchOrders(1, 10);
  console.log(`Fetched ${orders.data.length} orders`);
  
  if (orders.data.length > 0) {
    const details = await getOrderDetails(orders.data[0].order_code);
    console.log('Order details:', details.data);
  }
})();
```

## Support

For API support or integration assistance, contact the system administrator.

## Changelog

### 2025-10-09
- Initial REST API release
- Webhook system implementation
- Bearer token authentication
- API request audit logging
