import logging

from odoo.tools import SQL, column_exists, table_exists
from odoo.upgrade import util

_logger = logging.getLogger(__name__)

VERSION = '17.0.10.0.0'
MODULE = 'trilab_allegro'


# noinspection SqlWithoutWhere
def migrate(cr, version):
    _logger.info(f'end migration {VERSION} start (from {version})')

    if table_exists(cr, 'trilab_allegro_account') and table_exists(cr, 'trilab_allegro_marketplace'):
        _logger.info('migrate allegro account')
        # no need for marketplace migrations, should be created and propagated from XML

        if column_exists(cr, 'trilab_market_account', 'last_order_date'):
            order_date_field = 'last_order_date'
        else:
            order_date_field = 'order_sync_date'

        cr.execute(
            SQL(
                """INSERT INTO trilab_market_account
         (
           name,
           market,
           allegro_environment, 
           allegro_access_token, 
           allegro_access_token_validity,
           allegro_refresh_token,
           allegro_is_company,
           user_id,
           team_id,
           company_id,
           marketplace_id,
           state,
           qty_field,
           active,
           auto_sync_offer,
           sync_payment,
           sync_state,
           sync_price,
           sync_stock,
           order_confirm,
           send_invoice,
           debug_logging,
           %s,
           create_missing_product,
            webhook_key
        )
         SELECT 
           aa.name,
           'allegro',
           aa.environment,
           aa.user_access_token,
           aa.user_access_token_validity,
           aa.user_refresh_token,
           aa.is_company,
           aa.user_id,
           aa.team_id,
           aa.company_id,
           (SELECT id FROM trilab_market_marketplace WHERE name = am.name),
           aa.state,
           aa.qty_field,
           aa.active,
           true,
           aa.sync_payment,
           aa.sync_state,
           aa.sync_price,
           aa.sync_stock,
           aa.order_confirm,
           aa.send_invoice,
           aa.debug_logging,
           aa.last_order_date,
           true,
           gen_random_uuid()
        FROM trilab_allegro_account aa
         LEFT OUTER JOIN trilab_allegro_marketplace am ON am.id = aa.marketplace_id
         """,
                SQL.identifier(order_date_field),
            )
        )

        if table_exists(cr, 'trilab_allegro_payment_term'):
            _logger.info('migrate allegro payment terms')
            cr.execute("""INSERT INTO trilab_market_payment_terms
            (
                account_id,
                payment_term_id,
                allegro_code
            )
            SELECT 
                (SELECT id FROM trilab_market_account WHERE market = 'allegro' and name = aa.name),
                apt.payment_term_id,
                apt.name
            FROM trilab_allegro_payment_term apt
               LEFT OUTER JOIN trilab_allegro_account aa ON aa.id = apt.account_id
            """)

            _logger.info('drop allegro payment terms table')
            util.remove_model(cr, 'trilab_allegro.payment.term')

        if table_exists(cr, 'trilab_allegro_account_settings'):
            _logger.info('migrate allegro account settings')
            cr.execute("""INSERT INTO trilab_market_settings
            (
              account_id, 
              marketplace_id, 
              pricelist_id, 
              fiscal_position_id
            )
            SELECT
                (SELECT id FROM trilab_market_account WHERE market = 'allegro' and name = aa.name),
                (SELECT id FROM trilab_market_marketplace WHERE name = am.name),
                aas.pricelist_id,
                aas.fiscal_position_id
            FROM trilab_allegro_account_settings aas
               LEFT OUTER JOIN trilab_allegro_account aa ON aa.id = aas.account_id
               LEFT OUTER JOIN trilab_allegro_marketplace am ON am.id = aas.marketplace_id
            """)

            _logger.info('remove allegro account settings table')
            util.remove_model(cr, 'trilab_allegro.account.settings')

        if column_exists(cr, 'sale_order', 'x_allegro_account_id'):
            _logger.info('migrate x_allegro_account_id to x_market_account_id on sale.order')
            cr.execute("""UPDATE sale_order
                SET x_market_account_id = src.account_id,
                    x_market_settings_id = src.settings_id
                    FROM (SELECT tms.id settings_id, tms.account_id, taa.id a_id FROM trilab_market_settings tms
LEFT JOIN trilab_market_marketplace tmm ON tmm.id = tms.marketplace_id
LEFT JOIN trilab_market_account tma ON tma.id = tms.account_id
LEFT JOIN trilab_allegro_account taa ON taa.name = tma.name
LEFT JOIN trilab_allegro_marketplace tam ON tam.id = taa.marketplace_id
WHERE taa.id = 1 and tmm.name = tam.name) AS src
                WHERE src.a_id = sale_order.x_allegro_account_id""")

            _logger.info('remove sale.order.x_allegro_account_id field')
            util.remove_field(cr, 'sale.order', 'x_allegro_account_id')

        if column_exists(cr, 'stock_picking', 'x_allegro_account_id'):
            _logger.info('migrate x_allegro_account_id to x_market_account_id on stock.picking')
            cr.execute("""UPDATE stock_picking
                          SET x_market_account_id = src.account_id,
                              x_market_settings_id = src.settings_id
                          FROM (SELECT tms.id settings_id, tms.account_id, taa.id a_id FROM trilab_market_settings tms
                                                                                                LEFT JOIN trilab_market_marketplace tmm ON tmm.id = tms.marketplace_id
                                                                                                LEFT JOIN trilab_market_account tma ON tma.id = tms.account_id
                                                                                                LEFT JOIN trilab_allegro_account taa ON taa.name = tma.name
                                                                                                LEFT JOIN trilab_allegro_marketplace tam ON tam.id = taa.marketplace_id
                                WHERE taa.id = 1 and tmm.name = tam.name) AS src
                          WHERE src.a_id = stock_picking.x_allegro_account_id""")

            _logger.info('remove stock.picking.x_allegro_account_id field')
            util.remove_field(cr, 'stock.picking', 'x_allegro_account_id')

        _logger.info('remove allegro marketplace table')
        util.remove_model(cr, 'trilab_allegro.marketplace')

        _logger.info('remove allegro account table')
        util.remove_model(cr, 'trilab_allegro.account')

    if table_exists(cr, 'trilab_allegro_account_pair'):
        _logger.info('remove allegro account pair table')
        util.remove_model(cr, 'trilab_allegro.account.pair')

    _logger.info(f'end migration {VERSION} finished')
