import { sql } from "drizzle-orm";
import { pgTable, text, varchar, timestamp, decimal, jsonb, boolean, serial, index, integer } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

export const allegroConnections = pgTable("allegro_connections", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  clientId: text("client_id").notNull(),
  clientSecret: text("client_secret").notNull(),
  accessToken: text("access_token"),
  refreshToken: text("refresh_token"),
  tokenExpiresAt: timestamp("token_expires_at"),
  isActive: boolean("is_active").default(false),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const allegroOrders = pgTable("allegro_orders", {
  id: varchar("id").primaryKey(),
  allegroOrderId: text("allegro_order_id").notNull().unique(),
  buyerLogin: text("buyer_login").notNull(),
  buyerEmail: text("buyer_email"),
  totalAmount: decimal("total_amount", { precision: 10, scale: 2 }).notNull(),
  currency: text("currency").default("PLN"),
  paymentStatus: text("payment_status").notNull(),
  fulfillmentStatus: text("fulfillment_status"),
  itemsCount: text("items_count").notNull(),
  orderDate: timestamp("order_date").notNull(),
  paymentDate: timestamp("payment_date"),
  rawData: jsonb("raw_data"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const syncSettings = pgTable("sync_settings", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  autoRefreshEnabled: boolean("auto_refresh_enabled").default(true),
  refreshIntervalMinutes: text("refresh_interval_minutes").default("3"),
  lastSyncAt: timestamp("last_sync_at"),
  shoperAutoRefreshEnabled: boolean("shoper_auto_refresh_enabled").default(true),
  shoperRefreshIntervalMinutes: text("shoper_refresh_interval_minutes").default("5"),
  lastShoperSyncAt: timestamp("last_shoper_sync_at"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const orderChanges = pgTable("order_changes", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  orderId: varchar("order_id").notNull(),
  orderNumber: text("order_number"),
  orderCode: text("order_code"),
  source: text("source").notNull(),
  changeType: text("change_type").notNull(),
  fieldChanged: text("field_changed"),
  oldValue: text("old_value"),
  newValue: text("new_value"),
  detectedAt: timestamp("detected_at").defaultNow(),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertAllegroConnectionSchema = createInsertSchema(allegroConnections).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertAllegroOrderSchema = createInsertSchema(allegroOrders).omit({
  createdAt: true,
  updatedAt: true,
});

export const insertSyncSettingsSchema = createInsertSchema(syncSettings).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertOrderChangeSchema = createInsertSchema(orderChanges).omit({
  id: true,
  createdAt: true,
});

export type AllegroConnection = typeof allegroConnections.$inferSelect;
export type InsertAllegroConnection = z.infer<typeof insertAllegroConnectionSchema>;

export type AllegroOrder = typeof allegroOrders.$inferSelect & {
  orderCode?: string;
  orderNumber?: number;
  source?: string;
};
export type InsertAllegroOrder = z.infer<typeof insertAllegroOrderSchema>;

export type SyncSettings = typeof syncSettings.$inferSelect;
export type InsertSyncSettings = z.infer<typeof insertSyncSettingsSchema>;

export type OrderChange = typeof orderChanges.$inferSelect;
export type InsertOrderChange = z.infer<typeof insertOrderChangeSchema>;

export const orderStatusSchema = z.enum([
  "PAID",
  "UNPAID",
  "PENDING",
  "CANCELLED",
]);

export const fulfillmentStatusSchema = z.enum([
  "NEW",
  "PROCESSING",
  "READY_FOR_SHIPMENT",
  "SENT",
  "DELIVERED",
  "CANCELLED",
]);

export type OrderStatus = z.infer<typeof orderStatusSchema>;
export type FulfillmentStatus = z.infer<typeof fulfillmentStatusSchema>;

// Auth tables
export const sessions = pgTable(
  "sessions",
  {
    sid: varchar("sid").primaryKey(),
    sess: jsonb("sess").notNull(),
    expire: timestamp("expire").notNull(),
  },
  (table) => [index("IDX_session_expire").on(table.expire)],
);

export const users = pgTable("users", {
  id: serial("id").primaryKey(),
  username: varchar("username", { length: 255 }).notNull().unique(),
  email: varchar("email", { length: 255 }).notNull().unique(),
  password: text("password").notNull(),
  firstName: varchar("first_name", { length: 255 }),
  lastName: varchar("last_name", { length: 255 }),
  role: varchar("role", { length: 50 }).notNull().default("user"),
  permissions: jsonb("permissions").default(sql`'[]'::jsonb`),
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const passwordResetTokens = pgTable("password_reset_tokens", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id),
  token: varchar("token", { length: 255 }).notNull().unique(),
  expiresAt: timestamp("expires_at").notNull(),
  used: boolean("used").default(false),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertUserSchema = createInsertSchema(users, {
  username: z.string().min(3, "Username musi mieć minimum 3 znaki").max(255),
  email: z.string().email("Nieprawidłowy adres email"),
  password: z.string().min(6, "Hasło musi mieć minimum 6 znaków"),
}).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertPasswordResetTokenSchema = createInsertSchema(passwordResetTokens).omit({
  id: true,
  createdAt: true,
});

export const userRoleSchema = z.enum(["admin", "manager", "user"]);
export type UserRole = z.infer<typeof userRoleSchema>;

export const permissionSchema = z.enum([
  "orders.view",
  "orders.edit",
  "orders.delete",
  "settings.view",
  "settings.edit",
  "users.view",
  "users.edit",
  "sync.run",
]);
export type Permission = z.infer<typeof permissionSchema>;

export type User = typeof users.$inferSelect & {
  role: UserRole;
  permissions: Permission[];
};
export type InsertUser = z.infer<typeof insertUserSchema>;
export type PasswordResetToken = typeof passwordResetTokens.$inferSelect;
export type InsertPasswordResetToken = z.infer<typeof insertPasswordResetTokenSchema>;

// API Tokens for external integrations (e.g., Odoo)
export const apiTokens = pgTable("api_tokens", {
  id: serial("id").primaryKey(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  token: text("token").notNull().unique(),
  tokenPrefix: varchar("token_prefix", { length: 16 }),
  permissions: jsonb("permissions").default(sql`'[]'::jsonb`),
  isActive: boolean("is_active").default(true),
  lastUsedAt: timestamp("last_used_at"),
  expiresAt: timestamp("expires_at"),
  createdByUserId: integer("created_by_user_id").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Webhook configurations for pushing updates to external systems
export const webhookConfigs = pgTable("webhook_configs", {
  id: serial("id").primaryKey(),
  name: varchar("name", { length: 255 }).notNull(),
  url: text("url").notNull(),
  secret: text("secret"),
  events: jsonb("events").default(sql`'[]'::jsonb`),
  isActive: boolean("is_active").default(true),
  retryAttempts: integer("retry_attempts").default(3),
  lastTriggeredAt: timestamp("last_triggered_at"),
  createdBy: integer("created_by").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// API request logs for audit trail
export const apiRequestLogs = pgTable("api_request_logs", {
  id: serial("id").primaryKey(),
  tokenId: integer("token_id").references(() => apiTokens.id),
  method: varchar("method", { length: 10 }).notNull(),
  path: text("path").notNull(),
  statusCode: integer("status_code"),
  responseTime: integer("response_time"),
  ipAddress: varchar("ip_address", { length: 45 }),
  userAgent: text("user_agent"),
  createdAt: timestamp("created_at").defaultNow(),
});

// Webhook delivery logs
export const webhookLogs = pgTable("webhook_logs", {
  id: serial("id").primaryKey(),
  webhookId: integer("webhook_id").references(() => webhookConfigs.id),
  event: varchar("event", { length: 100 }).notNull(),
  payload: jsonb("payload"),
  statusCode: integer("status_code"),
  responseBody: text("response_body"),
  attemptNumber: integer("attempt_number").default(1),
  success: boolean("success").default(false),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertApiTokenSchema = createInsertSchema(apiTokens).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertWebhookConfigSchema = createInsertSchema(webhookConfigs).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type ApiToken = typeof apiTokens.$inferSelect;
export type InsertApiToken = z.infer<typeof insertApiTokenSchema>;
export type WebhookConfig = typeof webhookConfigs.$inferSelect;
export type InsertWebhookConfig = z.infer<typeof insertWebhookConfigSchema>;
export type ApiRequestLog = typeof apiRequestLogs.$inferSelect;
export type WebhookLog = typeof webhookLogs.$inferSelect;

// Odoo Integration tables
export const odooConfig = pgTable("odoo_config", {
  id: serial("id").primaryKey(),
  url: text("url").notNull(),
  database: varchar("database", { length: 255 }).notNull(),
  username: varchar("username", { length: 255 }).notNull(),
  isActive: boolean("is_active").default(true),
  autoConfirmOrders: boolean("auto_confirm_orders").default(false),
  syncIntervalMinutes: integer("sync_interval_minutes").default(3),
  lastSyncAt: timestamp("last_sync_at"),
  lastSyncStatus: varchar("last_sync_status", { length: 50 }),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const odooSyncQueue = pgTable("odoo_sync_queue", {
  id: serial("id").primaryKey(),
  orderCode: varchar("order_code", { length: 50 }).notNull(),
  orderId: integer("order_id").notNull(),
  source: varchar("source", { length: 20 }).notNull(),
  operation: varchar("operation", { length: 20 }).notNull().default("create"),
  status: varchar("status", { length: 50 }).notNull().default("pending"),
  attempts: integer("attempts").default(0),
  maxAttempts: integer("max_attempts").default(3),
  lastError: text("last_error"),
  odooOrderId: integer("odoo_order_id"),
  scheduledAt: timestamp("scheduled_at").defaultNow(),
  processedAt: timestamp("processed_at"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const odooSyncLogs = pgTable("odoo_sync_logs", {
  id: serial("id").primaryKey(),
  queueId: integer("queue_id").references(() => odooSyncQueue.id),
  orderCode: varchar("order_code", { length: 50 }),
  operation: varchar("operation", { length: 20 }).notNull(),
  status: varchar("status", { length: 50 }).notNull(),
  message: text("message"),
  errorDetails: jsonb("error_details"),
  requestPayload: jsonb("request_payload"),
  responseData: jsonb("response_data"),
  duration: integer("duration"),
  odooOrderId: integer("odoo_order_id"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertOdooConfigSchema = createInsertSchema(odooConfig).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertOdooSyncQueueSchema = createInsertSchema(odooSyncQueue).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertOdooSyncLogSchema = createInsertSchema(odooSyncLogs).omit({
  id: true,
  createdAt: true,
});

export type OdooConfig = typeof odooConfig.$inferSelect;
export type InsertOdooConfig = z.infer<typeof insertOdooConfigSchema>;
export type OdooSyncQueue = typeof odooSyncQueue.$inferSelect;
export type InsertOdooSyncQueue = z.infer<typeof insertOdooSyncQueueSchema>;
export type OdooSyncLog = typeof odooSyncLogs.$inferSelect;
export type InsertOdooSyncLog = z.infer<typeof insertOdooSyncLogSchema>;
