import {
  type AllegroConnection,
  type InsertAllegroConnection,
  type AllegroOrder,
  type InsertAllegroOrder,
  type SyncSettings,
  type InsertSyncSettings,
} from "@shared/schema";
import { randomUUID } from "crypto";

export interface IStorage {
  getAllegroConnection(): Promise<AllegroConnection | undefined>;
  createOrUpdateConnection(
    connection: InsertAllegroConnection
  ): Promise<AllegroConnection>;
  updateConnectionTokens(
    id: string,
    accessToken: string,
    refreshToken: string,
    expiresAt: Date
  ): Promise<void>;
  
  getOrders(): Promise<AllegroOrder[]>;
  getOrderByAllegroId(allegroOrderId: string): Promise<AllegroOrder | undefined>;
  createOrUpdateOrder(order: InsertAllegroOrder): Promise<AllegroOrder>;
  
  getSyncSettings(): Promise<SyncSettings | undefined>;
  createOrUpdateSyncSettings(
    settings: InsertSyncSettings
  ): Promise<SyncSettings>;
}

export class MemStorage implements IStorage {
  private connection: AllegroConnection | undefined;
  private orders: Map<string, AllegroOrder>;
  private syncSettings: SyncSettings | undefined;

  constructor() {
    this.orders = new Map();
  }

  async getAllegroConnection(): Promise<AllegroConnection | undefined> {
    return this.connection;
  }

  async createOrUpdateConnection(
    insertConnection: InsertAllegroConnection
  ): Promise<AllegroConnection> {
    const id = this.connection?.id || randomUUID();
    const connection: AllegroConnection = {
      id,
      clientId: insertConnection.clientId,
      clientSecret: insertConnection.clientSecret,
      accessToken: insertConnection.accessToken ?? null,
      refreshToken: insertConnection.refreshToken ?? null,
      tokenExpiresAt: insertConnection.tokenExpiresAt ?? null,
      isActive: insertConnection.isActive ?? null,
      createdAt: this.connection?.createdAt || new Date(),
      updatedAt: new Date(),
    };
    this.connection = connection;
    return connection;
  }

  async updateConnectionTokens(
    id: string,
    accessToken: string,
    refreshToken: string,
    expiresAt: Date
  ): Promise<void> {
    if (this.connection && this.connection.id === id) {
      this.connection = {
        ...this.connection,
        accessToken,
        refreshToken,
        tokenExpiresAt: expiresAt,
        isActive: true,
        updatedAt: new Date(),
      };
    }
  }

  async getOrders(): Promise<AllegroOrder[]> {
    return Array.from(this.orders.values()).sort(
      (a, b) =>
        new Date(b.orderDate).getTime() - new Date(a.orderDate).getTime()
    );
  }

  async getOrderByAllegroId(
    allegroOrderId: string
  ): Promise<AllegroOrder | undefined> {
    return Array.from(this.orders.values()).find(
      (order) => order.allegroOrderId === allegroOrderId
    );
  }

  async createOrUpdateOrder(
    insertOrder: InsertAllegroOrder
  ): Promise<AllegroOrder> {
    const existing = await this.getOrderByAllegroId(
      insertOrder.allegroOrderId
    );
    const id = existing?.id || insertOrder.id || randomUUID();
    
    const order: AllegroOrder = {
      id,
      allegroOrderId: insertOrder.allegroOrderId,
      buyerLogin: insertOrder.buyerLogin,
      buyerEmail: insertOrder.buyerEmail ?? null,
      totalAmount: insertOrder.totalAmount,
      currency: insertOrder.currency ?? null,
      paymentStatus: insertOrder.paymentStatus,
      fulfillmentStatus: insertOrder.fulfillmentStatus ?? null,
      itemsCount: insertOrder.itemsCount,
      orderDate: insertOrder.orderDate,
      paymentDate: insertOrder.paymentDate ?? null,
      rawData: insertOrder.rawData ?? null,
      createdAt: existing?.createdAt || new Date(),
      updatedAt: new Date(),
    };
    
    this.orders.set(id, order);
    return order;
  }

  async getSyncSettings(): Promise<SyncSettings | undefined> {
    return this.syncSettings;
  }

  async createOrUpdateSyncSettings(
    insertSettings: InsertSyncSettings
  ): Promise<SyncSettings> {
    const id = this.syncSettings?.id || randomUUID();
    const settings: SyncSettings = {
      id,
      autoRefreshEnabled: insertSettings.autoRefreshEnabled ?? null,
      refreshIntervalMinutes: insertSettings.refreshIntervalMinutes ?? null,
      lastSyncAt: insertSettings.lastSyncAt ?? null,
      shoperAutoRefreshEnabled: insertSettings.shoperAutoRefreshEnabled ?? null,
      shoperRefreshIntervalMinutes: insertSettings.shoperRefreshIntervalMinutes ?? null,
      lastShoperSyncAt: insertSettings.lastShoperSyncAt ?? null,
      createdAt: this.syncSettings?.createdAt || new Date(),
      updatedAt: new Date(),
    };
    this.syncSettings = settings;
    return settings;
  }
}

export const storage = new MemStorage();
