import axios from 'axios';
import fs from 'fs/promises';
import path from 'path';

let cachedToken: {
  token: string;
  expiresAt: Date;
} | null = null;

export async function getShoperToken(): Promise<string> {
  const shopUrl = process.env.SHOPER_SHOP_URL;
  const username = process.env.SHOPER_USERNAME;
  const password = process.env.SHOPER_PASSWORD;

  if (!shopUrl || !username || !password) {
    throw new Error('Shoper credentials not configured');
  }

  if (cachedToken && cachedToken.expiresAt > new Date()) {
    return cachedToken.token;
  }

  try {
    const credentials = Buffer.from(`${username}:${password}`).toString('base64');
    
    const response = await axios.post(
      `${shopUrl}/webapi/rest/auth`,
      {},
      {
        headers: {
          'Authorization': `Basic ${credentials}`,
          'Content-Type': 'application/json',
        },
      }
    );

    const { access_token, expires_in } = response.data;
    const expiresAt = new Date(Date.now() + (expires_in - 60) * 1000);

    cachedToken = {
      token: access_token,
      expiresAt,
    };

    console.log('✅ Shoper token obtained, expires at:', expiresAt);
    return access_token;
  } catch (error: any) {
    console.error('❌ Failed to get Shoper token:', error.response?.data || error.message);
    throw new Error('Failed to authenticate with Shoper API');
  }
}

export async function getShoperOrders(limit: number = 50, filters?: Record<string, any>): Promise<any[]> {
  const shopUrl = process.env.SHOPER_SHOP_URL;
  if (!shopUrl) {
    throw new Error('SHOPER_SHOP_URL not configured');
  }

  const token = await getShoperToken();

  try {
    const params: Record<string, any> = {
      limit,
      order: 'order_id DESC',
    };

    if (filters) {
      params.filters = JSON.stringify(filters);
    }

    const response = await axios.get(
      `${shopUrl}/webapi/rest/orders`,
      {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json',
        },
        params
      }
    );

    return response.data.list || [];
  } catch (error: any) {
    console.error('❌ Failed to get Shoper orders:', error.response?.data || error.message);
    throw new Error(`Failed to fetch Shoper orders: ${error.message}`);
  }
}

export async function getShoperOrder(orderId: string): Promise<any> {
  const shopUrl = process.env.SHOPER_SHOP_URL;
  if (!shopUrl) {
    throw new Error('SHOPER_SHOP_URL not configured');
  }

  const token = await getShoperToken();

  try {
    const response = await axios.get(
      `${shopUrl}/webapi/rest/orders/${orderId}`,
      {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json',
        },
      }
    );

    return response.data;
  } catch (error: any) {
    console.error(`❌ Failed to get Shoper order ${orderId}:`, error.response?.data || error.message);
    throw new Error(`Failed to fetch Shoper order: ${error.message}`);
  }
}

export async function getShoperProducts(limit: number = 100): Promise<any[]> {
  const shopUrl = process.env.SHOPER_SHOP_URL;
  if (!shopUrl) {
    throw new Error('SHOPER_SHOP_URL not configured');
  }

  const token = await getShoperToken();

  try {
    const response = await axios.get(
      `${shopUrl}/webapi/rest/products`,
      {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json',
        },
        params: {
          limit,
        }
      }
    );

    return response.data.list || [];
  } catch (error: any) {
    console.error('❌ Failed to get Shoper products:', error.response?.data || error.message);
    throw new Error(`Failed to fetch Shoper products: ${error.message}`);
  }
}

export async function getShoperOrderProducts(orderId: string | number): Promise<any[]> {
  const shopUrl = process.env.SHOPER_SHOP_URL;
  if (!shopUrl) {
    throw new Error('SHOPER_SHOP_URL not configured');
  }

  const token = await getShoperToken();

  try {
    const filters = { 'order_id': orderId };
    console.log(`📦 Fetching order-products for order ${orderId} with filters:`, JSON.stringify(filters));
    
    const response = await axios.get(
      `${shopUrl}/webapi/rest/order-products`,
      {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json',
        },
        params: {
          filters: JSON.stringify(filters),
          limit: 50,
        }
      }
    );

    const products = response.data.list || [];
    console.log(`✅ Received ${products.length} products for order ${orderId}`);
    return products;
  } catch (error: any) {
    console.error(`❌ Failed to get Shoper order products for order ${orderId}:`, error.response?.data || error.message);
    if (error.response) {
      console.error('Response status:', error.response.status);
      console.error('Response data:', JSON.stringify(error.response.data, null, 2));
    }
    throw new Error(`Failed to fetch order products for order ${orderId}: ${error.response?.status || error.message}`);
  }
}

export async function getShoperParcels(orderId: string | number): Promise<any[]> {
  const shopUrl = process.env.SHOPER_SHOP_URL;
  if (!shopUrl) {
    throw new Error('SHOPER_SHOP_URL not configured');
  }

  const token = await getShoperToken();

  try {
    const filters = { 'order_id': orderId };
    console.log(`📦 Fetching parcels for order ${orderId}`);
    
    const response = await axios.get(
      `${shopUrl}/webapi/rest/parcels`,
      {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json',
        },
        params: {
          filters: JSON.stringify(filters),
          limit: 50,
        }
      }
    );

    const parcels = response.data.list || [];
    console.log(`✅ Received ${parcels.length} parcels for order ${orderId}`);
    return parcels;
  } catch (error: any) {
    // Jeśli nie ma przesyłek lub błąd 404, zwróć pustą tablicę
    if (error.response?.status === 404) {
      console.log(`ℹ️ No parcels found for order ${orderId}`);
      return [];
    }
    console.error(`❌ Failed to get Shoper parcels for order ${orderId}:`, error.response?.data || error.message);
    return [];
  }
}

export async function getShoperPayments(): Promise<any[]> {
  const shopUrl = process.env.SHOPER_SHOP_URL;
  if (!shopUrl) {
    throw new Error('SHOPER_SHOP_URL not configured');
  }

  const token = await getShoperToken();

  try {
    let allPayments: any[] = [];
    let page = 1;
    let hasMore = true;
    const limit = 50;

    while (hasMore) {
      const response = await axios.get(
        `${shopUrl}/webapi/rest/payments`,
        {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json',
          },
          params: {
            limit,
            page,
          }
        }
      );

      const payments = response.data.list || [];
      allPayments = allPayments.concat(payments);
      
      const count = response.data.count || 0;
      hasMore = allPayments.length < count;
      page++;
      
      if (page > 10) {
        console.warn('⚠️ Too many pages for payment methods, stopping at page 10');
        break;
      }
    }

    console.log(`📥 Shoper API returned ${allPayments.length} payment methods total`);
    return allPayments;
  } catch (error: any) {
    console.error('❌ Failed to get Shoper payments:', error.response?.data || error.message);
    return [];
  }
}

export async function getShoperDeliveries(): Promise<any[]> {
  const shopUrl = process.env.SHOPER_SHOP_URL;
  if (!shopUrl) {
    throw new Error('SHOPER_SHOP_URL not configured');
  }

  const token = await getShoperToken();

  try {
    const response = await axios.get(
      `${shopUrl}/webapi/rest/deliveries`,
      {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json',
        },
        params: {
          limit: 100,
        }
      }
    );

    return response.data.list || [];
  } catch (error: any) {
    console.error('❌ Failed to get Shoper deliveries:', error.response?.data || error.message);
    return [];
  }
}

export async function downloadShoperProductImage(
  imageUrl: string,
  productId: string
): Promise<string | null> {
  try {
    if (!imageUrl) {
      console.log(`⚠️ No image URL provided for product ${productId}`);
      return null;
    }

    const imageResponse = await axios.get(imageUrl, {
      responseType: 'arraybuffer',
    });

    const productsDir = path.join(process.cwd(), 'attached_assets', 'products');
    await fs.mkdir(productsDir, { recursive: true });

    const filename = `shoper_${productId}_1.jpg`;
    const filepath = path.join(productsDir, filename);

    await fs.writeFile(filepath, imageResponse.data);
    console.log(`✅ Shoper image saved: ${filename}`);

    return filename;
  } catch (error: any) {
    console.error(`❌ Failed to download Shoper image for product ${productId}:`, error.message);
    return null;
  }
}
