import type { Express } from "express";
import { createServer, type Server } from "http";
import { storage } from "./storage";
import axios from "axios";
import { randomBytes } from "crypto";
import path from "path";
import fs from "fs/promises";
import { pool, saveOrderToPostgres, testConnection, getOrdersFromPostgres, getOrderFromPostgres, getOrderByNumberFromPostgres, createSyncLog, completeSyncLog, getSyncLogs, saveOrderToCommerce, saveShoperOrderToPostgres, getRecentlyUpdatedOrders, getRecentOrderChanges, logOrderChange, getSyncSettingsFromPostgres, createOrUpdateSyncSettingsInPostgres, getOrderStatistics, getOrdersChartData, getOrdersPeriodSummary, getTodayDetailedStats, getAllegroConnectionFromPostgres, saveAllegroConnectionToPostgres, updateAllegroConnectionTokens, renumberOrdersByDate, getUserByUsername, getUserByEmail, createUser, createPasswordResetToken, getPasswordResetToken, markTokenAsUsed, updateUserPassword, getAllUsers, updateUserRole, updateUserPermissions, updateUserStatus, deleteUser } from "./postgres";
import { setupAuth, isAuthenticated, hashPassword, comparePasswords } from "./auth";
import { hasPermission, type Permission } from "../shared/permissions";
import passport from "passport";
import { requireApiToken, logApiRequest, generateApiToken, hashToken } from "./api-auth";
import { getWebhookLogs, triggerOrderWebhooks } from "./webhooks";
import swaggerUi from "swagger-ui-express";
import { swaggerSpec } from "./swagger";

const ALLEGRO_AUTH_URL = "https://allegro.pl/auth/oauth/authorize";
const ALLEGRO_TOKEN_URL = "https://allegro.pl/auth/oauth/token";
const ALLEGRO_API_URL = "https://api.allegro.pl";

interface AllegroTokenResponse {
  access_token: string;
  refresh_token: string;
  expires_in: number;
  token_type: string;
}

interface AllegroOrderItem {
  id: string;
  buyer: {
    login: string;
    email?: string;
  };
  payment: {
    paidAmount?: {
      amount: string;
      currency: string;
    };
    status?: string;
  };
  fulfillment?: {
    status?: string;
  };
  lineItems: Array<{ id: string }>;
  updatedAt: string;
  summary?: {
    totalToPay?: {
      amount: string;
      currency: string;
    };
  };
}

let syncInterval: NodeJS.Timeout | null = null;
let isCurrentlySyncing = false;
let shoperSyncInterval: NodeJS.Timeout | null = null;
let recentUpdatesInterval: NodeJS.Timeout | null = null;
let odooSyncInterval: NodeJS.Timeout | null = null;
const oauthStates = new Map<string, { timestamp: number }>();

// Helper function to trigger webhooks after order save
async function triggerWebhooksForOrder(
  source: 'ALLEGRO' | 'SHOPER',
  sourceOrderId: string,
  isNew: boolean
): Promise<void> {
  try {
    // Fetch full order data from database
    const orderResult = await pool.query(`
      SELECT 
        o.*,
        COALESCE(
          json_agg(
            json_build_object(
              'id', oi.id,
              'name', oi.name,
              'offer_external_id', oi.offer_external_id,
              'quantity', oi.quantity,
              'unit_price', oi.unit_price,
              'price', oi.price,
              'image_url', oi.image_url,
              'returns_quantity', oi.returns_quantity
            ) ORDER BY oi.id
          ) FILTER (WHERE oi.id IS NOT NULL),
          '[]'::json
        ) as items
      FROM commerce.orders o
      LEFT JOIN commerce.order_items oi ON o.id = oi.order_id
      WHERE o.source_order_id = $1 AND o.source = $2
      GROUP BY o.id
    `, [sourceOrderId, source]);

    if (orderResult.rows.length > 0) {
      const fullOrderData = orderResult.rows[0];
      
      // Determine event type based on whether it's new
      const eventType = isNew ? 'order.created' : 'order.updated';
      
      // Add to Odoo sync queue
      if (fullOrderData.order_code) {
        try {
          const { addOrderToSyncQueue } = await import('./odoo-sync.js');
          await addOrderToSyncQueue(
            fullOrderData.order_code,
            fullOrderData.order_number,
            fullOrderData.source,
            isNew ? 'create' : 'update'
          );
        } catch (odooError) {
          console.error('❌ Failed to add', fullOrderData.order_code, 'to sync queue:', odooError);
        }
      } else {
        console.warn(`⚠️ Skipping Odoo sync for order without order_code (source: ${source}, id: ${sourceOrderId})`);
      }
      
      // Check if payment status is PAID (trigger order.paid)
      if (!isNew && fullOrderData.payment_status === 'PAID') {
        await triggerOrderWebhooks('order.paid', fullOrderData);
      }
      
      // Check if order has tracking numbers (trigger order.shipped)
      const trackingNumbers = fullOrderData.tracking_numbers;
      if (!isNew && trackingNumbers && (
        (typeof trackingNumbers === 'string' && trackingNumbers.trim() !== '') ||
        (Array.isArray(trackingNumbers) && trackingNumbers.length > 0)
      )) {
        await triggerOrderWebhooks('order.shipped', fullOrderData);
      }
      
      // Always trigger main event (created or updated)
      await triggerOrderWebhooks(eventType, fullOrderData);
    }
  } catch (webhookError) {
    console.error('⚠️ Error triggering webhooks:', webhookError);
    // Don't fail if webhooks fail
  }
}

async function refreshAccessToken(
  clientId: string,
  clientSecret: string,
  refreshToken: string
): Promise<AllegroTokenResponse> {
  const credentials = Buffer.from(`${clientId}:${clientSecret}`).toString(
    "base64"
  );

  const response = await axios.post<AllegroTokenResponse>(
    ALLEGRO_TOKEN_URL,
    new URLSearchParams({
      grant_type: "refresh_token",
      refresh_token: refreshToken,
    }),
    {
      headers: {
        Authorization: `Basic ${credentials}`,
        "Content-Type": "application/x-www-form-urlencoded",
      },
    }
  );

  return response.data;
}

async function fetchOrdersFromAllegro(accessToken: string) {
  try {
    const allOrders: AllegroOrderItem[] = [];
    let offset = 0;
    const limit = 100;
    let hasMore = true;

    console.log(`🔄 Starting paginated fetch from Allegro (limit: ${limit} per page)...`);

    while (hasMore) {
      const response = await axios.get<{ checkoutForms: AllegroOrderItem[] }>(
        `${ALLEGRO_API_URL}/order/checkout-forms`,
        {
          headers: {
            Authorization: `Bearer ${accessToken}`,
            Accept: "application/vnd.allegro.public.v1+json",
          },
          params: {
            limit,
            offset,
          },
        }
      );

      const orders = response.data.checkoutForms || [];
      allOrders.push(...orders);

      console.log(`📦 Fetched ${orders.length} orders (offset: ${offset}, total so far: ${allOrders.length})`);

      if (orders.length < limit) {
        hasMore = false;
        console.log(`✅ Finished fetching - got ${allOrders.length} total orders from Allegro`);
      } else {
        offset += limit;
      }
    }

    return allOrders;
  } catch (error) {
    console.error("Error fetching orders from Allegro:", error);
    throw error;
  }
}

async function fetchRawOrderFromAllegro(accessToken: string, orderId: string) {
  try {
    const response = await axios.get(
      `${ALLEGRO_API_URL}/order/checkout-forms/${orderId}`,
      {
        headers: {
          Authorization: `Bearer ${accessToken}`,
          Accept: "application/vnd.allegro.public.v1+json",
        },
      }
    );

    return response.data;
  } catch (error) {
    console.error("Error fetching raw order from Allegro:", error);
    throw error;
  }
}

async function fetchShipmentsFromAllegro(accessToken: string, orderId: string) {
  try {
    const response = await axios.get(
      `${ALLEGRO_API_URL}/order/checkout-forms/${orderId}/shipments`,
      {
        headers: {
          Authorization: `Bearer ${accessToken}`,
          Accept: "application/vnd.allegro.public.v1+json",
        },
      }
    );

    return response.data.shipments || [];
  } catch (error) {
    // Jeśli nie ma przesyłek lub błąd 404, zwróć pustą tablicę
    if (axios.isAxiosError(error) && error.response?.status === 404) {
      return [];
    }
    console.error(`Error fetching shipments for order ${orderId}:`, error);
    return [];
  }
}

async function fetchCustomerReturnsFromAllegro(accessToken: string, orderId: string) {
  try {
    console.log(`🔍 DEBUG: Fetching customer returns for order ${orderId}`);
    const response = await axios.get(
      `${ALLEGRO_API_URL}/order/customer-returns`,
      {
        headers: {
          Authorization: `Bearer ${accessToken}`,
          Accept: "application/vnd.allegro.beta.v1+json",
        },
        params: {
          'checkout-form.id': orderId,
        },
      }
    );

    const customerReturns = response.data.customerReturns || [];
    console.log(`📦 DEBUG: Found ${customerReturns.length} customer returns for order ${orderId}`);
    if (customerReturns.length > 0) {
      console.log(`📦 DEBUG: Returns data:`, JSON.stringify(customerReturns, null, 2));
    }
    
    return customerReturns;
  } catch (error) {
    // Jeśli nie ma zwrotów lub błąd 404, zwróć pustą tablicę
    if (axios.isAxiosError(error) && error.response?.status === 404) {
      console.log(`⚠️ DEBUG: No customer returns found (404) for order ${orderId}`);
      return [];
    }
    console.error(`Error fetching customer returns for order ${orderId}:`, error);
    return [];
  }
}

async function syncOrders() {
  if (isCurrentlySyncing) {
    console.log("Sync already in progress, skipping...");
    return;
  }

  isCurrentlySyncing = true;
  const syncStartedAt = new Date();
  let logId: number | null = null;
  let ordersNew = 0;
  let ordersUpdated = 0;
  let ordersFetched = 0;
  
  try {
    logId = await createSyncLog(syncStartedAt);
    
    const connection = await storage.getAllegroConnection();
    if (!connection || !connection.accessToken || !connection.isActive) {
      console.log("No active connection, skipping sync");
      if (logId) {
        await completeSyncLog(logId, 'ERROR', 0, 0, 0, 'No active connection');
      }
      return;
    }

    let accessToken = connection.accessToken;
    
    if (
      connection.tokenExpiresAt &&
      new Date(connection.tokenExpiresAt) < new Date()
    ) {
      if (connection.refreshToken) {
        console.log("Refreshing expired access token...");
        const credentials = await getCredentials();
        if (!credentials) {
          throw new Error("No credentials available for token refresh");
        }
        
        const tokenData = await refreshAccessToken(
          credentials.clientId,
          credentials.clientSecret,
          connection.refreshToken
        );
        
        const expiresAt = new Date(
          Date.now() + tokenData.expires_in * 1000
        );
        
        await storage.updateConnectionTokens(
          connection.id,
          tokenData.access_token,
          tokenData.refresh_token,
          expiresAt
        );
        
        accessToken = tokenData.access_token;
      } else {
        throw new Error("No refresh token available");
      }
    }

    // Incremental sync - pobierz tylko zaktualizowane zamówienia
    const currentSettings = await getSyncSettingsFromPostgres();
    const lastSyncAt = currentSettings?.lastSyncAt;
    
    let allegroOrders = [];
    if (lastSyncAt) {
      // Incremental sync - tylko zaktualizowane od ostatniej synchronizacji (z paginacją)
      const cutoffTime = new Date(lastSyncAt).toISOString();
      console.log(`📦 Incremental sync - fetching Allegro orders updated after ${cutoffTime}`);
      
      let offset = 0;
      const limit = 100;
      let hasMore = true;

      while (hasMore) {
        const response = await axios.get<{ checkoutForms: AllegroOrderItem[] }>(
          `${ALLEGRO_API_URL}/order/checkout-forms`,
          {
            headers: {
              Authorization: `Bearer ${accessToken}`,
              Accept: "application/vnd.allegro.public.v1+json",
            },
            params: {
              "updatedAt.gte": cutoffTime,
              limit,
              offset,
            },
          }
        );
        
        const orders = response.data.checkoutForms || [];
        allegroOrders.push(...orders);
        
        console.log(`📦 Incremental: fetched ${orders.length} orders (offset: ${offset}, total: ${allegroOrders.length})`);
        
        if (orders.length < limit) {
          hasMore = false;
        } else {
          offset += limit;
        }
      }
    } else {
      // First sync - fetch all orders
      console.log(`📦 First sync - fetching all Allegro orders`);
      allegroOrders = await fetchOrdersFromAllegro(accessToken);
    }
    
    ordersFetched = allegroOrders.length;

    for (const allegroOrder of allegroOrders) {
      const paymentStatus =
        allegroOrder.payment?.status?.toUpperCase() || "PENDING";
      const totalAmount =
        allegroOrder.summary?.totalToPay?.amount ||
        allegroOrder.payment?.paidAmount?.amount ||
        "0";

      await storage.createOrUpdateOrder({
        id: allegroOrder.id,
        allegroOrderId: allegroOrder.id,
        buyerLogin: allegroOrder.buyer.login,
        buyerEmail: allegroOrder.buyer.email,
        totalAmount,
        currency:
          allegroOrder.summary?.totalToPay?.currency ||
          allegroOrder.payment?.paidAmount?.currency ||
          "PLN",
        paymentStatus,
        fulfillmentStatus:
          allegroOrder.fulfillment?.status?.toUpperCase() || "NEW",
        itemsCount: allegroOrder.lineItems.length.toString(),
        orderDate: new Date(allegroOrder.updatedAt),
        paymentDate:
          paymentStatus === "PAID" ? new Date(allegroOrder.updatedAt) : null,
        rawData: allegroOrder,
      });

      try {
        // Pobierz dane o przesyłkach dla tego zamówienia
        const shipments = await fetchShipmentsFromAllegro(accessToken, allegroOrder.id);
        const customerReturns = await fetchCustomerReturnsFromAllegro(accessToken, allegroOrder.id);
        
        await saveOrderToPostgres(allegroOrder, shipments, customerReturns, accessToken);
        
        const result = await saveOrderToCommerce(
          'ALLEGRO',
          allegroOrder.id,
          allegroOrder,
          shipments,
          customerReturns,
          accessToken
        );
        
        // Trigger webhooks after successful save
        await triggerWebhooksForOrder('ALLEGRO', allegroOrder.id, result.isNew);
        
        if (result.isNew) {
          ordersNew++;
        } else {
          ordersUpdated++;
        }
        
        console.log(`✅ Saved Allegro order ${allegroOrder.id} to commerce.orders and allegro.orders (${shipments.length} shipments)`);
      } catch (error) {
        console.error(`Failed to save order ${allegroOrder.id} to PostgreSQL:`, error);
      }
    }

    const settings = await getSyncSettingsFromPostgres();
    await createOrUpdateSyncSettingsInPostgres({
      autoRefreshEnabled: settings?.autoRefreshEnabled ?? true,
      refreshIntervalMinutes: settings?.refreshIntervalMinutes ?? "3",
      lastSyncAt: new Date(),
    });

    // Przebuduj numery zamówień według daty po synchronizacji
    // ❌ WYŁĄCZONE - numerowanie nie powinno się zmieniać po utworzeniu zamówienia
    // await renumberOrdersByDate();

    console.log(`Successfully synced ${allegroOrders.length} orders from Allegro (${ordersNew} new, ${ordersUpdated} updated)`);
    
    if (logId) {
      await completeSyncLog(logId, 'SUCCESS', ordersFetched, ordersNew, ordersUpdated);
    }
  } catch (error) {
    console.error("Sync error:", error);
    if (logId) {
      await completeSyncLog(logId, 'ERROR', ordersFetched, ordersNew, ordersUpdated, error instanceof Error ? error.message : 'Unknown error');
    }
    throw error;
  } finally {
    isCurrentlySyncing = false;
  }
}

function startSyncScheduler(intervalMinutes: number) {
  if (syncInterval) {
    clearInterval(syncInterval);
    syncInterval = null;
  }

  console.log(`Starting auto-sync scheduler with ${intervalMinutes} minute interval`);
  
  syncInterval = setInterval(
    () => {
      console.log("Running scheduled sync...");
      syncOrders().catch(err => console.error("Scheduled sync failed:", err));
    },
    intervalMinutes * 60 * 1000
  );
}

function stopSyncScheduler() {
  if (syncInterval) {
    clearInterval(syncInterval);
    syncInterval = null;
    console.log("Stopped auto-sync scheduler");
  }
}

function startShoperSyncScheduler(intervalMinutes: number) {
  if (shoperSyncInterval) {
    clearInterval(shoperSyncInterval);
    shoperSyncInterval = null;
  }

  console.log(`Starting Shoper auto-sync scheduler with ${intervalMinutes} minute interval`);
  
  shoperSyncInterval = setInterval(
    () => {
      console.log("Running scheduled Shoper sync...");
      syncShoperOrders().catch(err => console.error("Scheduled Shoper sync failed:", err));
    },
    intervalMinutes * 60 * 1000
  );
}

function stopShoperSyncScheduler() {
  if (shoperSyncInterval) {
    clearInterval(shoperSyncInterval);
    shoperSyncInterval = null;
    console.log("Stopped Shoper auto-sync scheduler");
  }
}

function startOdooSyncScheduler(intervalMinutes: number) {
  if (odooSyncInterval) {
    clearInterval(odooSyncInterval);
    odooSyncInterval = null;
  }

  console.log(`Starting Odoo sync scheduler with ${intervalMinutes} minute interval`);
  
  odooSyncInterval = setInterval(
    async () => {
      console.log("Running scheduled Odoo sync...");
      const { syncRecentOrdersToOdoo, processSyncQueue } = await import('./odoo-sync.js');
      console.log('🔧 Imported Odoo sync functions');
      // Najpierw dodaj nowe zamówienia do kolejki
      await syncRecentOrdersToOdoo(intervalMinutes).catch(err => console.error("Scheduled Odoo recent orders sync failed:", err));
      console.log('🔧 syncRecentOrdersToOdoo completed');
      // Potem przetwórz kolejkę (syncRecentOrdersToOdoo JUŻ wywołuje processSyncQueue, więc NIE wywołuj go ponownie!)
      // await processSyncQueue().catch(err => console.error("Scheduled Odoo queue processing failed:", err));
      console.log('🔧 Scheduled Odoo sync completed');
    },
    intervalMinutes * 60 * 1000
  );
}

function stopOdooSyncScheduler() {
  if (odooSyncInterval) {
    clearInterval(odooSyncInterval);
    odooSyncInterval = null;
    console.log("Stopped Odoo sync scheduler");
  }
}

// Recent updates checker removed - Allegro now uses incremental sync like Shoper

async function getCredentials() {
  const clientId = process.env.ALLEGRO_CLIENT_ID;
  const clientSecret = process.env.ALLEGRO_CLIENT_SECRET;
  
  if (clientId && clientSecret) {
    return { clientId, clientSecret, fromEnv: true };
  }
  
  const connection = await storage.getAllegroConnection();
  if (connection) {
    return { 
      clientId: connection.clientId, 
      clientSecret: connection.clientSecret,
      fromEnv: false 
    };
  }
  
  return null;
}

export async function autoConnectAllegro(): Promise<boolean> {
  try {
    const connection = await getAllegroConnectionFromPostgres();
    
    if (!connection) {
      console.log("No Allegro connection found - skipping auto-connect");
      return false;
    }
    
    if (!connection.accessToken || !connection.refreshToken) {
      console.log("No tokens available - manual OAuth required");
      return false;
    }
    
    // Check if token is expired
    if (connection.tokenExpiresAt && new Date(connection.tokenExpiresAt) < new Date()) {
      console.log("🔄 Allegro token expired - refreshing...");
      
      const credentials = await getCredentials();
      if (!credentials) {
        console.log("❌ No credentials available for token refresh");
        return false;
      }
      
      const tokenData = await refreshAccessToken(
        credentials.clientId,
        credentials.clientSecret,
        connection.refreshToken
      );
      
      const expiresAt = new Date(Date.now() + tokenData.expires_in * 1000);
      
      await updateAllegroConnectionTokens(
        tokenData.access_token,
        tokenData.refresh_token,
        expiresAt
      );
      
      // Also update MemStorage for immediate use
      await storage.updateConnectionTokens(
        connection.id,
        tokenData.access_token,
        tokenData.refresh_token,
        expiresAt
      );
      
      console.log("✅ Allegro token refreshed successfully");
    } else {
      console.log("✅ Allegro token still valid");
    }
    
    // Load connection into MemStorage for use during this session
    await storage.createOrUpdateConnection({
      clientId: connection.clientId,
      clientSecret: connection.clientSecret,
      accessToken: connection.accessToken,
      refreshToken: connection.refreshToken,
      tokenExpiresAt: connection.tokenExpiresAt ?? undefined,
      isActive: true,
    });
    
    console.log("✅ Allegro connection loaded and ready");
    
    return true;
  } catch (error) {
    console.error("❌ Auto-connect to Allegro failed:", error);
    return false;
  }
}

let isShoperSyncing = false;

async function syncShoperOrders() {
  if (isShoperSyncing) {
    console.log("Shoper sync already in progress, skipping...");
    return;
  }

  isShoperSyncing = true;
  const syncStartedAt = new Date();
  let ordersNew = 0;
  let ordersUpdated = 0;
  let ordersFetched = 0;
  let ordersFailed = 0;
  
  try {
    const { getShoperOrders, getShoperOrderProducts, getShoperParcels, getShoperPayments, getShoperDeliveries } = await import('./shoper-api.js');
    const { saveOrderToCommerce, saveShoperOrderToPostgres } = await import('./postgres.js');

    console.log("🛒 Starting Shoper orders sync...");
    
    // Pobierz słowniki metod płatności i dostaw
    const payments = await getShoperPayments();
    const deliveries = await getShoperDeliveries();
    
    // Use translated title (pl_PL) for payment method name, fallback to technical name
    const paymentMap = new Map(payments.map((p: any) => [p.payment_id, p.translations?.pl_PL?.title || p.name]));
    const deliveryMap = new Map(deliveries.map((d: any) => [d.delivery_id, d.translations?.pl_PL?.name || d.name]));
    
    console.log(`📋 Loaded ${payments.length} payment methods and ${deliveries.length} delivery methods`);
    console.log(`💳 Payment methods:`, Array.from(paymentMap.entries()));
    console.log(`🚚 Delivery methods:`, Array.from(deliveryMap.entries()));
    
    const settings = await getSyncSettingsFromPostgres();
    const lastShoperSyncAt = settings?.lastShoperSyncAt;
    
    let filters: Record<string, any> | undefined;
    if (lastShoperSyncAt) {
      const filterDate = new Date(lastShoperSyncAt).toISOString().replace('T', ' ').replace('Z', '');
      filters = {
        updated_at: {
          ">": filterDate
        }
      };
      console.log(`🔍 Using incremental sync filter: updated_at > ${filterDate}`);
    } else {
      console.log("📦 First sync - fetching all orders");
    }
    
    const shoperOrders = await getShoperOrders(100, filters);
    ordersFetched = shoperOrders.length;

    console.log(`📦 Fetched ${ordersFetched} orders from Shoper API`);

    for (const shoperOrder of shoperOrders) {
      try {
        const orderId = shoperOrder.order_id || shoperOrder.id;
        
        console.log(`📦 Processing Shoper order ${orderId}...`);
        
        // Wzbogać zamówienie o nazwy metod płatności i dostaw
        if (shoperOrder.payment_id && paymentMap.has(shoperOrder.payment_id.toString())) {
          shoperOrder.payment_method_name = paymentMap.get(shoperOrder.payment_id.toString());
        } else if (shoperOrder.payment_id) {
          console.log(`⚠️ Payment ID ${shoperOrder.payment_id} not found in paymentMap`);
        }
        if (shoperOrder.shipping_id && deliveryMap.has(shoperOrder.shipping_id.toString())) {
          shoperOrder.delivery_method_name = deliveryMap.get(shoperOrder.shipping_id.toString());
        }
        
        try {
          const orderProducts = await getShoperOrderProducts(orderId);
          shoperOrder.products_data = orderProducts;
          console.log(`✅ Fetched ${orderProducts.length} products for order ${orderId}`);
        } catch (productError) {
          console.warn(`⚠️ Could not fetch products for order ${orderId}, will save order without line items:`, productError);
          shoperOrder.products_data = [];
        }
        
        // Pobierz przesyłki dla zamówienia
        let parcels: any[] = [];
        try {
          parcels = await getShoperParcels(orderId);
          console.log(`✅ Fetched ${parcels.length} parcels for order ${orderId}`);
        } catch (parcelError) {
          console.warn(`⚠️ Could not fetch parcels for order ${orderId}:`, parcelError);
        }
        
        await saveShoperOrderToPostgres(shoperOrder, parcels);
        
        const result = await saveOrderToCommerce(
          'SHOPER',
          orderId?.toString(),
          shoperOrder,
          parcels,
          [] // Shoper nie ma zwrotów
        );
        
        // Trigger webhooks after successful save
        await triggerWebhooksForOrder('SHOPER', orderId?.toString(), result.isNew);
        
        if (result.isNew) {
          ordersNew++;
        } else {
          ordersUpdated++;
        }
        
        console.log(`✅ Saved Shoper order ${orderId} to commerce.orders and shoper.orders (${parcels.length} parcels)`);
      } catch (error) {
        ordersFailed++;
        console.error(`❌ Failed to process Shoper order ${shoperOrder.order_id}:`, error);
      }
    }

    await createOrUpdateSyncSettingsInPostgres({
      autoRefreshEnabled: settings?.autoRefreshEnabled ?? true,
      refreshIntervalMinutes: settings?.refreshIntervalMinutes ?? "3",
      lastSyncAt: settings?.lastSyncAt || null,
      shoperAutoRefreshEnabled: settings?.shoperAutoRefreshEnabled ?? true,
      shoperRefreshIntervalMinutes: settings?.shoperRefreshIntervalMinutes ?? "5",
      lastShoperSyncAt: new Date(),
    });

    // Przebuduj numery zamówień według daty po synchronizacji
    // ❌ WYŁĄCZONE - numerowanie nie powinno się zmieniać po utworzeniu zamówienia
    // await renumberOrdersByDate();

    const message = `Successfully synced Shoper orders: ${ordersNew} new, ${ordersUpdated} updated, ${ordersFailed} failed out of ${ordersFetched} total`;
    console.log(`✅ ${message}`);
    
    return { 
      success: true, 
      ordersFetched, 
      ordersNew, 
      ordersUpdated,
      ordersFailed
    };
  } catch (error) {
    console.error("❌ Shoper sync error:", error);
    throw error;
  } finally {
    isShoperSyncing = false;
  }
}

export async function registerRoutes(app: Express): Promise<Server> {
  // Setup authentication
  setupAuth(app);

  // Swagger API Documentation (accessible to authenticated users)
  app.use('/api-docs', swaggerUi.serve);
  app.get('/api-docs', swaggerUi.setup(swaggerSpec, {
    customCss: '.swagger-ui .topbar { display: none }',
    customSiteTitle: 'Alpma OMS API Documentation',
  }));

  // Auth routes
  app.post("/api/register", async (req, res, next) => {
    try {
      const { username, email, password, firstName, lastName } = req.body;

      // Check if username already exists
      const existingUser = await getUserByUsername(username);
      if (existingUser) {
        return res.status(400).json({ message: "Nazwa użytkownika jest już zajęta" });
      }

      // Check if email already exists
      const existingEmail = await getUserByEmail(email);
      if (existingEmail) {
        return res.status(400).json({ message: "Email jest już zarejestrowany" });
      }

      // Hash password and create user
      const hashedPassword = await hashPassword(password);
      const user = await createUser({
        username,
        email,
        password: hashedPassword,
        firstName,
        lastName,
      });

      // Log in the user after registration
      req.login(user, (err) => {
        if (err) return next(err);
        const { password: _, ...userWithoutPassword } = user;
        res.status(201).json(userWithoutPassword);
      });
    } catch (error: any) {
      console.error("Registration error:", error);
      res.status(500).json({ message: "Błąd podczas rejestracji" });
    }
  });

  app.post("/api/login", (req, res, next) => {
    passport.authenticate("local", (err: any, user: any, info: any) => {
      if (err) {
        return next(err);
      }
      if (!user) {
        return res.status(401).json({ message: info?.message || "Nieprawidłowe dane logowania" });
      }
      req.login(user, (err) => {
        if (err) {
          return next(err);
        }
        res.status(200).json(user);
      });
    })(req, res, next);
  });

  app.post("/api/logout", (req, res, next) => {
    req.logout((err) => {
      if (err) return next(err);
      res.sendStatus(200);
    });
  });

  app.get("/api/user", (req, res) => {
    if (!req.isAuthenticated()) {
      return res.status(401).json({ message: "Unauthorized" });
    }
    res.json(req.user);
  });

  // Password reset routes
  app.post("/api/password-reset/request", async (req, res) => {
    try {
      const { email } = req.body;
      const user = await getUserByEmail(email);
      
      if (!user) {
        // Don't reveal if email exists or not for security
        return res.status(200).json({ 
          message: "Jeśli email istnieje w systemie, otrzymasz link do resetowania hasła" 
        });
      }

      // Generate reset token
      const resetToken = randomBytes(32).toString("hex");
      const expiresAt = new Date(Date.now() + 3600000); // 1 hour
      
      await createPasswordResetToken(user.id, resetToken, expiresAt);

      // In production, send email with reset link here
      // For now, just return the token (development only)
      res.status(200).json({ 
        message: "Link do resetowania hasła został wysłany na podany email",
        resetToken, // Remove in production!
      });
    } catch (error) {
      console.error("Password reset request error:", error);
      res.status(500).json({ message: "Błąd podczas żądania resetowania hasła" });
    }
  });

  app.post("/api/password-reset/verify", async (req, res) => {
    try {
      const { token } = req.body;
      const resetToken = await getPasswordResetToken(token);
      
      if (!resetToken) {
        return res.status(400).json({ message: "Nieprawidłowy lub wygasły token" });
      }

      res.status(200).json({ message: "Token jest prawidłowy" });
    } catch (error) {
      console.error("Token verification error:", error);
      res.status(500).json({ message: "Błąd podczas weryfikacji tokenu" });
    }
  });

  app.post("/api/password-reset/reset", async (req, res) => {
    try {
      const { token, newPassword } = req.body;
      const resetToken = await getPasswordResetToken(token);
      
      if (!resetToken) {
        return res.status(400).json({ message: "Nieprawidłowy lub wygasły token" });
      }

      // Hash new password and update
      const hashedPassword = await hashPassword(newPassword);
      await updateUserPassword(resetToken.user_id, hashedPassword);
      await markTokenAsUsed(token);

      res.status(200).json({ message: "Hasło zostało pomyślnie zresetowane" });
    } catch (error) {
      console.error("Password reset error:", error);
      res.status(500).json({ message: "Błąd podczas resetowania hasła" });
    }
  });

  // Permission middleware
  const requirePermission = (permission: Permission) => {
    return (req: any, res: any, next: any) => {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }
      
      const userRole = req.user.role;
      if (!hasPermission(userRole, permission)) {
        return res.status(403).json({ 
          message: "Forbidden - Insufficient permissions",
          required: permission,
          role: userRole
        });
      }
      
      next();
    };
  };

  // User management routes (admin only)
  const requireAdmin = (req: any, res: any, next: any) => {
    if (!req.isAuthenticated()) {
      return res.status(401).json({ message: "Unauthorized" });
    }
    if (req.user.role !== 'admin') {
      return res.status(403).json({ message: "Forbidden - Admin access required" });
    }
    next();
  };

  app.get("/api/users", requireAdmin, async (req, res) => {
    try {
      const users = await getAllUsers();
      // Don't send passwords to frontend
      const usersWithoutPasswords = users.map(({ password, ...user }: any) => user);
      res.json(usersWithoutPasswords);
    } catch (error) {
      console.error("Get users error:", error);
      res.status(500).json({ message: "Błąd podczas pobierania użytkowników" });
    }
  });

  app.put("/api/users/:id/role", requireAdmin, async (req, res) => {
    try {
      const userId = parseInt(req.params.id);
      const { role } = req.body;
      
      if (!['admin', 'manager', 'user'].includes(role)) {
        return res.status(400).json({ message: "Nieprawidłowa rola" });
      }

      const user = await updateUserRole(userId, role);
      const { password, ...userWithoutPassword } = user;
      res.json(userWithoutPassword);
    } catch (error) {
      console.error("Update user role error:", error);
      res.status(500).json({ message: "Błąd podczas aktualizacji roli" });
    }
  });

  app.put("/api/users/:id/permissions", requireAdmin, async (req, res) => {
    try {
      const userId = parseInt(req.params.id);
      const { permissions } = req.body;
      
      const user = await updateUserPermissions(userId, permissions);
      const { password, ...userWithoutPassword } = user;
      res.json(userWithoutPassword);
    } catch (error) {
      console.error("Update user permissions error:", error);
      res.status(500).json({ message: "Błąd podczas aktualizacji uprawnień" });
    }
  });

  app.put("/api/users/:id/status", requireAdmin, async (req, res) => {
    try {
      const userId = parseInt(req.params.id);
      const { isActive } = req.body;
      
      const user = await updateUserStatus(userId, isActive);
      const { password, ...userWithoutPassword } = user;
      res.json(userWithoutPassword);
    } catch (error) {
      console.error("Update user status error:", error);
      res.status(500).json({ message: "Błąd podczas aktualizacji statusu" });
    }
  });

  app.delete("/api/users/:id", requireAdmin, async (req, res) => {
    try {
      const userId = parseInt(req.params.id);
      
      // Prevent deleting yourself
      if (req.user?.id === userId) {
        return res.status(400).json({ message: "Nie możesz usunąć własnego konta" });
      }

      await deleteUser(userId);
      res.status(200).json({ message: "Użytkownik został usunięty" });
    } catch (error) {
      console.error("Delete user error:", error);
      res.status(500).json({ message: "Błąd podczas usuwania użytkownika" });
    }
  });

  app.get("/api/allegro/auth", requirePermission('manage_credentials'), async (req, res) => {
    try {
      const credentials = await getCredentials();
      if (!credentials) {
        return res.status(400).json({ error: "No credentials configured" });
      }

      const state = randomBytes(32).toString("hex");
      oauthStates.set(state, { timestamp: Date.now() });
      
      setTimeout(() => oauthStates.delete(state), 10 * 60 * 1000);

      const redirectUri = `https://${req.get("host")}/api/allegro/callback`;
      
      const authUrl = `${ALLEGRO_AUTH_URL}?response_type=code&client_id=${credentials.clientId}&redirect_uri=${encodeURIComponent(redirectUri)}&state=${state}`;

      res.redirect(authUrl);
    } catch (error) {
      console.error("Auth initiation error:", error);
      res.status(500).json({ error: "Auth initiation failed" });
    }
  });

  app.get("/api/allegro/callback", async (req, res) => {
    try {
      const { code, state } = req.query;
      
      if (!code || typeof code !== "string") {
        return res.redirect("/settings?error=no_code");
      }

      if (!state || typeof state !== "string" || !oauthStates.has(state)) {
        return res.redirect("/settings?error=invalid_state");
      }

      oauthStates.delete(state);

      const credentials = await getCredentials();
      if (!credentials) {
        return res.redirect("/settings?error=no_credentials");
      }

      const redirectUri = `https://${req.get("host")}/api/allegro/callback`;
      const authHeader = Buffer.from(
        `${credentials.clientId}:${credentials.clientSecret}`
      ).toString("base64");

      const response = await axios.post<AllegroTokenResponse>(
        ALLEGRO_TOKEN_URL,
        new URLSearchParams({
          grant_type: "authorization_code",
          code,
          redirect_uri: redirectUri,
        }),
        {
          headers: {
            Authorization: `Basic ${authHeader}`,
            "Content-Type": "application/x-www-form-urlencoded",
          },
        }
      );

      const { access_token, refresh_token, expires_in } = response.data;
      const expiresAt = new Date(Date.now() + expires_in * 1000);

      // Zapisz token do PostgreSQL (persystencja po restarcie)
      await saveAllegroConnectionToPostgres({
        clientId: credentials.clientId,
        clientSecret: credentials.clientSecret,
        accessToken: access_token,
        refreshToken: refresh_token,
        tokenExpiresAt: expiresAt,
        isActive: true,
      });

      // Zapisz też do MemStorage (do użycia w tej sesji)
      await storage.createOrUpdateConnection({
        clientId: credentials.clientId,
        clientSecret: credentials.clientSecret,
        accessToken: access_token,
        refreshToken: refresh_token,
        tokenExpiresAt: expiresAt,
        isActive: true,
      });

      await syncOrders();

      const settings = await getSyncSettingsFromPostgres();
      if (settings?.autoRefreshEnabled) {
        const interval = parseInt(settings.refreshIntervalMinutes || "3");
        startSyncScheduler(interval);
      }

      res.redirect("/settings?connected=true");
    } catch (error) {
      console.error("OAuth callback error:", error);
      res.redirect("/settings?error=auth_failed");
    }
  });

  app.get("/api/allegro/connection", async (req, res) => {
    try {
      const credentials = await getCredentials();
      if (!credentials) {
        return res.json({ 
          isActive: false,
          hasCredentials: false,
          fromEnv: false
        });
      }

      const connection = await storage.getAllegroConnection();
      
      res.json({
        isActive: connection?.isActive || false,
        hasCredentials: true,
        fromEnv: credentials.fromEnv,
        clientId: credentials.clientId,
      });
    } catch (error) {
      console.error("Get connection error:", error);
      res.status(500).json({ error: "Failed to get connection status" });
    }
  });

  // Bardziej specyficzne ścieżki muszą być PRZED ogólnymi
  app.get("/api/orders/statistics", async (req, res) => {
    try {
      const stats = await getOrderStatistics();
      res.json(stats);
    } catch (error) {
      console.error("Get order statistics error:", error);
      res.status(500).json({ error: "Failed to fetch statistics" });
    }
  });

  app.get("/api/orders/chart", async (req, res) => {
    try {
      const days = parseInt(req.query.days as string) || 30;
      const chartData = await getOrdersChartData(days);
      res.json(chartData);
    } catch (error) {
      console.error("Get chart data error:", error);
      res.status(500).json({ error: "Failed to fetch chart data" });
    }
  });

  app.get("/api/orders/period-summary", async (req, res) => {
    try {
      const days = parseInt(req.query.days as string) || 30;
      const summary = await getOrdersPeriodSummary(days);
      res.json(summary);
    } catch (error) {
      console.error("Get period summary error:", error);
      res.status(500).json({ error: "Failed to fetch period summary" });
    }
  });

  app.get("/api/orders/today-stats", async (req, res) => {
    try {
      const stats = await getTodayDetailedStats();
      res.json(stats);
    } catch (error) {
      console.error("Get today stats error:", error);
      res.status(500).json({ error: "Failed to fetch today stats" });
    }
  });

  app.get("/api/orders/by-number/:orderNumber", async (req, res) => {
    try {
      const { orderNumber } = req.params;
      const order = await getOrderByNumberFromPostgres(parseInt(orderNumber));
      
      if (!order) {
        return res.status(404).json({ error: "Order not found" });
      }
      
      res.json(order);
    } catch (error) {
      console.error("Get order by number error:", error);
      res.status(500).json({ error: "Failed to fetch order" });
    }
  });

  app.get("/api/orders/count", async (req, res) => {
    try {
      const { getOrdersCount } = await import('./postgres.js');
      const search = typeof req.query.search === 'string' ? req.query.search : undefined;
      const sourceFilter = typeof req.query.sourceFilter === 'string' ? req.query.sourceFilter : undefined;
      const dateRangeType = typeof req.query.dateRangeType === 'string' ? req.query.dateRangeType : undefined;
      const customDateFrom = typeof req.query.customDateFrom === 'string' ? req.query.customDateFrom : undefined;
      const customDateTo = typeof req.query.customDateTo === 'string' ? req.query.customDateTo : undefined;
      
      const count = await getOrdersCount(
        search,
        sourceFilter,
        dateRangeType,
        customDateFrom,
        customDateTo
      );
      res.json({ count });
    } catch (error) {
      console.error("Get orders count error:", error);
      res.status(500).json({ error: "Failed to fetch orders count" });
    }
  });

  app.get("/api/orders/:allegroOrderId/raw", async (req, res) => {
    try {
      const { allegroOrderId } = req.params;
      const connection = await storage.getAllegroConnection();
      
      if (!connection || !connection.accessToken) {
        return res.status(401).json({ error: "No active Allegro connection" });
      }

      const rawOrder = await fetchRawOrderFromAllegro(connection.accessToken, allegroOrderId);
      res.json(rawOrder);
    } catch (error: any) {
      console.error("Get raw order error:", error);
      res.status(500).json({ error: error.message || "Failed to fetch raw order" });
    }
  });

  app.get("/api/orders/:allegroOrderId", async (req, res) => {
    try {
      const { allegroOrderId } = req.params;
      const order = await getOrderFromPostgres(allegroOrderId);
      
      if (!order) {
        return res.status(404).json({ error: "Order not found" });
      }
      
      res.json(order);
    } catch (error) {
      console.error("Get order error:", error);
      res.status(500).json({ error: "Failed to fetch order" });
    }
  });

  app.get("/api/orders", async (req, res) => {
    try {
      const sortBy = typeof req.query.sortBy === 'string' ? req.query.sortBy : 'order_date';
      const sortOrder = (typeof req.query.sortOrder === 'string' && req.query.sortOrder.toUpperCase() === 'ASC') ? 'ASC' : 'DESC';
      const limit = typeof req.query.limit === 'string' ? parseInt(req.query.limit, 10) : 50;
      const offset = typeof req.query.offset === 'string' ? parseInt(req.query.offset, 10) : 0;
      const search = typeof req.query.search === 'string' ? req.query.search : undefined;
      const sourceFilter = typeof req.query.sourceFilter === 'string' ? req.query.sourceFilter : undefined;
      const dateRangeType = typeof req.query.dateRangeType === 'string' ? req.query.dateRangeType : undefined;
      const customDateFrom = typeof req.query.customDateFrom === 'string' ? req.query.customDateFrom : undefined;
      const customDateTo = typeof req.query.customDateTo === 'string' ? req.query.customDateTo : undefined;
      
      const orders = await getOrdersFromPostgres(
        sortBy, 
        sortOrder, 
        limit, 
        offset, 
        search,
        sourceFilter,
        dateRangeType,
        customDateFrom,
        customDateTo
      );
      res.json(orders);
    } catch (error) {
      console.error("Get orders error:", error);
      res.status(500).json({ error: "Failed to fetch orders" });
    }
  });

  app.post("/api/products/download-images", async (req, res) => {
    try {
      const { downloadProductImages } = await import('./allegro-api.js');
      const { getUniqueProductsFromPostgres } = await import('./postgres.js');
      
      const connection = await storage.getAllegroConnection();
      if (!connection || !connection.accessToken) {
        return res.status(401).json({ error: "Brak połączenia z Allegro" });
      }
      
      const products = await getUniqueProductsFromPostgres();
      
      console.log('🔍 DEBUG - First product from DB:', products[0]);
      
      if (products.length === 0) {
        return res.json({ success: 0, failed: 0, message: "Nie znaleziono produktów" });
      }

      const items = products.map((p: any) => ({
        productName: p.name,
        externalId: p.offer_external_id
      }));

      console.log('🔍 DEBUG - First mapped item:', items[0]);
      console.log(`📦 Starting download of ${items.length} product images by searching product names`);
      const result = await downloadProductImages(items, connection.accessToken);
      res.json({ 
        success: result.success, 
        failed: result.failed,
        total: items.length,
        message: `Pobrano ${result.success}/${items.length} zdjęć produktów`
      });
    } catch (error: any) {
      console.error("Download images error:", error);
      res.status(500).json({ error: error.message || "Nie udało się pobrać zdjęć" });
    }
  });

  app.get("/api/sync/status", async (req, res) => {
    try {
      const settings = await getSyncSettingsFromPostgres();
      res.json({
        lastSyncAt: settings?.lastSyncAt || null,
        isRefreshing: isCurrentlySyncing,
      });
    } catch (error) {
      console.error("Get sync status error:", error);
      res.status(500).json({ error: "Failed to get sync status" });
    }
  });

  app.post("/api/sync/settings", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { autoRefreshEnabled, refreshIntervalMinutes } = req.body;

      const currentSettings = await getSyncSettingsFromPostgres();
      const settings = await createOrUpdateSyncSettingsInPostgres({
        autoRefreshEnabled,
        refreshIntervalMinutes,
        lastSyncAt: currentSettings?.lastSyncAt || null,
      });

      if (autoRefreshEnabled) {
        const interval = parseInt(refreshIntervalMinutes);
        startSyncScheduler(interval);
      } else {
        stopSyncScheduler();
      }

      res.json(settings);
    } catch (error) {
      console.error("Save sync settings error:", error);
      res.status(500).json({ error: "Failed to save sync settings" });
    }
  });

  app.get("/api/sync/settings", requirePermission('manage_settings'), async (req, res) => {
    try {
      const settings = await getSyncSettingsFromPostgres();
      if (!settings) {
        return res.json({
          autoRefreshEnabled: true,
          refreshIntervalMinutes: "3",
        });
      }
      res.json(settings);
    } catch (error) {
      console.error("Get sync settings error:", error);
      res.status(500).json({ error: "Failed to get sync settings" });
    }
  });

  app.get("/api/sync/logs", requirePermission('view_sync_logs'), async (req, res) => {
    try {
      const logs = await getSyncLogs(50);
      res.json(logs);
    } catch (error) {
      console.error("Get sync logs error:", error);
      res.status(500).json({ error: "Failed to fetch sync logs" });
    }
  });

  app.post("/api/sync/manual", requirePermission('manage_sync'), async (req, res) => {
    try {
      await syncOrders();
      res.json({ success: true });
    } catch (error) {
      console.error("Manual sync error:", error);
      res.status(500).json({ error: "Failed to sync orders" });
    }
  });

  app.post("/api/sync/date-range", requirePermission('manage_sync'), async (req, res) => {
    try {
      const { startDate, endDate } = req.body;
      
      if (!startDate || !endDate) {
        return res.status(400).json({ error: "startDate and endDate are required" });
      }
      
      const connection = await storage.getAllegroConnection();
      if (!connection || !connection.accessToken || !connection.isActive) {
        return res.status(400).json({ error: "No active Allegro connection" });
      }

      let accessToken = connection.accessToken;
      
      if (
        connection.tokenExpiresAt &&
        new Date(connection.tokenExpiresAt) < new Date()
      ) {
        if (connection.refreshToken) {
          const credentials = await getCredentials();
          if (!credentials) {
            return res.status(400).json({ error: "No credentials available for token refresh" });
          }
          
          const tokenData = await refreshAccessToken(
            credentials.clientId,
            credentials.clientSecret,
            connection.refreshToken
          );
          
          const expiresAt = new Date(Date.now() + tokenData.expires_in * 1000);
          
          await storage.updateConnectionTokens(
            connection.id,
            tokenData.access_token,
            tokenData.refresh_token,
            expiresAt
          );
          
          accessToken = tokenData.access_token;
        } else {
          return res.status(400).json({ error: "Token expired and no refresh token available" });
        }
      }

      const startDateTime = new Date(startDate);
      startDateTime.setHours(0, 0, 0, 0);
      const start = startDateTime.toISOString();
      
      const endDateTime = new Date(endDate);
      endDateTime.setHours(23, 59, 59, 999);
      const end = endDateTime.toISOString();
      
      console.log(`📅 Fetching Allegro orders from ${start} to ${end}`);
      
      let allegroOrders: AllegroOrderItem[] = [];
      let offset = 0;
      const limit = 100;
      let hasMore = true;
      
      while (hasMore) {
        const response = await axios.get<{ checkoutForms: AllegroOrderItem[] }>(
          `${ALLEGRO_API_URL}/order/checkout-forms`,
          {
            headers: {
              Authorization: `Bearer ${accessToken}`,
              Accept: "application/vnd.allegro.public.v1+json",
            },
            params: {
              "updatedAt.gte": start,
              "updatedAt.lte": end,
              limit,
              offset,
            },
          }
        );
        
        const fetchedOrders = response.data.checkoutForms || [];
        allegroOrders = allegroOrders.concat(fetchedOrders);
        
        if (fetchedOrders.length < limit) {
          hasMore = false;
        } else {
          offset += limit;
        }
      }
      
      console.log(`📦 Fetched ${allegroOrders.length} total orders from Allegro`);
      let ordersNew = 0;
      let ordersUpdated = 0;
      
      for (const allegroOrder of allegroOrders) {
        try {
          const shipments = await fetchShipmentsFromAllegro(accessToken, allegroOrder.id);
          const customerReturns = await fetchCustomerReturnsFromAllegro(accessToken, allegroOrder.id);
          await saveOrderToPostgres(allegroOrder, shipments, customerReturns, accessToken);
          
          const result = await saveOrderToCommerce(
            'ALLEGRO',
            allegroOrder.id,
            allegroOrder,
            shipments,
            customerReturns,
            accessToken
          );
          
          if (result.isNew) {
            ordersNew++;
          } else {
            ordersUpdated++;
          }
        } catch (error) {
          console.error(`Failed to save order ${allegroOrder.id}:`, error);
        }
      }
      
      res.json({ 
        success: true, 
        ordersFetched: allegroOrders.length,
        ordersNew,
        ordersUpdated
      });
    } catch (error: any) {
      console.error("Allegro date range sync error:", error);
      res.status(500).json({ error: error.message || "Failed to sync Allegro orders by date range" });
    }
  });

  app.post("/api/orders/:orderId/refresh", async (req, res) => {
    try {
      const { orderId } = req.params;
      
      const orderResult = await pool.query(`
        SELECT id, source, source_order_id 
        FROM commerce.orders 
        WHERE id = $1
      `, [orderId]);
      
      if (orderResult.rows.length === 0) {
        return res.status(404).json({ error: "Order not found" });
      }
      
      const order = orderResult.rows[0];
      const { source, source_order_id } = order;
      
      if (source === 'ALLEGRO') {
        const connection = await storage.getAllegroConnection();
        if (!connection || !connection.accessToken || !connection.isActive) {
          return res.status(400).json({ error: "No active Allegro connection" });
        }

        let accessToken = connection.accessToken;
        
        if (
          connection.tokenExpiresAt &&
          new Date(connection.tokenExpiresAt) < new Date()
        ) {
          if (connection.refreshToken) {
            const credentials = await getCredentials();
            if (!credentials) {
              return res.status(400).json({ error: "No credentials available for token refresh" });
            }
            
            const tokenData = await refreshAccessToken(
              credentials.clientId,
              credentials.clientSecret,
              connection.refreshToken
            );
            
            const expiresAt = new Date(Date.now() + tokenData.expires_in * 1000);
            
            await storage.updateConnectionTokens(
              connection.id,
              tokenData.access_token,
              tokenData.refresh_token,
              expiresAt
            );
            
            accessToken = tokenData.access_token;
          } else {
            return res.status(400).json({ error: "Token expired and no refresh token available" });
          }
        }
        
        const allegroOrder = await fetchRawOrderFromAllegro(accessToken, source_order_id);
        const shipments = await fetchShipmentsFromAllegro(accessToken, source_order_id);
        const customerReturns = await fetchCustomerReturnsFromAllegro(accessToken, source_order_id);
        
        // Pobierz zwroty płatności (stabilniejsze dane niż customerReturns)
        const { fetchPaymentRefunds } = await import('./allegro-api.js');
        const refunds = await fetchPaymentRefunds(source_order_id, accessToken);
        
        // Dodaj refund_reconciliation do allegroOrder.payment
        if (refunds && refunds.length > 0 && allegroOrder.payment) {
          allegroOrder.payment.refundReconciliation = refunds;
          allegroOrder.payment.refundAmount = refunds[0]?.totalValue?.amount || 0;
          allegroOrder.payment.refundDate = refunds[0]?.createdAt || null;
        }
        
        await saveOrderToPostgres(allegroOrder, shipments, customerReturns, accessToken);
        await saveOrderToCommerce('ALLEGRO', source_order_id, allegroOrder, shipments, customerReturns, accessToken);
        
        res.json({ success: true, platform: 'ALLEGRO' });
      } else if (source === 'SHOPER') {
        const { getShoperOrders, getShoperOrderProducts, getShoperParcels, getShoperPayments, getShoperDeliveries } = await import('./shoper-api.js');
        
        const payments = await getShoperPayments();
        const deliveries = await getShoperDeliveries();
        
        // Use translated title (pl_PL) for payment method name, fallback to technical name
        const paymentMap = new Map(payments.map((p: any) => [p.payment_id, p.translations?.pl_PL?.title || p.name]));
        const deliveryMap = new Map(deliveries.map((d: any) => [d.delivery_id, d.name]));
        
        const shoperOrders = await getShoperOrders(1, { order_id: source_order_id });
        
        if (shoperOrders.length === 0) {
          return res.status(404).json({ error: "Order not found on Shoper platform" });
        }
        
        const shoperOrder = shoperOrders[0];
        
        if (shoperOrder.payment_id && paymentMap.has(shoperOrder.payment_id.toString())) {
          shoperOrder.payment_method_name = paymentMap.get(shoperOrder.payment_id.toString());
        }
        if (shoperOrder.shipping_id && deliveryMap.has(shoperOrder.shipping_id.toString())) {
          shoperOrder.delivery_method_name = deliveryMap.get(shoperOrder.shipping_id.toString());
        }
        
        const orderProducts = await getShoperOrderProducts(source_order_id);
        shoperOrder.products_data = orderProducts;
        
        let parcels: any[] = [];
        try {
          parcels = await getShoperParcels(source_order_id);
        } catch (parcelError) {
          console.warn(`⚠️ Could not fetch parcels for order ${source_order_id}`);
        }
        
        await saveShoperOrderToPostgres(shoperOrder, parcels);
        await saveOrderToCommerce('SHOPER', source_order_id, shoperOrder, parcels, []);
        
        res.json({ success: true, platform: 'SHOPER' });
      } else {
        return res.status(400).json({ error: "Unknown platform" });
      }
    } catch (error: any) {
      console.error("Refresh order error:", error);
      res.status(500).json({ error: error.message || "Failed to refresh order" });
    }
  });

  app.post("/api/shoper/sync", requirePermission('manage_sync'), async (req, res) => {
    try {
      const result = await syncShoperOrders();
      res.json(result);
    } catch (error: any) {
      console.error("Shoper sync error:", error);
      res.status(500).json({ error: error.message || "Failed to sync Shoper orders" });
    }
  });

  app.post("/api/shoper/sync/date-range", requirePermission('manage_sync'), async (req, res) => {
    try {
      const { startDate, endDate } = req.body;
      
      if (!startDate || !endDate) {
        return res.status(400).json({ error: "startDate and endDate are required" });
      }
      
      const { getShoperOrders, getShoperOrderProducts, getShoperParcels, getShoperPayments, getShoperDeliveries } = await import('./shoper-api.js');
      const { saveOrderToCommerce, saveShoperOrderToPostgres } = await import('./postgres.js');

      const startDateTime = new Date(startDate);
      startDateTime.setHours(0, 0, 0, 0);
      const start = startDateTime.toISOString().replace('T', ' ').replace('Z', '');
      
      const endDateTime = new Date(endDate);
      endDateTime.setHours(23, 59, 59, 999);
      const end = endDateTime.toISOString().replace('T', ' ').replace('Z', '');
      
      console.log(`📅 Fetching Shoper orders from ${start} to ${end}`);
      
      const payments = await getShoperPayments();
      const deliveries = await getShoperDeliveries();
      
      // Use translated title (pl_PL) for payment method name, fallback to technical name
      const paymentMap = new Map(payments.map((p: any) => [p.payment_id, p.translations?.pl_PL?.title || p.name]));
      const deliveryMap = new Map(deliveries.map((d: any) => [d.delivery_id, d.name]));
      
      const filters = {
        updated_at: {
          ">=": start,
          "<=": end
        }
      };
      
      const shoperOrders = await getShoperOrders(100, filters);
      let ordersNew = 0;
      let ordersUpdated = 0;
      let ordersFailed = 0;

      console.log(`📦 Fetched ${shoperOrders.length} orders from Shoper API`);

      for (const shoperOrder of shoperOrders) {
        try {
          const orderId = shoperOrder.order_id || shoperOrder.id;
          
          if (shoperOrder.payment_id && paymentMap.has(shoperOrder.payment_id.toString())) {
            shoperOrder.payment_method_name = paymentMap.get(shoperOrder.payment_id.toString());
          }
          if (shoperOrder.shipping_id && deliveryMap.has(shoperOrder.shipping_id.toString())) {
            shoperOrder.delivery_method_name = deliveryMap.get(shoperOrder.shipping_id.toString());
          }
          
          try {
            const orderProducts = await getShoperOrderProducts(orderId);
            shoperOrder.products_data = orderProducts;
          } catch (productError) {
            console.warn(`⚠️ Could not fetch products for order ${orderId}`);
            shoperOrder.products_data = [];
          }
          
          let parcels: any[] = [];
          try {
            parcels = await getShoperParcels(orderId);
          } catch (parcelError) {
            console.warn(`⚠️ Could not fetch parcels for order ${orderId}`);
          }
          
          await saveShoperOrderToPostgres(shoperOrder, parcels);
          
          const result = await saveOrderToCommerce(
            'SHOPER',
            orderId?.toString(),
            shoperOrder,
            parcels,
            [] // Shoper nie ma zwrotów
          );
          
          if (result.isNew) {
            ordersNew++;
          } else {
            ordersUpdated++;
          }
        } catch (error) {
          ordersFailed++;
          console.error(`❌ Failed to process Shoper order ${shoperOrder.order_id}:`, error);
        }
      }
      
      res.json({ 
        success: true, 
        ordersFetched: shoperOrders.length,
        ordersNew,
        ordersUpdated,
        ordersFailed
      });
    } catch (error: any) {
      console.error("Shoper date range sync error:", error);
      res.status(500).json({ error: error.message || "Failed to sync Shoper orders by date range" });
    }
  });

  app.get("/api/shoper/sync/status", async (req, res) => {
    try {
      const settings = await getSyncSettingsFromPostgres();
      res.json({
        lastShoperSyncAt: settings?.lastShoperSyncAt || null,
        isRefreshing: isShoperSyncing,
      });
    } catch (error) {
      console.error("Get Shoper sync status error:", error);
      res.status(500).json({ error: "Failed to get Shoper sync status" });
    }
  });

  app.post("/api/shoper/sync/settings", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { shoperAutoRefreshEnabled, shoperRefreshIntervalMinutes } = req.body;

      const currentSettings = await getSyncSettingsFromPostgres();
      const settings = await createOrUpdateSyncSettingsInPostgres({
        autoRefreshEnabled: currentSettings?.autoRefreshEnabled ?? true,
        refreshIntervalMinutes: currentSettings?.refreshIntervalMinutes ?? "3",
        lastSyncAt: currentSettings?.lastSyncAt || null,
        shoperAutoRefreshEnabled,
        shoperRefreshIntervalMinutes,
        lastShoperSyncAt: currentSettings?.lastShoperSyncAt || null,
      });

      if (shoperAutoRefreshEnabled) {
        const interval = parseInt(shoperRefreshIntervalMinutes);
        startShoperSyncScheduler(interval);
      } else {
        stopShoperSyncScheduler();
      }

      res.json(settings);
    } catch (error) {
      console.error("Save Shoper sync settings error:", error);
      res.status(500).json({ error: "Failed to save Shoper sync settings" });
    }
  });

  app.get("/api/shoper/sync/settings", requirePermission('manage_settings'), async (req, res) => {
    try {
      const settings = await getSyncSettingsFromPostgres();
      if (!settings) {
        return res.json({
          shoperAutoRefreshEnabled: true,
          shoperRefreshIntervalMinutes: "5",
        });
      }
      res.json({
        shoperAutoRefreshEnabled: settings.shoperAutoRefreshEnabled,
        shoperRefreshIntervalMinutes: settings.shoperRefreshIntervalMinutes,
        lastShoperSyncAt: settings.lastShoperSyncAt,
      });
    } catch (error) {
      console.error("Get Shoper sync settings error:", error);
      res.status(500).json({ error: "Failed to get Shoper sync settings" });
    }
  });

  // Recent Updates - pokazuje ostatnio zmodyfikowane zamówienia
  app.get("/api/recent-updates", async (req, res) => {
    try {
      // Pobierz zamówienia zmodyfikowane w ciągu ostatnich 24h
      const recentOrders = await getRecentlyUpdatedOrders(1440); // 24 godziny
      res.json(recentOrders);
    } catch (error) {
      console.error("Get recent updates error:", error);
      res.status(500).json({ error: "Failed to fetch recent updates" });
    }
  });

  // Order Changes - historia zmian w zamówieniach
  app.get("/api/order-changes", async (req, res) => {
    try {
      const { orderCode, limit = 100 } = req.query;
      
      let query = `
        SELECT 
          id, order_id, order_number, order_code, source, change_type,
          field_changed, old_value, new_value, detected_at
        FROM public.order_changes
      `;
      
      const params: any[] = [];
      if (orderCode) {
        query += ` WHERE order_code = $1`;
        params.push(orderCode);
        query += ` ORDER BY detected_at DESC LIMIT $2`;
        params.push(limit);
      } else {
        query += ` ORDER BY detected_at DESC LIMIT $1`;
        params.push(limit);
      }
      
      const result = await pool.query(query, params);
      res.json(result.rows);
    } catch (error) {
      console.error("Get order changes error:", error);
      res.status(500).json({ error: "Failed to fetch order changes" });
    }
  });

  app.get("/api/products/images/:filename", async (req, res) => {
    try {
      const { filename } = req.params;
      const imagePath = path.join(process.cwd(), "attached_assets", "products", filename);
      
      const fileExists = await fs.access(imagePath).then(() => true).catch(() => false);
      
      if (!fileExists) {
        return res.status(404).json({ error: "Image not found" });
      }
      
      const ext = path.extname(filename).toLowerCase();
      const contentType = ext === '.jpg' || ext === '.jpeg' ? 'image/jpeg' : 
                         ext === '.png' ? 'image/png' : 
                         ext === '.webp' ? 'image/webp' : 'image/jpeg';
      
      res.setHeader('Content-Type', contentType);
      res.setHeader('Cache-Control', 'public, max-age=31536000');
      
      const imageBuffer = await fs.readFile(imagePath);
      res.send(imageBuffer);
    } catch (error) {
      console.error("Error serving product image:", error);
      res.status(500).json({ error: "Failed to serve image" });
    }
  });

  // ============================================================================
  // API TOKENS MANAGEMENT (for admins)
  // ============================================================================
  
  // GET /api/api-tokens - List all API tokens (admin only)
  app.get("/api/api-tokens", requireAdmin, async (req, res) => {
    try {
      const result = await pool.query(
        `SELECT 
          id,
          name,
          description,
          token_prefix,
          is_active,
          expires_at,
          last_used_at,
          created_by,
          created_at
        FROM api_tokens
        ORDER BY created_at DESC`
      );

      res.json(result.rows);
    } catch (error) {
      console.error("Get API tokens error:", error);
      res.status(500).json({ error: "Failed to fetch API tokens" });
    }
  });

  // POST /api/api-tokens - Create new API token (admin only)
  app.post("/api/api-tokens", requireAdmin, async (req, res) => {
    try {
      const { name, description, expiresInDays } = req.body;
      
      if (!name) {
        return res.status(400).json({ error: "Token name is required" });
      }

      // Generate token
      const token = generateApiToken();
      const hashedToken = hashToken(token);
      const tokenPrefix = token.substring(0, 8);

      // Calculate expiry date
      let expiresAt = null;
      if (expiresInDays && expiresInDays > 0) {
        expiresAt = new Date();
        expiresAt.setDate(expiresAt.getDate() + parseInt(expiresInDays));
      }

      const userId = (req.user as any)?.id;

      const result = await pool.query(
        `INSERT INTO api_tokens (
          name, 
          description, 
          token, 
          token_prefix, 
          is_active, 
          expires_at, 
          created_by,
          created_at
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
        RETURNING id, name, description, token_prefix, is_active, expires_at, created_at`,
        [name, description, hashedToken, tokenPrefix, true, expiresAt, userId, new Date()]
      );

      // Return token ONLY once during creation
      res.json({
        ...result.rows[0],
        token, // Full token returned only on creation
      });
    } catch (error) {
      console.error("Create API token error:", error);
      res.status(500).json({ error: "Failed to create API token" });
    }
  });

  // PUT /api/api-tokens/:id - Update API token (activate/deactivate)
  app.put("/api/api-tokens/:id", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;
      const { isActive, name, description } = req.body;

      const updates: string[] = [];
      const params: any[] = [];
      let paramIndex = 1;

      if (typeof isActive === 'boolean') {
        updates.push(`is_active = $${paramIndex}`);
        params.push(isActive);
        paramIndex++;
      }

      if (name !== undefined) {
        updates.push(`name = $${paramIndex}`);
        params.push(name);
        paramIndex++;
      }

      if (description !== undefined) {
        updates.push(`description = $${paramIndex}`);
        params.push(description);
        paramIndex++;
      }

      if (updates.length === 0) {
        return res.status(400).json({ error: "No updates provided" });
      }

      params.push(id);
      const result = await pool.query(
        `UPDATE api_tokens 
         SET ${updates.join(', ')}
         WHERE id = $${paramIndex}
         RETURNING id, name, description, token_prefix, is_active, expires_at, last_used_at, created_at`,
        params
      );

      if (result.rows.length === 0) {
        return res.status(404).json({ error: "API token not found" });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Update API token error:", error);
      res.status(500).json({ error: "Failed to update API token" });
    }
  });

  // DELETE /api/api-tokens/:id - Delete API token
  app.delete("/api/api-tokens/:id", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;

      const result = await pool.query(
        'DELETE FROM api_tokens WHERE id = $1 RETURNING id',
        [id]
      );

      if (result.rows.length === 0) {
        return res.status(404).json({ error: "API token not found" });
      }

      res.json({ success: true, message: "API token deleted successfully" });
    } catch (error) {
      console.error("Delete API token error:", error);
      res.status(500).json({ error: "Failed to delete API token" });
    }
  });

  // GET /api/api-tokens/logs - Get API request logs (admin only)
  app.get("/api/api-tokens/logs", requireAdmin, async (req, res) => {
    try {
      const limit = parseInt(req.query.limit as string) || 100;
      const tokenId = req.query.token_id as string;

      let query = `
        SELECT 
          l.id,
          l.token_id,
          l.method,
          l.path,
          l.status_code,
          l.response_time,
          l.ip_address,
          l.user_agent,
          l.created_at,
          t.name as token_name
        FROM api_request_logs l
        LEFT JOIN api_tokens t ON l.token_id = t.id
        ${tokenId ? 'WHERE l.token_id = $1' : ''}
        ORDER BY l.created_at DESC
        LIMIT ${tokenId ? '$2' : '$1'}
      `;

      const params = tokenId ? [tokenId, limit] : [limit];
      const result = await pool.query(query, params);

      res.json(result.rows);
    } catch (error) {
      console.error("Get API logs error:", error);
      res.status(500).json({ error: "Failed to fetch API logs" });
    }
  });

  // ============================================================================
  // WEBHOOKS MANAGEMENT (for admins)
  // ============================================================================
  
  // GET /api/webhooks - List all webhooks (admin only)
  app.get("/api/webhooks", requireAdmin, async (req, res) => {
    try {
      const result = await pool.query(
        `SELECT 
          id,
          name,
          url,
          events,
          is_active,
          retry_attempts,
          last_triggered_at,
          created_by,
          created_at,
          updated_at
        FROM webhook_configs
        ORDER BY created_at DESC`
      );

      res.json(result.rows);
    } catch (error) {
      console.error("Get webhooks error:", error);
      res.status(500).json({ error: "Failed to fetch webhooks" });
    }
  });

  // POST /api/webhooks - Create new webhook (admin only)
  app.post("/api/webhooks", requireAdmin, async (req, res) => {
    try {
      const { name, url, secret, events, retryAttempts } = req.body;
      
      if (!name || !url || !events) {
        return res.status(400).json({ error: "Name, URL, and events are required" });
      }

      const userId = (req.user as any)?.id;
      const webhookSecret = secret || randomBytes(32).toString('hex');

      const result = await pool.query(
        `INSERT INTO webhook_configs (
          name, 
          url, 
          secret, 
          events, 
          is_active, 
          retry_attempts,
          created_by,
          created_at
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
        RETURNING id, name, url, events, is_active, retry_attempts, created_at`,
        [name, url, webhookSecret, JSON.stringify(events), true, retryAttempts || 3, userId, new Date()]
      );

      res.json({
        ...result.rows[0],
        secret: webhookSecret, // Return secret only on creation
      });
    } catch (error) {
      console.error("Create webhook error:", error);
      res.status(500).json({ error: "Failed to create webhook" });
    }
  });

  // PUT /api/webhooks/:id - Update webhook (admin only)
  app.put("/api/webhooks/:id", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;
      const { isActive, name, url, events, retryAttempts } = req.body;

      const updates: string[] = [];
      const params: any[] = [];
      let paramIndex = 1;

      if (typeof isActive === 'boolean') {
        updates.push(`is_active = $${paramIndex}`);
        params.push(isActive);
        paramIndex++;
      }

      if (name !== undefined) {
        updates.push(`name = $${paramIndex}`);
        params.push(name);
        paramIndex++;
      }

      if (url !== undefined) {
        updates.push(`url = $${paramIndex}`);
        params.push(url);
        paramIndex++;
      }

      if (events !== undefined) {
        updates.push(`events = $${paramIndex}`);
        params.push(JSON.stringify(events));
        paramIndex++;
      }

      if (retryAttempts !== undefined) {
        updates.push(`retry_attempts = $${paramIndex}`);
        params.push(retryAttempts);
        paramIndex++;
      }

      if (updates.length === 0) {
        return res.status(400).json({ error: "No updates provided" });
      }

      updates.push(`updated_at = $${paramIndex}`);
      params.push(new Date());
      paramIndex++;

      params.push(id);
      const result = await pool.query(
        `UPDATE webhook_configs 
         SET ${updates.join(', ')}
         WHERE id = $${paramIndex}
         RETURNING id, name, url, events, is_active, retry_attempts, last_triggered_at, created_at, updated_at`,
        params
      );

      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Webhook not found" });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Update webhook error:", error);
      res.status(500).json({ error: "Failed to update webhook" });
    }
  });

  // DELETE /api/webhooks/:id - Delete webhook (admin only)
  app.delete("/api/webhooks/:id", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;

      const result = await pool.query(
        'DELETE FROM webhook_configs WHERE id = $1 RETURNING id',
        [id]
      );

      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Webhook not found" });
      }

      res.json({ success: true, message: "Webhook deleted successfully" });
    } catch (error) {
      console.error("Delete webhook error:", error);
      res.status(500).json({ error: "Failed to delete webhook" });
    }
  });

  // GET /api/webhooks/logs - Get webhook delivery logs (admin only)
  app.get("/api/webhooks/logs", requireAdmin, async (req, res) => {
    try {
      const limit = parseInt(req.query.limit as string) || 100;
      const webhookId = req.query.webhook_id ? parseInt(req.query.webhook_id as string) : undefined;

      const logs = await getWebhookLogs(webhookId, limit);
      res.json(logs);
    } catch (error) {
      console.error("Get webhook logs error:", error);
      res.status(500).json({ error: "Failed to fetch webhook logs" });
    }
  });

  // POST /api/webhooks/:id/test - Send test webhook (admin only)
  app.post("/api/webhooks/:id/test", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;
      const { sendWebhook } = await import('./webhooks.js');

      // Get webhook config to check events
      const webhookResult = await pool.query(
        'SELECT * FROM webhook_configs WHERE id = $1',
        [id]
      );

      if (webhookResult.rows.length === 0) {
        return res.status(404).json({ error: "Webhook not found" });
      }

      const webhook = webhookResult.rows[0];
      const events = webhook.events || [];

      // Select first configured event, or default to order.created
      const testEvent = events[0] || 'order.created';

      // Create test payload based on event type
      let testPayload: any = {
        order_code: 'TEST-001',
        order_number: '1',
        source: 'ALLEGRO',
        source_order_id: 'test-order-id-123',
        buyer_email: 'test@example.com',
        buyer_first_name: 'Jan',
        buyer_last_name: 'Testowy',
        total_amount: '999.99',
        payment_amount: '999.99',
        currency: 'PLN',
        payment_status: 'PENDING',
        payment_type: 'ONLINE',
        delivery_method: 'DPD Kurier',
        tracking_numbers: [],
        created_at: new Date().toISOString(),
        items: [
          {
            product_name: 'Testowy Produkt',
            quantity: 1,
            unit_price: '999.99',
            total_price: '999.99'
          }
        ]
      };

      // Adjust payload based on event type
      if (testEvent === 'order.paid') {
        testPayload.payment_status = 'PAID';
        testPayload.payment_date = new Date().toISOString();
      } else if (testEvent === 'order.shipped') {
        testPayload.tracking_numbers = ['TEST123456789'];
        testPayload.shipments = [{
          carrier: 'DPD',
          tracking_number: 'TEST123456789',
          shipped_at: new Date().toISOString()
        }];
      }

      // Send test webhook
      const success = await sendWebhook(parseInt(id), testEvent, testPayload);

      if (success) {
        res.json({ 
          success: true, 
          message: `Test webhook wysłany pomyślnie (event: ${testEvent})` 
        });
      } else {
        res.status(500).json({ 
          success: false, 
          error: "Nie udało się wysłać testowego webhooka" 
        });
      }
    } catch (error) {
      console.error("Send test webhook error:", error);
      res.status(500).json({ error: "Failed to send test webhook" });
    }
  });

  // ============================================================================
  // ODOO INTEGRATION ENDPOINTS
  // ============================================================================

  // Test Odoo connection
  app.post("/api/odoo/test-connection", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { createOdooClient } = await import('./odoo-client.js');
      const client = await createOdooClient();
      
      if (!client) {
        return res.status(400).json({ 
          success: false, 
          error: "Odoo credentials not configured" 
        });
      }

      const result = await client.testConnection();
      res.json(result);
    } catch (error) {
      console.error("Test Odoo connection error:", error);
      res.status(500).json({ 
        success: false, 
        error: error instanceof Error ? error.message : "Unknown error" 
      });
    }
  });

  // Get Odoo config
  app.get("/api/odoo/config", requirePermission('manage_settings'), async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT * FROM odoo_config WHERE is_active = true ORDER BY id DESC LIMIT 1
      `);

      if (result.rows.length === 0) {
        return res.json({ 
          configured: false,
          url: process.env.ODOO_URL || null,
          database: process.env.ODOO_DATABASE || null,
          username: process.env.ODOO_USERNAME || null
        });
      }

      res.json({ 
        configured: true,
        ...result.rows[0]
      });
    } catch (error) {
      console.error("Get Odoo config error:", error);
      res.status(500).json({ error: "Failed to get Odoo config" });
    }
  });

  // Update Odoo config
  app.post("/api/odoo/config", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { syncIntervalMinutes, isActive, autoConfirmOrders } = req.body;

      const existingConfig = await pool.query(`
        SELECT * FROM odoo_config WHERE is_active = true ORDER BY id DESC LIMIT 1
      `);

      if (existingConfig.rows.length > 0) {
        await pool.query(`
          UPDATE odoo_config
          SET 
            sync_interval_minutes = $1,
            is_active = $2,
            auto_confirm_orders = $3,
            updated_at = NOW()
          WHERE id = $4
        `, [syncIntervalMinutes || 3, isActive !== false, autoConfirmOrders || false, existingConfig.rows[0].id]);
      } else {
        await pool.query(`
          INSERT INTO odoo_config (
            url, database, username, sync_interval_minutes, is_active, auto_confirm_orders
          ) VALUES ($1, $2, $3, $4, $5, $6)
        `, [
          process.env.ODOO_URL,
          process.env.ODOO_DATABASE,
          process.env.ODOO_USERNAME,
          syncIntervalMinutes || 3,
          isActive !== false,
          autoConfirmOrders || false
        ]);
      }

      if (isActive !== false) {
        const interval = syncIntervalMinutes || 3;
        startOdooSyncScheduler(interval);
        console.log(`✅ Odoo sync scheduler restarted (${interval} min interval)`);
      } else {
        stopOdooSyncScheduler();
        console.log('⏸️  Odoo sync scheduler stopped');
      }

      res.json({ success: true });
    } catch (error) {
      console.error("Update Odoo config error:", error);
      res.status(500).json({ error: "Failed to update Odoo config" });
    }
  });

  // Get Odoo sync status
  app.get("/api/odoo/sync/status", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { getOdooSyncStatus } = await import('./odoo-sync.js');
      const status = await getOdooSyncStatus();
      res.json(status);
    } catch (error) {
      console.error("Get Odoo sync status error:", error);
      res.status(500).json({ error: "Failed to get sync status" });
    }
  });

  // Manual Odoo sync
  app.post("/api/odoo/sync/manual", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { processSyncQueue } = await import('./odoo-sync.js');
      const result = await processSyncQueue();
      res.json(result);
    } catch (error) {
      console.error("Manual Odoo sync error:", error);
      res.status(500).json({ error: "Failed to run manual sync" });
    }
  });

  // Get Odoo sync logs
  app.get("/api/odoo/sync/logs", requirePermission('manage_settings'), async (req, res) => {
    try {
      const limit = parseInt(req.query.limit as string) || 100;
      const { getOdooSyncLogs } = await import('./odoo-sync.js');
      const logs = await getOdooSyncLogs(limit);
      res.json(logs);
    } catch (error) {
      console.error("Get Odoo sync logs error:", error);
      res.status(500).json({ error: "Failed to get sync logs" });
    }
  });

  // Retry failed Odoo sync
  app.post("/api/odoo/sync/retry/:id", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { id } = req.params;
      const { retryFailedSync, processSyncQueue } = await import('./odoo-sync.js');
      
      await retryFailedSync(parseInt(id));
      await processSyncQueue();
      
      res.json({ success: true });
    } catch (error) {
      console.error("Retry Odoo sync error:", error);
      res.status(500).json({ error: "Failed to retry sync" });
    }
  });

  // ============================================================================
  // EXTERNAL API ENDPOINTS (for Odoo and other integrations)
  // ============================================================================
  
  // GET /api/external/orders - List all orders with pagination and filtering
  app.get("/api/external/orders", requireApiToken, logApiRequest, async (req, res) => {
    try {
      const page = parseInt(req.query.page as string) || 1;
      const limit = parseInt(req.query.limit as string) || 50;
      const offset = (page - 1) * limit;
      
      // Optional filters
      const source = req.query.source as string; // 'allegro' or 'shoper'
      const paymentStatus = req.query.payment_status as string;
      const dateFrom = req.query.date_from as string;
      const dateTo = req.query.date_to as string;

      // Build WHERE clause
      let whereClauses: string[] = [];
      let params: any[] = [];
      let paramIndex = 1;

      if (source) {
        whereClauses.push(`source = $${paramIndex}`);
        params.push(source);
        paramIndex++;
      }

      if (paymentStatus) {
        whereClauses.push(`payment_status = $${paramIndex}`);
        params.push(paymentStatus);
        paramIndex++;
      }

      if (dateFrom) {
        whereClauses.push(`order_date >= $${paramIndex}`);
        params.push(dateFrom);
        paramIndex++;
      }

      if (dateTo) {
        whereClauses.push(`order_date <= $${paramIndex}`);
        params.push(dateTo);
        paramIndex++;
      }

      const whereClause = whereClauses.length > 0 ? `WHERE ${whereClauses.join(' AND ')}` : '';

      // Get total count
      const countResult = await pool.query(
        `SELECT COUNT(*) as total FROM commerce.orders ${whereClause}`,
        params
      );
      const total = parseInt(countResult.rows[0].total);

      // Get orders with pagination
      const ordersResult = await pool.query(
        `SELECT 
          order_number,
          order_code,
          source_order_id,
          source,
          order_date,
          buyer_first_name,
          buyer_last_name,
          buyer_email,
          buyer_phone,
          buyer_login,
          delivery_method,
          delivery_amount,
          delivery_address,
          invoice_required,
          buyer_company_name,
          tax_id,
          billing_address,
          total_to_pay_amount as total_amount,
          total_to_pay_currency as currency,
          payment_status,
          payment_type,
          payment_amount,
          tracking_numbers,
          buyer_notes,
          has_returns,
          refund_amount,
          refund_date,
          created_at,
          updated_at
        FROM commerce.orders 
        ${whereClause}
        ORDER BY order_date DESC
        LIMIT $${paramIndex} OFFSET $${paramIndex + 1}`,
        [...params, limit, offset]
      );

      res.json({
        success: true,
        data: ordersResult.rows,
        pagination: {
          page,
          limit,
          total,
          totalPages: Math.ceil(total / limit),
        },
      });
    } catch (error) {
      console.error("External API - Get orders error:", error);
      res.status(500).json({ 
        success: false, 
        error: "Failed to fetch orders" 
      });
    }
  });

  // GET /api/external/orders/:id - Get order details by order_code (AL-123 or SH-456)
  app.get("/api/external/orders/:orderCode", requireApiToken, logApiRequest, async (req, res) => {
    try {
      const { orderCode } = req.params;

      const result = await pool.query(
        `SELECT 
          order_number,
          order_code,
          source_order_id,
          source,
          order_date,
          buyer_first_name,
          buyer_last_name,
          buyer_email,
          buyer_phone,
          buyer_login,
          delivery_method,
          delivery_amount,
          delivery_address,
          invoice_required,
          buyer_company_name,
          tax_id,
          billing_address,
          total_to_pay_amount as total_amount,
          total_to_pay_currency as currency,
          payment_status,
          payment_type,
          payment_amount,
          tracking_numbers,
          buyer_notes,
          shipments,
          has_returns,
          refund_amount,
          refund_date,
          created_at,
          updated_at
        FROM commerce.orders 
        WHERE order_code = $1`,
        [orderCode]
      );

      if (result.rows.length === 0) {
        return res.status(404).json({ 
          success: false, 
          error: "Order not found" 
        });
      }

      res.json({
        success: true,
        data: result.rows[0],
      });
    } catch (error) {
      console.error("External API - Get order details error:", error);
      res.status(500).json({ 
        success: false, 
        error: "Failed to fetch order details" 
      });
    }
  });

  // GET /api/external/products - Get unique products list
  app.get("/api/external/products", requireApiToken, logApiRequest, async (req, res) => {
    try {
      const result = await pool.query(
        `SELECT 
          id,
          product_id,
          external_id,
          name,
          description,
          image_url,
          source,
          category,
          created_at,
          updated_at
        FROM catalog.products
        ORDER BY name ASC`
      );

      res.json({
        success: true,
        data: result.rows,
        total: result.rows.length,
      });
    } catch (error) {
      console.error("External API - Get products error:", error);
      res.status(500).json({ 
        success: false, 
        error: "Failed to fetch products" 
      });
    }
  });

  const connection = await getAllegroConnectionFromPostgres();
  const settings = await getSyncSettingsFromPostgres();
  
  console.log('🔍 Allegro auto-sync check:', {
    hasConnection: !!connection,
    isActive: connection?.isActive,
    autoRefreshEnabled: settings?.autoRefreshEnabled,
    interval: settings?.refreshIntervalMinutes
  });
  
  if (connection?.isActive && settings?.autoRefreshEnabled) {
    const interval = parseInt(settings.refreshIntervalMinutes || "3");
    console.log(`✅ Allegro auto-sync initialized (${interval} min interval)`);
    startSyncScheduler(interval);
  } else {
    console.log('⚠️  Allegro auto-sync NOT started:', {
      reason: !connection ? 'No connection' : 
              !connection.isActive ? 'Connection not active' :
              !settings?.autoRefreshEnabled ? 'Auto-refresh disabled' : 'Unknown'
    });
  }

  if (settings?.shoperAutoRefreshEnabled) {
    const interval = parseInt(settings.shoperRefreshIntervalMinutes || "5");
    startShoperSyncScheduler(interval);
    console.log("✅ Shoper auto-sync initialized");
  }

  // Initialize Odoo sync scheduler
  try {
    const odooConfigResult = await pool.query(`
      SELECT * FROM odoo_config WHERE is_active = true ORDER BY id DESC LIMIT 1
    `);
    
    if (odooConfigResult.rows.length > 0) {
      const odooConfig = odooConfigResult.rows[0];
      const interval = odooConfig.sync_interval_minutes || 3;
      startOdooSyncScheduler(interval);
      console.log(`✅ Odoo sync scheduler initialized (${interval} min interval)`);
    }
  } catch (error) {
    console.log('⚠️  Odoo config not found, scheduler not started');
  }

  // Recent updates checker removed - Allegro now uses incremental sync

  const httpServer = createServer(app);
  return httpServer;
}
