import { Pool } from 'pg';
import { triggerOrderWebhooks } from './webhooks.js';

export const pool = new Pool({
  connectionString: process.env.DATABASE_URL,
  connectionTimeoutMillis: 5000,
  query_timeout: 5000,
});

export async function saveOrderToPostgres(allegroOrder: any, shipments: any[] = [], customerReturns: any[] = [], accessToken?: string): Promise<{ isNew: boolean }> {
  const client = await pool.connect();
  
  try {
    await client.query('BEGIN');

    const buyer = allegroOrder.buyer || {};
    const delivery = allegroOrder.delivery || {};
    const payment = allegroOrder.payment || {};
    const lineItems = allegroOrder.lineItems || [];

    const existsResult = await client.query(
      'SELECT id FROM allegro.orders WHERE order_id = $1',
      [allegroOrder.id]
    );
    const isNew = existsResult.rows.length === 0;

    const invoice = allegroOrder.invoice || {};
    const invoiceAddress = invoice.address || {};

    // Pobierz zwroty płatności z API /payments/refunds
    let refundAmount = 0;
    let refundDate: string | null = null;
    const refundReconciliation: any[] = [];
    
    if (accessToken) {
      const { fetchPaymentRefunds } = await import('./allegro-api.js');
      const refunds = await fetchPaymentRefunds(allegroOrder.id, accessToken);
      
      for (const refund of refunds) {
        const amount = parseFloat(refund.totalValue?.amount || '0');
        refundAmount += amount;
        refundReconciliation.push(refund);
        
        // Zapisz datę ostatniego zwrotu
        if (refund.createdAt) {
          if (!refundDate || new Date(refund.createdAt) > new Date(refundDate)) {
            refundDate = refund.createdAt;
          }
        }
      }
    }

    // Sprawdź czy są zwroty produktów (type: QUANTITY) w refund_reconciliation
    const hasReturnsInReconciliation = refundReconciliation.some(refund => 
      refund.lineItems && refund.lineItems.some((item: any) => item.type === 'QUANTITY')
    );
    const hasReturns = hasReturnsInReconciliation;

    await client.query(`
      INSERT INTO allegro.orders (
        order_id, seller_id, seller_login, order_date, seller_status,
        marketplace, buyer_id, buyer_login, buyer_email, buyer_company,
        buyer_name, buyer_phone, buyer_address, buyer_zip, buyer_city,
        buyer_country_code, payment_id, payment_status, payment_provider,
        allegro_pay, payment_amount, payment_currency, payment_type,
        invoice_required, delivery_method, delivery_amount, delivery_currency,
        total_to_pay_amount, total_to_pay_currency, buyer_notes, smart,
        tracking_numbers, payment_last_date, shipments, 
        billing_address,
        refund_amount, refund_date, refund_reconciliation, has_returns,
        imported_at
      ) VALUES (
        $1, $2, $3, $4, $5, $6, $7, $8, $9, $10,
        $11, $12, $13, $14, $15, $16, $17, $18, $19,
        $20, $21, $22, $23, $24, $25, $26, $27, $28, $29, $30, $31, $32, $33, $34,
        $35, $36, $37, $38, $39, NOW()
      )
      ON CONFLICT (order_id) DO UPDATE SET
        seller_status = EXCLUDED.seller_status,
        payment_status = EXCLUDED.payment_status,
        payment_amount = EXCLUDED.payment_amount,
        payment_type = EXCLUDED.payment_type,
        invoice_required = EXCLUDED.invoice_required,
        payment_last_date = EXCLUDED.payment_last_date,
        tracking_numbers = EXCLUDED.tracking_numbers,
        shipments = EXCLUDED.shipments,
        buyer_notes = EXCLUDED.buyer_notes,
        delivery_amount = EXCLUDED.delivery_amount,
        buyer_address = EXCLUDED.buyer_address,
        buyer_zip = EXCLUDED.buyer_zip,
        buyer_city = EXCLUDED.buyer_city,
        billing_address = EXCLUDED.billing_address,
        refund_amount = EXCLUDED.refund_amount,
        refund_date = EXCLUDED.refund_date,
        refund_reconciliation = EXCLUDED.refund_reconciliation,
        has_returns = EXCLUDED.has_returns,
        imported_at = EXCLUDED.imported_at
    `, [
      allegroOrder.id,
      allegroOrder.seller?.id,
      allegroOrder.seller?.login,
      allegroOrder.boughtAt || allegroOrder.updatedAt,
      allegroOrder.status,
      'allegro-pl',
      buyer.id,
      buyer.login,
      buyer.email,
      buyer.companyName,
      buyer.firstName && buyer.lastName ? `${buyer.firstName} ${buyer.lastName}` : buyer.login,
      buyer.phoneNumber,
      delivery.address?.street,
      delivery.address?.zipCode,
      delivery.address?.city,
      delivery.address?.countryCode,
      payment.id,
      payment.paidAmount?.amount && parseFloat(payment.paidAmount.amount) > 0 ? 'PAID' : 'UNPAID',
      payment.provider,
      payment.type === 'ONLINE' ? true : false,
      payment.paidAmount?.amount,
      payment.paidAmount?.currency,
      payment.type || null,
      invoice.required === true,
      delivery.method?.name,
      delivery.cost?.amount,
      delivery.cost?.currency,
      allegroOrder.summary?.totalToPay?.amount,
      allegroOrder.summary?.totalToPay?.currency,
      allegroOrder.messageToSeller,
      delivery.smart === true,
      delivery.shipments && delivery.shipments.length > 0 
        ? delivery.shipments.map((s: any) => s.carrierId).join('|') 
        : null,
      payment.finishedAt,
      JSON.stringify(shipments),
      invoiceAddress.street || null,
      refundAmount || 0,
      refundDate || null,
      JSON.stringify(refundReconciliation),
      hasReturns
    ]);

    // Najpierw wstaw wszystkie order_items
    for (const item of lineItems) {
      const imageUrl = 
        item.offer?.headline?.imageUrl || 
        item.offer?.primaryImage?.url || 
        item.offer?.images?.[0]?.url || 
        item.offer?.image?.url || 
        null;
      
      const externalId = item.offer?.external?.id;
      let productId = null;
      
      if (externalId) {
        const productResult = await client.query(`
          INSERT INTO allegro.products (
            product_id, external_id, name, description, image_url, updated_at
          ) VALUES (
            $1, $2, $3, $4, $5, NOW()
          )
          ON CONFLICT (external_id) DO UPDATE SET
            name = EXCLUDED.name,
            description = EXCLUDED.description,
            image_url = COALESCE(EXCLUDED.image_url, allegro.products.image_url),
            updated_at = NOW()
          RETURNING id
        `, [
          item.offer?.id,
          externalId,
          item.offer?.name,
          item.offer?.description || null,
          imageUrl
        ]);
        
        productId = productResult.rows[0]?.id;
      }
      
      await client.query(`
        INSERT INTO allegro.order_items (
          order_id, line_item_id, offer_id, offer_external_id, name,
          quantity, price, currency, tax_rate, tax_subject,
          sending_country, returns_quantity, image_url, product_id, imported_at
        ) VALUES (
          $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, NOW()
        )
        ON CONFLICT (line_item_id) DO UPDATE SET
          price = EXCLUDED.price,
          quantity = EXCLUDED.quantity,
          returns_quantity = EXCLUDED.returns_quantity,
          image_url = EXCLUDED.image_url,
          product_id = EXCLUDED.product_id,
          imported_at = EXCLUDED.imported_at
      `, [
        allegroOrder.id,
        item.id,
        item.offer?.id,
        externalId,
        item.offer?.name,
        item.quantity,
        item.price?.amount,
        item.price?.currency,
        item.reconciliation?.tax?.percentage,
        item.reconciliation?.tax?.subjectType,
        'PL',
        0,
        imageUrl,
        productId
      ]);
    }

    // Aktualizuj returns_quantity na podstawie refund_reconciliation (bardziej stabilne dane)
    // ID w refund_reconciliation.lineItems to checkoutFormLineItemId (line_item_id)
    if (refundReconciliation && Array.isArray(refundReconciliation)) {
      for (const refund of refundReconciliation) {
        const refundLineItems = refund.lineItems || [];
        for (const refundLineItem of refundLineItems) {
          if (refundLineItem.id && refundLineItem.quantity) {
            // ID w refund_reconciliation to line_item_id (checkoutFormLineItemId)
            await client.query(`
              UPDATE allegro.order_items
              SET returns_quantity = $1
              WHERE order_id = $2
                AND line_item_id = $3
            `, [refundLineItem.quantity, allegroOrder.id, refundLineItem.id]);
          }
        }
      }
    }

    await client.query('COMMIT');
    console.log(`✅ Saved order ${allegroOrder.id} to PostgreSQL (${isNew ? 'NEW' : 'UPDATED'})`);
    
    return { isNew };
  } catch (error) {
    await client.query('ROLLBACK');
    console.error(`❌ Error saving order to PostgreSQL:`, error);
    throw error;
  } finally {
    client.release();
  }
}

export async function saveShoperOrderToPostgres(shoperOrder: any, shipments: any[] = []): Promise<{ isNew: boolean }> {
  const client = await pool.connect();
  
  try {
    await client.query('BEGIN');

    const orderId = shoperOrder.order_id || shoperOrder.id;
    const billingAddress = shoperOrder.billing_address || {};
    const shippingAddress = shoperOrder.shipping_address || {};
    const products = shoperOrder.products_data || [];

    const existsResult = await client.query(
      'SELECT id FROM shoper.orders WHERE shoper_order_id = $1',
      [orderId]
    );
    const isNew = existsResult.rows.length === 0;

    const buyerLogin = billingAddress.firstname && billingAddress.lastname 
      ? `${billingAddress.firstname} ${billingAddress.lastname}` 
      : shoperOrder.user_login || 'Unknown';

    // Konwertuj daty z czasu polskiego (Europe/Warsaw) na UTC
    const convertPolandTimeToUTC = (dateStr: string | null) => {
      if (!dateStr || dateStr === '0000-00-00 00:00:00') return null;
      // Daty z Shoper API są w czasie polskim (UTC+2 w lecie, UTC+1 w zimie)
      // Aby przekonwertować na UTC, odejmujemy 2 godziny (dla czasu letniego)
      const date = new Date(dateStr);
      date.setHours(date.getHours() - 2);
      return date.toISOString();
    };

    const orderDateUTC = convertPolandTimeToUTC(shoperOrder.date || shoperOrder.confirm_date);
    const paidDateUTC = convertPolandTimeToUTC(shoperOrder.paid_date);

    await client.query(`
      INSERT INTO shoper.orders (
        id, shoper_order_id, buyer_login, buyer_email, total_amount, currency,
        payment_status, fulfillment_status, items_count, order_date, 
        payment_date, shipments, raw_data, created_at, updated_at
      ) VALUES (
        gen_random_uuid(), $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, NOW(), NOW()
      )
      ON CONFLICT (shoper_order_id) DO UPDATE SET
        payment_status = EXCLUDED.payment_status,
        fulfillment_status = EXCLUDED.fulfillment_status,
        total_amount = EXCLUDED.total_amount,
        payment_date = EXCLUDED.payment_date,
        shipments = EXCLUDED.shipments,
        raw_data = EXCLUDED.raw_data,
        updated_at = NOW()
    `, [
      orderId,
      buyerLogin,
      billingAddress.email || shoperOrder.email,
      shoperOrder.sum,
      shoperOrder.currency || 'PLN',
      shoperOrder.paid_date ? 'PAID' : 'PENDING',
      shoperOrder.status_id || 'NEW',
      products.length.toString(),
      orderDateUTC,
      paidDateUTC,
      JSON.stringify(shipments),
      JSON.stringify(shoperOrder)
    ]);

    for (const product of products) {
      // Determine payment type: if paid, it's ONLINE; if unpaid and method includes POBRANIE, it's COD
      const isPaid = shoperOrder.paid_date || parseFloat(shoperOrder.paid || '0') > 0;
      const paymentType = (!isPaid && shoperOrder.payment_method_name?.toUpperCase().includes('POBRANIE'))
        ? 'CASH_ON_DELIVERY' 
        : 'ONLINE';
      
      await client.query(`
        INSERT INTO shoper.order_items (
          id, order_id, shoper_product_id, product_name, quantity,
          unit_price, total_price, image_url, payment_type, invoice_required,
          raw_data, created_at
        ) VALUES (
          gen_random_uuid(), 
          (SELECT id FROM shoper.orders WHERE shoper_order_id = $1),
          $2, $3, $4, $5, $6, $7, $8, $9, $10, NOW()
        )
        ON CONFLICT (order_id, shoper_product_id) DO UPDATE SET
          product_name = EXCLUDED.product_name,
          quantity = EXCLUDED.quantity,
          unit_price = EXCLUDED.unit_price,
          total_price = EXCLUDED.total_price,
          image_url = EXCLUDED.image_url,
          payment_type = EXCLUDED.payment_type,
          invoice_required = EXCLUDED.invoice_required,
          raw_data = EXCLUDED.raw_data
      `, [
        orderId,
        product.product_id || product.id,
        product.name || product.translation?.name,
        product.quantity || 1,
        product.price || 0,
        (product.price || 0) * (product.quantity || 1),
        product.main_image || null,
        paymentType,
        shoperOrder.invoice === '1' || shoperOrder.invoice === true,
        JSON.stringify(product)
      ]);
    }

    await client.query('COMMIT');
    console.log(`✅ Saved Shoper order ${orderId} to shoper.orders (${isNew ? 'NEW' : 'UPDATED'})`);
    
    return { isNew };
  } catch (error) {
    await client.query('ROLLBACK');
    console.error(`❌ Error saving Shoper order to shoper.orders:`, error);
    throw error;
  } finally {
    client.release();
  }
}

function buildOrderFromRow(orderRow: any, items: any[]) {
  const rawPayload = orderRow.raw_payload || {};
  const billingAddress = orderRow.billing_address || {};
  const deliveryAddress = orderRow.delivery_address || {};
  
  return {
    id: orderRow.id,
    allegroOrderId: orderRow.source_order_id,
    orderCode: orderRow.order_code,
    orderNumber: orderRow.order_number,
    source: orderRow.source,
    hasReturns: orderRow.has_returns || false,
    boughtAt: orderRow.order_date,
    updatedAt: orderRow.updated_at,
    status: orderRow.status,
    marketplace: rawPayload.marketplace || (orderRow.source === 'ALLEGRO' ? 'allegro-pl' : null),
    buyer: {
      id: rawPayload.buyer_id,
      login: orderRow.buyer_login,
      email: orderRow.buyer_email,
      companyName: orderRow.buyer_company,
      firstName: orderRow.buyer_first_name,
      lastName: orderRow.buyer_last_name,
      phoneNumber: orderRow.buyer_phone
    },
    payment: {
      id: orderRow.payment_id,
      type: orderRow.payment_type || null,
      provider: orderRow.payment_provider,
      paidAmount: {
        amount: orderRow.payment_amount,
        currency: orderRow.payment_currency
      },
      finishedAt: orderRow.payment_date || rawPayload.payment_last_date,
      refundAmount: orderRow.refund_amount || 0,
      refundDate: orderRow.refund_date,
      refundReconciliation: orderRow.refund_reconciliation || []
    },
    invoice: {
      required: orderRow.invoice_required === true,
      address: billingAddress
    },
    billingAddress: {
      companyName: billingAddress.companyName || orderRow.buyer_company,
      firstName: billingAddress.firstName || orderRow.buyer_first_name,
      lastName: billingAddress.lastName || orderRow.buyer_last_name,
      street: billingAddress.street || orderRow.buyer_address,
      zipCode: billingAddress.zipCode || orderRow.buyer_zip,
      city: billingAddress.city || orderRow.buyer_city,
      countryCode: billingAddress.countryCode || orderRow.buyer_country_code,
      phoneNumber: billingAddress.phoneNumber || orderRow.buyer_phone,
      email: billingAddress.email || orderRow.buyer_email,
      taxId: orderRow.tax_id
    },
    delivery: {
      method: {
        name: orderRow.delivery_method
      },
      address: deliveryAddress.street ? {
        firstName: deliveryAddress.firstName,
        lastName: deliveryAddress.lastName,
        companyName: deliveryAddress.companyName,
        street: deliveryAddress.street,
        zipCode: deliveryAddress.zipCode,
        city: deliveryAddress.city,
        countryCode: deliveryAddress.countryCode,
        phoneNumber: deliveryAddress.phoneNumber
      } : {
        street: orderRow.buyer_address,
        zipCode: orderRow.buyer_zip,
        city: orderRow.buyer_city,
        countryCode: orderRow.buyer_country_code
      },
      cost: {
        amount: orderRow.delivery_amount,
        currency: orderRow.delivery_currency
      },
      smart: rawPayload.smart,
      shipments: orderRow.shipments || []
    },
    summary: {
      totalToPay: {
        amount: orderRow.total_to_pay_amount,
        currency: orderRow.total_to_pay_currency
      }
    },
    messageToSeller: orderRow.buyer_notes,
    lineItems: items.map((item: any) => ({
      id: item.id,
      offer: {
        id: item.offer_id,
        name: item.product_name || item.name,
        external: {
          id: item.offer_external_id
        }
      },
      quantity: item.quantity,
      returnsQuantity: item.returns_quantity || 0,
      price: {
        amount: item.price,
        currency: item.currency
      },
      reconciliation: {
        quantity: item.quantity,
        tax: {
          percentage: item.tax_rate,
          subjectType: item.tax_subject
        }
      },
      imageUrl: item.product_image_url || item.image_url,
      description: item.product_description
    })),
    rawPayload: rawPayload
  };
}

export async function getOrdersFromPostgres(
  sortBy: string = 'order_date',
  sortOrder: 'ASC' | 'DESC' = 'DESC',
  limit: number = 50,
  offset: number = 0,
  search?: string,
  sourceFilter?: string,
  dateRangeType?: string,
  customDateFrom?: string,
  customDateTo?: string
) {
  const startTime = Date.now();
  const client = await pool.connect();
  
  try {
    // Whitelist dozwolonych kolumn do sortowania
    const allowedSortColumns = [
      'order_number',
      'order_date',
      'total_to_pay_amount',
      'buyer_login',
      'buyer_email',
      'status',
      'payment_status',
      'source',
      'created_at'
    ];
    
    const sortColumn = allowedSortColumns.includes(sortBy) ? sortBy : 'order_date';
    const sortDirection = sortOrder === 'ASC' ? 'ASC' : 'DESC';
    
    // Buduj WHERE clause dla wyszukiwania i filtrów
    const conditions: string[] = [];
    let queryParams: any[] = [];
    let paramIndex = 1;
    
    if (search && search.trim()) {
      const searchTerm = `%${search.trim().toLowerCase()}%`;
      conditions.push(`(
        LOWER(order_code) LIKE $${paramIndex} OR
        CAST(order_number AS TEXT) LIKE $${paramIndex} OR
        LOWER(buyer_first_name) LIKE $${paramIndex} OR
        LOWER(buyer_last_name) LIKE $${paramIndex} OR
        LOWER(CONCAT(buyer_first_name, ' ', buyer_last_name)) LIKE $${paramIndex} OR
        LOWER(buyer_email) LIKE $${paramIndex}
      )`);
      queryParams.push(searchTerm);
      paramIndex++;
    }
    
    // Filtr źródła
    if (sourceFilter && sourceFilter !== 'all') {
      conditions.push(`source = $${paramIndex}`);
      queryParams.push(sourceFilter);
      paramIndex++;
    }
    
    // Filtr zakresu dat
    if (dateRangeType && dateRangeType !== 'all') {
      const now = new Date();
      let startDate: Date | null = null;
      let endDate: Date | null = null;

      if (dateRangeType === 'today') {
        startDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 0, 0, 0);
        endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
      } else if (dateRangeType === 'yesterday') {
        const yesterday = new Date(now);
        yesterday.setDate(yesterday.getDate() - 1);
        startDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 0, 0, 0);
        endDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 23, 59, 59);
      } else if (dateRangeType === '7days') {
        startDate = new Date(now);
        startDate.setDate(startDate.getDate() - 6);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
      } else if (dateRangeType === '30days') {
        startDate = new Date(now);
        startDate.setDate(startDate.getDate() - 29);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
      } else if (dateRangeType === 'custom' && customDateFrom && customDateTo) {
        startDate = new Date(customDateFrom);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(customDateTo);
        endDate.setHours(23, 59, 59, 999);
      }

      if (startDate && endDate) {
        conditions.push(`order_date >= $${paramIndex}`);
        queryParams.push(startDate.toISOString());
        paramIndex++;
        conditions.push(`order_date <= $${paramIndex}`);
        queryParams.push(endDate.toISOString());
        paramIndex++;
      }
    }
    
    const whereClause = conditions.length > 0 ? `WHERE ${conditions.join(' AND ')}` : '';
    queryParams.push(limit, offset);
    
    // Pobierz zamówienia z paginacją i wyszukiwaniem
    const t1 = Date.now();
    const ordersResult = await client.query(`
      SELECT * FROM commerce.orders 
      ${whereClause}
      ORDER BY ${sortColumn} ${sortDirection}
      LIMIT $${paramIndex} OFFSET $${paramIndex + 1}
    `, queryParams);
    console.log(`📊 Orders query: ${Date.now() - t1}ms, rows: ${ordersResult.rows.length}, limit: ${limit}, offset: ${offset}, search: ${search || 'none'}`);
    
    if (ordersResult.rows.length === 0) {
      return [];
    }
    
    // Pobierz wszystkie produkty dla wszystkich zamówień w JEDNYM zapytaniu
    const orderIds = ordersResult.rows.map(row => row.id);
    const t2 = Date.now();
    const itemsResult = await client.query(`
      SELECT 
        oi.*,
        p.name as product_name,
        p.description as product_description,
        p.image_url as product_image_url
      FROM commerce.order_items oi
      LEFT JOIN catalog.products p ON oi.product_id = p.id
      WHERE oi.order_id = ANY($1)
      ORDER BY oi.order_id, oi.id
    `, [orderIds]);
    console.log(`📊 Items query: ${Date.now() - t2}ms, rows: ${itemsResult.rows.length}`);
    
    // Grupuj produkty po order_id
    const t3 = Date.now();
    const itemsByOrderId = new Map<string, any[]>();
    for (const item of itemsResult.rows) {
      const orderId = item.order_id;
      if (!itemsByOrderId.has(orderId)) {
        itemsByOrderId.set(orderId, []);
      }
      itemsByOrderId.get(orderId)!.push(item);
    }
    console.log(`📊 Group items: ${Date.now() - t3}ms`);
    
    // Zbuduj zamówienia z odpowiednimi produktami
    const t4 = Date.now();
    const orders = ordersResult.rows.map(orderRow => {
      const items = itemsByOrderId.get(orderRow.id) || [];
      return buildOrderFromRow(orderRow, items);
    });
    console.log(`📊 Build orders: ${Date.now() - t4}ms`);
    console.log(`📊 TOTAL getOrdersFromPostgres: ${Date.now() - startTime}ms`);
    
    return orders;
  } catch (error) {
    console.error('❌ Error fetching orders from PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getOrdersCount(
  search?: string,
  sourceFilter?: string,
  dateRangeType?: string,
  customDateFrom?: string,
  customDateTo?: string
) {
  const client = await pool.connect();
  
  try {
    const conditions: string[] = [];
    let queryParams: any[] = [];
    let paramIndex = 1;
    
    if (search && search.trim()) {
      const searchTerm = `%${search.trim().toLowerCase()}%`;
      conditions.push(`(
        LOWER(order_code) LIKE $${paramIndex} OR
        CAST(order_number AS TEXT) LIKE $${paramIndex} OR
        LOWER(buyer_first_name) LIKE $${paramIndex} OR
        LOWER(buyer_last_name) LIKE $${paramIndex} OR
        LOWER(CONCAT(buyer_first_name, ' ', buyer_last_name)) LIKE $${paramIndex} OR
        LOWER(buyer_email) LIKE $${paramIndex}
      )`);
      queryParams.push(searchTerm);
      paramIndex++;
    }
    
    // Filtr źródła
    if (sourceFilter && sourceFilter !== 'all') {
      conditions.push(`source = $${paramIndex}`);
      queryParams.push(sourceFilter);
      paramIndex++;
    }
    
    // Filtr zakresu dat
    if (dateRangeType && dateRangeType !== 'all') {
      const now = new Date();
      let startDate: Date | null = null;
      let endDate: Date | null = null;

      if (dateRangeType === 'today') {
        startDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 0, 0, 0);
        endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
      } else if (dateRangeType === 'yesterday') {
        const yesterday = new Date(now);
        yesterday.setDate(yesterday.getDate() - 1);
        startDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 0, 0, 0);
        endDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 23, 59, 59);
      } else if (dateRangeType === '7days') {
        startDate = new Date(now);
        startDate.setDate(startDate.getDate() - 6);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
      } else if (dateRangeType === '30days') {
        startDate = new Date(now);
        startDate.setDate(startDate.getDate() - 29);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
      } else if (dateRangeType === 'custom' && customDateFrom && customDateTo) {
        startDate = new Date(customDateFrom);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(customDateTo);
        endDate.setHours(23, 59, 59, 999);
      }

      if (startDate && endDate) {
        conditions.push(`order_date >= $${paramIndex}`);
        queryParams.push(startDate.toISOString());
        paramIndex++;
        conditions.push(`order_date <= $${paramIndex}`);
        queryParams.push(endDate.toISOString());
        paramIndex++;
      }
    }
    
    const whereClause = conditions.length > 0 ? `WHERE ${conditions.join(' AND ')}` : '';
    
    const result = await client.query(
      `SELECT COUNT(*) as count FROM commerce.orders ${whereClause}`,
      queryParams
    );
    return parseInt(result.rows[0].count, 10);
  } catch (error) {
    console.error('❌ Error counting orders:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getOrderByNumberFromPostgres(orderNumber: number) {
  const client = await pool.connect();
  
  try {
    const orderResult = await client.query(`
      SELECT * FROM commerce.orders 
      WHERE order_number = $1
    `, [orderNumber]);
    
    if (orderResult.rows.length === 0) {
      return null;
    }
    
    const orderRow = orderResult.rows[0];
    
    const itemsResult = await client.query(`
      SELECT 
        oi.*,
        p.name as product_name,
        p.description as product_description,
        p.image_url as product_image_url
      FROM commerce.order_items oi
      LEFT JOIN catalog.products p ON oi.product_id = p.id
      WHERE oi.order_id = $1
    `, [orderRow.id]);
    
    return buildOrderFromRow(orderRow, itemsResult.rows);
  } catch (error) {
    console.error('❌ Error fetching order by number from PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getOrderFromPostgres(allegroOrderId: string) {
  const client = await pool.connect();
  
  try {
    const orderResult = await client.query(`
      SELECT * FROM commerce.orders 
      WHERE source_order_id = $1
    `, [allegroOrderId]);
    
    if (orderResult.rows.length === 0) {
      return null;
    }
    
    const orderRow = orderResult.rows[0];
    
    const itemsResult = await client.query(`
      SELECT 
        oi.*,
        p.name as product_name,
        p.description as product_description,
        p.image_url as product_image_url
      FROM commerce.order_items oi
      LEFT JOIN catalog.products p ON oi.product_id = p.id
      WHERE oi.order_id = $1
    `, [orderRow.id]);
    
    return buildOrderFromRow(orderRow, itemsResult.rows);
  } catch (error) {
    console.error('❌ Error fetching order from PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getUniqueProductsFromPostgres() {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT product_id as offer_id, external_id as offer_external_id, name
      FROM catalog.products
      WHERE external_id IS NOT NULL
        AND name IS NOT NULL
        AND name != ''
      ORDER BY name
    `);
    
    return result.rows;
  } catch (error) {
    console.error('❌ Error fetching unique products from PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getUniqueProductsWithImagesFromPostgres() {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT DISTINCT ON (external_id)
        product_id as offer_id, 
        external_id as offer_external_id, 
        name,
        image_url
      FROM catalog.products
      WHERE external_id IS NOT NULL
      ORDER BY external_id, updated_at DESC
    `);
    
    console.log(`📦 Found ${result.rows.length} unique products`);
    const withImages = result.rows.filter(p => p.image_url).length;
    console.log(`🖼️ ${withImages} products have image URLs`);
    
    return result.rows;
  } catch (error) {
    console.error('❌ Error fetching unique products with images from PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function updateProductImageUrl(externalId: string, imageUrl: string) {
  const client = await pool.connect();
  
  try {
    await client.query(`
      UPDATE catalog.products
      SET image_url = $1, updated_at = NOW()
      WHERE external_id = $2
    `, [imageUrl, externalId]);
    
    await client.query(`
      UPDATE commerce.order_items
      SET image_url = $1
      WHERE offer_external_id = $2
    `, [imageUrl, externalId]);
  } catch (error) {
    console.error(`❌ Error updating image URL for product ${externalId}:`, error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getAllProducts() {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT * FROM catalog.products
      ORDER BY updated_at DESC
    `);
    
    return result.rows;
  } catch (error) {
    console.error('❌ Error fetching products from PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function testConnection() {
  try {
    const client = await pool.connect();
    const result = await client.query('SELECT NOW()');
    console.log('✅ PostgreSQL connection successful:', result.rows[0]);
    client.release();
    return true;
  } catch (error) {
    console.error('❌ PostgreSQL connection failed:', error);
    return false;
  }
}

export async function createSyncLog(startedAt: Date) {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      INSERT INTO allegro.sync_logs (sync_started_at, status)
      VALUES ($1, 'RUNNING')
      RETURNING id
    `, [startedAt]);
    
    return result.rows[0].id;
  } catch (error) {
    console.error('❌ Error creating sync log:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function completeSyncLog(
  logId: number, 
  status: 'SUCCESS' | 'ERROR',
  ordersFetched: number,
  ordersNew: number,
  ordersUpdated: number,
  errorMessage?: string
) {
  const client = await pool.connect();
  
  try {
    const completedAt = new Date();
    const startResult = await client.query(
      'SELECT sync_started_at FROM allegro.sync_logs WHERE id = $1',
      [logId]
    );
    
    const syncDuration = startResult.rows[0] 
      ? completedAt.getTime() - new Date(startResult.rows[0].sync_started_at).getTime()
      : 0;
    
    await client.query(`
      UPDATE allegro.sync_logs
      SET sync_completed_at = $1,
          status = $2,
          orders_fetched = $3,
          orders_new = $4,
          orders_updated = $5,
          error_message = $6,
          sync_duration_ms = $7
      WHERE id = $8
    `, [completedAt, status, ordersFetched, ordersNew, ordersUpdated, errorMessage, syncDuration, logId]);
  } catch (error) {
    console.error('❌ Error completing sync log:', error);
  } finally {
    client.release();
  }
}

export async function getSyncLogs(limit: number = 50) {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT * FROM allegro.sync_logs
      ORDER BY sync_started_at DESC
      LIMIT $1
    `, [limit]);
    
    return result.rows;
  } catch (error) {
    console.error('❌ Error fetching sync logs:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function saveOrderToCommerce(
  source: 'ALLEGRO' | 'SHOPER',
  sourceOrderId: string,
  orderData: any,
  shipments: any[] = [],
  customerReturns: any[] = [],
  accessToken?: string
): Promise<{ isNew: boolean }> {
  const client = await pool.connect();
  
  try {
    // Wymuś odświeżenie snapshotu - usuń stary cache połączenia
    await client.query('DISCARD ALL');
    await client.query('BEGIN');

    const existsResult = await client.query(
      'SELECT id FROM commerce.orders WHERE source_order_id = $1 AND source = $2',
      [sourceOrderId, source]
    );
    const isNew = existsResult.rows.length === 0;

    let buyer, payment, delivery, lineItems, orderDate, billingAddress;

    if (source === 'ALLEGRO') {
      buyer = orderData.buyer || {};
      payment = orderData.payment || {};
      delivery = orderData.delivery || {};
      lineItems = orderData.lineItems || [];
      orderDate = orderData.boughtAt || orderData.updatedAt;
      billingAddress = orderData.invoice || {}; // Allegro: invoice jako billing address
    } else if (source === 'SHOPER') {
      billingAddress = orderData.billing_address || {};
      const deliveryAddress = orderData.delivery_address || {};
      buyer = {
        login: billingAddress.firstname && billingAddress.lastname 
          ? `${billingAddress.firstname} ${billingAddress.lastname}` 
          : orderData.user_login || 'Unknown',
        email: billingAddress.email || orderData.email,
        firstName: billingAddress.firstname,
        lastName: billingAddress.lastname,
        companyName: billingAddress.company,
        phoneNumber: billingAddress.phone || deliveryAddress.phone,
      };
      
      const paidAmount = parseFloat(orderData.paid || '0');
      const totalAmount = parseFloat(orderData.sum || '0');
      
      console.log(`🔍 DEBUG - Shoper payment raw data for order ${orderData.order_id}:`);
      console.log(`   orderData.paid = "${orderData.paid}" → paidAmount = ${paidAmount}`);
      console.log(`   orderData.sum = "${orderData.sum}" → totalAmount = ${totalAmount}`);
      console.log(`   orderData.is_paid = ${orderData.is_paid}`);
      
      let paymentStatus = 'PENDING';
      if (orderData.is_paid || paidAmount >= totalAmount) {
        paymentStatus = 'PAID';
      } else if (paidAmount > 0) {
        paymentStatus = 'PARTIAL';
      } else {
        paymentStatus = 'UNPAID';
      }
      
      console.log(`   Calculated paymentStatus = "${paymentStatus}"`);
      
      payment = {
        id: orderData.payment_id,
        status: paymentStatus,
        // If payment is PAID or PARTIAL, it's ONLINE. Only UNPAID with POBRANIE method is COD
        type: (paymentStatus === 'UNPAID' && orderData.payment_method_name?.toUpperCase().includes('POBRANIE')) 
          ? 'CASH_ON_DELIVERY' 
          : 'ONLINE',
        provider: orderData.payment_method_name,
        paidAmount: {
          amount: paidAmount,
          currency: orderData.currency || 'PLN'
        }
      };
      
      // Mapuj adres dostawy z poprawnych pól Shoper API
      const deliveryAddressObj: any = {};
      if (deliveryAddress.street1) {
        deliveryAddressObj.street = deliveryAddress.street1;
      }
      if (deliveryAddress.postcode) {
        deliveryAddressObj.zipCode = deliveryAddress.postcode;
      }
      if (deliveryAddress.city) {
        deliveryAddressObj.city = deliveryAddress.city;
      }
      if (deliveryAddress.country_code) {
        deliveryAddressObj.countryCode = deliveryAddress.country_code;
      }
      if (deliveryAddress.firstname) {
        deliveryAddressObj.firstName = deliveryAddress.firstname;
      }
      if (deliveryAddress.lastname) {
        deliveryAddressObj.lastName = deliveryAddress.lastname;
      }
      if (deliveryAddress.company) {
        deliveryAddressObj.companyName = deliveryAddress.company;
      }
      if (deliveryAddress.phone) {
        deliveryAddressObj.phoneNumber = deliveryAddress.phone;
      }
      
      delivery = {
        method: { name: orderData.delivery_method_name },
        address: Object.keys(deliveryAddressObj).length > 0 ? deliveryAddressObj : undefined,
        cost: {
          amount: orderData.delivery_price,
          currency: orderData.currency || 'PLN'
        }
      } as any;
      
      // Dodaj delivery_date jeśli istnieje
      if (orderData.delivery_date) {
        (delivery as any).time = {
          from: orderData.delivery_date,
          to: orderData.delivery_date
        };
      }
      
      lineItems = orderData.products_data || [];
      orderDate = orderData.date || orderData.confirm_date;
    }

    // Przetwórz shipments z Shoper do jednolitego formatu
    if (source === 'SHOPER' && shipments && shipments.length > 0) {
      const processedShipments: any[] = [];
      const carrierName = delivery.method?.name || 'Unknown';
      
      for (const parcel of shipments) {
        // shipping_code zawiera numery przesyłek oddzielone przecinkami
        const trackingNumbers = (parcel.shipping_code || '').split(',').filter((n: string) => n.trim());
        
        // Jeśli są numery przesyłek, utwórz wpis dla każdego
        if (trackingNumbers.length > 0) {
          for (const trackingNumber of trackingNumbers) {
            processedShipments.push({
              waybill: trackingNumber.trim(),
              carrierId: carrierName,
              createdAt: parcel.send_date && parcel.send_date !== '0000-00-00 00:00:00' ? parcel.send_date : parcel.order_date,
              sent: parcel.sent === '1' || parcel.sent === 1
            });
          }
        } else if (carrierName && carrierName !== 'Unknown') {
          // Jeśli nie ma numerów, ale jest metoda dostawy, pokaż to
          processedShipments.push({
            waybill: null,
            carrierId: carrierName,
            createdAt: parcel.send_date && parcel.send_date !== '0000-00-00 00:00:00' ? parcel.send_date : parcel.order_date,
            sent: parcel.sent === '1' || parcel.sent === 1
          });
        }
      }
      
      shipments = processedShipments;
    }

    // Pobierz zwroty płatności z API /payments/refunds (tylko dla ALLEGRO)
    let refundAmount = 0;
    let refundDate: string | null = null;
    const refundReconciliation: any[] = [];
    
    if (source === 'ALLEGRO' && accessToken) {
      const { fetchPaymentRefunds } = await import('./allegro-api.js');
      const refunds = await fetchPaymentRefunds(sourceOrderId, accessToken);
      
      for (const refund of refunds) {
        const amount = parseFloat(refund.totalValue?.amount || '0');
        refundAmount += amount;
        refundReconciliation.push(refund);
        
        if (refund.createdAt) {
          if (!refundDate || new Date(refund.createdAt) > new Date(refundDate)) {
            refundDate = refund.createdAt;
          }
        }
      }
    }

    // Sprawdź czy są zwroty produktów (type: QUANTITY) w refund_reconciliation
    const hasReturnsInReconciliation = refundReconciliation.some(refund => 
      refund.lineItems && refund.lineItems.some((item: any) => item.type === 'QUANTITY')
    );
    const hasReturns = hasReturnsInReconciliation;

    // Wzbogać surowe dane API o informacje o zwrotach i refundach
    const enrichedRawPayload = {
      ...orderData,
      payment: {
        ...payment,
        refundAmount: refundAmount || 0,
        refundDate: refundDate || null,
        refundReconciliation: refundReconciliation
      },
      customerReturns: customerReturns,
      shipments: shipments
    };

    // 🔍 Detect changes before UPDATE
    let hasActualChanges = isNew; // Nowe zamówienie zawsze ma zmiany
    
    if (!isNew) {
      // Wymusz odświeżenie snapshotu przed SELECT
      await client.query('SAVEPOINT before_select');
      
      const previousState = await client.query(`
        SELECT 
          id, order_number, order_code, status, payment_status, payment_amount,
          tracking_numbers, refund_amount, has_returns, buyer_address, buyer_city
        FROM commerce.orders 
        WHERE source_order_id = $1 AND source = $2
        FOR UPDATE NOWAIT
      `, [sourceOrderId, source]);
      
      await client.query('RELEASE SAVEPOINT before_select');

      console.log(`🔍 DEBUG - SELECT previousState result for ${source}-${sourceOrderId}:`, JSON.stringify(previousState.rows, null, 2));

      if (previousState.rows.length > 0) {
        const prev = previousState.rows[0];
        const orderId = prev.id;
        const orderNumber = prev.order_number?.toString();
        const orderCode = prev.order_code;

        // Nowy stan
        const newStatus = source === 'ALLEGRO' ? (orderData.fulfillment?.status || orderData.status) : (orderData.status_id || 'new');
        const newPaymentStatus = payment.status || 'PENDING';
        const newPaymentAmount = payment.paidAmount?.amount ?? orderData.sum;
        const newRefundAmount = refundAmount || 0;
        const newHasReturns = hasReturns;
        const newTrackingNumbers = orderData.delivery?.trackingNumber || (orderData.packages ? JSON.stringify(orderData.packages) : null);
        const newBuyerAddress = delivery.address ? 
          `${delivery.address.street || ''}, ${delivery.address.zipCode || ''} ${delivery.address.city || ''}` : null;

        // Funkcja porównująca wartości numeryczne
        const compareNumeric = (oldVal: any, newVal: any): boolean => {
          const oldNum = parseFloat(oldVal?.toString() || '0');
          const newNum = parseFloat(newVal?.toString() || '0');
          return Math.abs(oldNum - newNum) > 0.001; // Tolerancja na błędy zaokrągleń
        };

        // DEBUG: Log wartości przed porównaniem
        console.log(`🔍 DEBUG - Porównanie wartości dla ${orderCode}:`);
        console.log(`   payment_status: prev="${prev.payment_status}" new="${newPaymentStatus}"`);
        console.log(`   payment_amount: prev="${prev.payment_amount}" (${typeof prev.payment_amount}) new="${newPaymentAmount}" (${typeof newPaymentAmount})`);
        console.log(`   buyer_address: prev="${prev.buyer_address}" new="${newBuyerAddress}"`);

        // Porównaj pola
        const changes: Array<{field: string; oldValue: string; newValue: string}> = [];

        if (prev.status !== newStatus) {
          changes.push({ field: 'status', oldValue: prev.status || 'null', newValue: newStatus || 'null' });
        }
        if (prev.payment_status !== newPaymentStatus) {
          changes.push({ field: 'payment_status', oldValue: prev.payment_status || 'null', newValue: newPaymentStatus });
        }
        if (compareNumeric(prev.payment_amount, newPaymentAmount)) {
          console.log(`   ⚠️ compareNumeric zwrócił TRUE: ${prev.payment_amount} vs ${newPaymentAmount}`);
          changes.push({ 
            field: 'payment_amount', 
            oldValue: prev.payment_amount?.toString() || '0', 
            newValue: newPaymentAmount?.toString() || '0' 
          });
        } else {
          console.log(`   ✅ compareNumeric zwrócił FALSE: ${prev.payment_amount} === ${newPaymentAmount}`);
        }
        if (compareNumeric(prev.refund_amount, newRefundAmount)) {
          changes.push({ 
            field: 'refund_amount', 
            oldValue: prev.refund_amount?.toString() || '0', 
            newValue: newRefundAmount?.toString() || '0' 
          });
        }
        if (prev.has_returns !== newHasReturns) {
          changes.push({ field: 'has_returns', oldValue: prev.has_returns ? 'true' : 'false', newValue: newHasReturns ? 'true' : 'false' });
        }
        if (prev.tracking_numbers !== newTrackingNumbers) {
          changes.push({ field: 'tracking_numbers', oldValue: prev.tracking_numbers || 'null', newValue: newTrackingNumbers || 'null' });
        }
        if (prev.buyer_address !== newBuyerAddress) {
          changes.push({ field: 'buyer_address', oldValue: prev.buyer_address || 'null', newValue: newBuyerAddress || 'null' });
        }

        // Oznacz czy są faktyczne zmiany
        hasActualChanges = changes.length > 0;

        // Loguj zmiany
        if (changes.length > 0) {
          console.log(`🔄 Wykryto ${changes.length} zmian w zamówieniu ${orderCode}:`);
          for (const change of changes) {
            console.log(`   📝 ${change.field}: "${change.oldValue}" → "${change.newValue}"`);
            
            // Zapisz do bazy
            await logOrderChange(
              orderId,
              orderNumber || '',
              orderCode || '',
              source,
              'UPDATED',
              change.field,
              change.oldValue,
              change.newValue
            );
          }
        } else {
          console.log(`   ✅ Brak zmian w zamówieniu ${orderCode} - pomijam aktualizację`);
          // Jeśli brak zmian, zwróć bez aktualizacji bazy
          await client.query('COMMIT');
          return { isNew: false };
        }
      }
    }

    const paymentAmountParam = payment.paidAmount?.amount ?? orderData.sum;
    console.log(`🔍 DEBUG - INSERT parameter $19 (payment_amount) for ${source}-${sourceOrderId}:`);
    console.log(`   payment.paidAmount = ${JSON.stringify(payment.paidAmount)}`);
    console.log(`   payment.paidAmount?.amount = ${payment.paidAmount?.amount}`);
    console.log(`   orderData.sum = ${orderData.sum}`);
    console.log(`   Final value (payment.paidAmount?.amount ?? orderData.sum) = ${paymentAmountParam}`);

    const orderResult = await client.query(`
      INSERT INTO commerce.orders (
        source, source_order_id, order_date, status,
        buyer_login, buyer_email, buyer_company, buyer_first_name, buyer_last_name,
        buyer_phone, buyer_address, buyer_zip, buyer_city, buyer_country_code,
        payment_id, payment_status, payment_type, payment_provider,
        payment_amount, payment_currency,
        delivery_method, delivery_amount, delivery_currency,
        delivery_address, billing_address, billing_zip, billing_city, billing_country_code, tax_id,
        total_to_pay_amount, total_to_pay_currency,
        buyer_notes, invoice_required, tracking_numbers, shipments,
        refund_amount, refund_date, refund_reconciliation, has_returns,
        raw_payload, created_at, updated_at
      ) VALUES (
        $1, $2, $3, $4, $5, $6, $7, $8, $9, $10,
        $11, $12, $13, $14, $15, $16, $17, $18, $19, $20,
        $21, $22, $23, $24, $25, $26, $27, $28, $29, $30, $31, $32, $33, $34, $35, $36, $37, $38, $39, $40, NOW(), NOW()
      )
      ON CONFLICT (source, source_order_id) DO UPDATE SET
        status = EXCLUDED.status,
        payment_status = EXCLUDED.payment_status,
        payment_type = EXCLUDED.payment_type,
        payment_provider = EXCLUDED.payment_provider,
        payment_amount = EXCLUDED.payment_amount,
        buyer_address = EXCLUDED.buyer_address,
        buyer_zip = EXCLUDED.buyer_zip,
        buyer_city = EXCLUDED.buyer_city,
        delivery_address = EXCLUDED.delivery_address,
        billing_address = EXCLUDED.billing_address,
        billing_zip = EXCLUDED.billing_zip,
        billing_city = EXCLUDED.billing_city,
        billing_country_code = EXCLUDED.billing_country_code,
        tax_id = EXCLUDED.tax_id,
        tracking_numbers = EXCLUDED.tracking_numbers,
        shipments = EXCLUDED.shipments,
        refund_amount = EXCLUDED.refund_amount,
        refund_date = EXCLUDED.refund_date,
        refund_reconciliation = EXCLUDED.refund_reconciliation,
        has_returns = EXCLUDED.has_returns,
        raw_payload = EXCLUDED.raw_payload,
        updated_at = NOW()
      RETURNING id, order_number, order_code, payment_amount AS returned_payment_amount
    `, [
      source,
      sourceOrderId,
      orderDate,
      source === 'ALLEGRO' ? (orderData.fulfillment?.status || orderData.status) : (orderData.status_id || 'new'),
      buyer.login,
      buyer.email,
      buyer.companyName,
      buyer.firstName,
      buyer.lastName,
      buyer.phoneNumber,
      delivery.address ? 
        `${delivery.address.street || ''}, ${delivery.address.zipCode || ''} ${delivery.address.city || ''}` : null,
      delivery.address?.zipCode,
      delivery.address?.city,
      delivery.address?.countryCode || 'PL',
      payment.id,
      payment.status || 'PENDING',
      payment.type || (source === 'SHOPER' && orderData.cod ? 'CASH_ON_DELIVERY' : 'ONLINE'),
      payment.provider,
      payment.paidAmount?.amount ?? orderData.sum,
      payment.paidAmount?.currency || orderData.currency || 'PLN',
      delivery.method?.name || orderData.delivery_method_name,
      delivery.cost?.amount ?? orderData.delivery_price,
      delivery.cost?.currency || orderData.currency || 'PLN',
      delivery.address ? JSON.stringify(delivery.address) : null, // $24 - delivery_address JSONB
      billingAddress ? JSON.stringify(billingAddress) : null, // $25 - billing_address JSONB
      billingAddress?.zipCode || null, // $26 - billing_zip
      billingAddress?.city || null, // $27 - billing_city
      billingAddress?.countryCode || null, // $28 - billing_country_code
      billingAddress?.company?.taxId || null, // $29 - tax_id
      orderData.summary?.totalToPay?.amount ?? orderData.sum,
      orderData.summary?.totalToPay?.currency || orderData.currency || 'PLN',
      orderData.messageToSeller || orderData.notes,
      orderData.invoice?.required || orderData.want_invoice || false,
      orderData.delivery?.trackingNumber || (orderData.packages ? JSON.stringify(orderData.packages) : null),
      JSON.stringify(shipments),
      refundAmount || 0,
      refundDate || null,
      JSON.stringify(refundReconciliation),
      hasReturns,
      enrichedRawPayload
    ]);

    const commerceOrderId = orderResult.rows[0].id;
    console.log(`🔍 DEBUG - RETURNING after INSERT/UPDATE for ${source}-${sourceOrderId}:`);
    console.log(`   returned_payment_amount = ${orderResult.rows[0].returned_payment_amount}`);

    // Najpierw wstaw wszystkie order_items bez returns_quantity
    for (const item of lineItems) {
      let productId = null;
      let externalId, itemName, quantity, unitPrice, totalPrice, imageUrl;

      if (source === 'ALLEGRO') {
        externalId = item.offer?.external?.id;
        itemName = item.offer?.name;
        quantity = item.quantity;
        unitPrice = item.price?.amount;
        totalPrice = parseFloat(unitPrice) * quantity;
        imageUrl = item.imageUrl || item.offer?.images?.[0]?.url;
      } else if (source === 'SHOPER') {
        externalId = item.code || item.product_id?.toString();
        itemName = item.name || item.translation?.name;
        quantity = item.quantity;
        unitPrice = item.price || item.price_brutto;
        totalPrice = parseFloat(unitPrice) * quantity;
        imageUrl = item.main_image?.url || item.image_url;
      }

      if (externalId) {
        const productResult = await client.query(
          'SELECT id FROM catalog.products WHERE external_id = $1 AND source = $2',
          [externalId, source]
        );
        if (productResult.rows.length > 0) {
          productId = productResult.rows[0].id;
        } else {
          const newProductResult = await client.query(`
            INSERT INTO catalog.products (source, product_id, external_id, name, image_url, raw_data)
            VALUES ($1, $2, $3, $4, $5, $6)
            ON CONFLICT (source, product_id) DO UPDATE SET
              name = EXCLUDED.name,
              image_url = EXCLUDED.image_url,
              raw_data = EXCLUDED.raw_data,
              updated_at = NOW()
            RETURNING id
          `, [
            source,
            source === 'ALLEGRO' ? item.offer?.id : item.product_id,
            externalId,
            itemName,
            imageUrl,
            item
          ]);
          productId = newProductResult.rows[0].id;
        }
      }

      await client.query(`
        INSERT INTO commerce.order_items (
          order_id, source, product_id, offer_id, offer_external_id, offer_name,
          quantity, unit_price, total_price, currency,
          tax_rate, tax_subject, country_code, image_url,
          payment_type, invoice_required, returns_quantity, raw_data
        ) VALUES (
          $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17, $18
        )
        ON CONFLICT (order_id, offer_id) DO UPDATE SET
          quantity = EXCLUDED.quantity,
          unit_price = EXCLUDED.unit_price,
          total_price = EXCLUDED.total_price,
          image_url = EXCLUDED.image_url,
          payment_type = EXCLUDED.payment_type,
          invoice_required = EXCLUDED.invoice_required,
          raw_data = EXCLUDED.raw_data
      `, [
        commerceOrderId,
        source,
        productId,
        source === 'ALLEGRO' ? item.offer?.id : item.product_id,
        externalId,
        itemName,
        quantity,
        unitPrice,
        totalPrice,
        orderData.currency || 'PLN',
        source === 'ALLEGRO' ? item.reconciliation?.tax?.percentage : item.tax_rate,
        source === 'ALLEGRO' ? item.reconciliation?.tax?.subjectType : null,
        'PL',
        imageUrl,
        payment.type,
        orderData.invoice?.required || orderData.want_invoice || false,
        0,
        item
      ]);
    }

    // Aktualizuj returns_quantity na podstawie refund_reconciliation (bardziej stabilne dane)
    // ID w refund_reconciliation.lineItems to checkoutFormLineItemId (line_item_id z Allegro)
    if (source === 'ALLEGRO' && refundReconciliation && Array.isArray(refundReconciliation)) {
      for (const refund of refundReconciliation) {
        const refundLineItems = refund.lineItems || [];
        for (const refundLineItem of refundLineItems) {
          if (refundLineItem.id && refundLineItem.quantity) {
            // ID w refund_reconciliation to line_item_id (checkoutFormLineItemId)
            // Musimy znaleźć offer_id poprzez allegro.order_items
            await client.query(`
              UPDATE commerce.order_items
              SET returns_quantity = $1
              WHERE order_id = $2
                AND offer_id IN (
                  SELECT offer_id 
                  FROM allegro.order_items 
                  WHERE line_item_id = $3 
                    AND order_id = $4
                )
            `, [refundLineItem.quantity, commerceOrderId, refundLineItem.id, sourceOrderId]);
          }
        }
      }
    }

    await client.query('COMMIT');
    
    console.log(`✅ Order ${sourceOrderId} saved to commerce.orders (${source})`);
    
    return { isNew };
  } catch (error) {
    await client.query('ROLLBACK');
    console.error(`❌ Error saving order ${sourceOrderId} to commerce:`, error);
    throw error;
  } finally {
    client.release();
  }
}

export async function logOrderChange(
  orderId: string,
  orderNumber: string,
  orderCode: string,
  source: string,
  changeType: 'NEW' | 'UPDATED',
  fieldChanged?: string,
  oldValue?: string,
  newValue?: string
) {
  const client = await pool.connect();
  
  try {
    // Sprawdź czy taka sama zmiana już istnieje (ostatnie 5 minut)
    if (fieldChanged && newValue !== undefined) {
      const existingChange = await client.query(`
        SELECT id FROM public.order_changes
        WHERE order_code = $1
          AND field_changed = $2
          AND new_value = $3
          AND detected_at >= NOW() - INTERVAL '5 minutes'
        LIMIT 1
      `, [orderCode, fieldChanged, newValue]);
      
      if (existingChange.rows.length > 0) {
        console.log(`⏭️  Pominięto duplikat zmiany: ${orderCode} - ${fieldChanged}`);
        return; // Nie zapisuj duplikatu
      }
    }
    
    await client.query(`
      INSERT INTO public.order_changes (
        id, order_id, order_number, order_code, source, change_type, 
        field_changed, old_value, new_value, detected_at
      ) VALUES (
        gen_random_uuid(), $1, $2, $3, $4, $5, $6, $7, $8, NOW()
      )
    `, [orderId, orderNumber, orderCode, source, changeType, fieldChanged, oldValue, newValue]);
  } catch (error) {
    console.error('❌ Error logging order change:', error);
  } finally {
    client.release();
  }
}

export async function getRecentOrderChanges(limitMinutes: number = 60) {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT 
        oc.*,
        o.order_number,
        o.order_code,
        o.source,
        o.total_to_pay_amount,
        o.total_to_pay_currency,
        o.status,
        o.buyer_login,
        o.order_date
      FROM public.order_changes oc
      LEFT JOIN commerce.orders o ON oc.order_id = o.id
      WHERE oc.detected_at >= NOW() - INTERVAL '${limitMinutes} minutes'
      ORDER BY oc.detected_at DESC
      LIMIT 100
    `);
    
    return result.rows;
  } catch (error) {
    console.error('❌ Error fetching recent order changes:', error);
    return [];
  } finally {
    client.release();
  }
}

export async function getRecentlyUpdatedOrders(timeWindowMinutes: number = 10) {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT 
        o.*,
        CASE 
          WHEN o.created_at >= NOW() - INTERVAL '${timeWindowMinutes} minutes' THEN 'NEW'
          ELSE 'UPDATED'
        END as change_type
      FROM commerce.orders o
      WHERE o.updated_at >= NOW() - INTERVAL '${timeWindowMinutes} minutes'
         OR o.created_at >= NOW() - INTERVAL '${timeWindowMinutes} minutes'
      ORDER BY 
        CASE 
          WHEN o.updated_at > o.created_at THEN o.updated_at 
          ELSE o.created_at 
        END DESC
      LIMIT 100
    `);
    
    return result.rows;
  } catch (error) {
    console.error('❌ Error fetching recently updated orders:', error);
    return [];
  } finally {
    client.release();
  }
}

export async function getSyncSettingsFromPostgres() {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT * FROM sync_settings
      ORDER BY created_at DESC
      LIMIT 1
    `);
    
    if (result.rows.length === 0) {
      return null;
    }
    
    const row = result.rows[0];
    return {
      id: row.id,
      autoRefreshEnabled: row.auto_refresh_enabled,
      refreshIntervalMinutes: row.refresh_interval_minutes,
      lastSyncAt: row.last_sync_at,
      shoperAutoRefreshEnabled: row.shoper_auto_refresh_enabled,
      shoperRefreshIntervalMinutes: row.shoper_refresh_interval_minutes,
      lastShoperSyncAt: row.last_shoper_sync_at,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  } catch (error) {
    console.error('❌ Error fetching sync settings:', error);
    return null;
  } finally {
    client.release();
  }
}

export async function createOrUpdateSyncSettingsInPostgres(settings: any) {
  const client = await pool.connect();
  
  try {
    // Sprawdź czy istnieją ustawienia
    const existing = await client.query(`
      SELECT id FROM sync_settings LIMIT 1
    `);
    
    if (existing.rows.length === 0) {
      // Utwórz nowe
      const result = await client.query(`
        INSERT INTO sync_settings (
          auto_refresh_enabled,
          refresh_interval_minutes,
          last_sync_at,
          shoper_auto_refresh_enabled,
          shoper_refresh_interval_minutes,
          last_shoper_sync_at,
          created_at,
          updated_at
        ) VALUES ($1, $2, $3, $4, $5, $6, NOW(), NOW())
        RETURNING *
      `, [
        settings.autoRefreshEnabled ?? true,
        settings.refreshIntervalMinutes ?? '3',
        settings.lastSyncAt ?? null,
        settings.shoperAutoRefreshEnabled ?? true,
        settings.shoperRefreshIntervalMinutes ?? '5',
        settings.lastShoperSyncAt ?? null,
      ]);
      
      const row = result.rows[0];
      return {
        id: row.id,
        autoRefreshEnabled: row.auto_refresh_enabled,
        refreshIntervalMinutes: row.refresh_interval_minutes,
        lastSyncAt: row.last_sync_at,
        shoperAutoRefreshEnabled: row.shoper_auto_refresh_enabled,
        shoperRefreshIntervalMinutes: row.shoper_refresh_interval_minutes,
        lastShoperSyncAt: row.last_shoper_sync_at,
        createdAt: row.created_at,
        updatedAt: row.updated_at,
      };
    } else {
      // Zaktualizuj istniejące
      const result = await client.query(`
        UPDATE sync_settings
        SET 
          auto_refresh_enabled = COALESCE($1, auto_refresh_enabled),
          refresh_interval_minutes = COALESCE($2, refresh_interval_minutes),
          last_sync_at = COALESCE($3, last_sync_at),
          shoper_auto_refresh_enabled = COALESCE($4, shoper_auto_refresh_enabled),
          shoper_refresh_interval_minutes = COALESCE($5, shoper_refresh_interval_minutes),
          last_shoper_sync_at = COALESCE($6, last_shoper_sync_at),
          updated_at = NOW()
        WHERE id = $7
        RETURNING *
      `, [
        settings.autoRefreshEnabled,
        settings.refreshIntervalMinutes,
        settings.lastSyncAt,
        settings.shoperAutoRefreshEnabled,
        settings.shoperRefreshIntervalMinutes,
        settings.lastShoperSyncAt,
        existing.rows[0].id,
      ]);
      
      const row = result.rows[0];
      return {
        id: row.id,
        autoRefreshEnabled: row.auto_refresh_enabled,
        refreshIntervalMinutes: row.refresh_interval_minutes,
        lastSyncAt: row.last_sync_at,
        shoperAutoRefreshEnabled: row.shoper_auto_refresh_enabled,
        shoperRefreshIntervalMinutes: row.shoper_refresh_interval_minutes,
        lastShoperSyncAt: row.last_shoper_sync_at,
        createdAt: row.created_at,
        updatedAt: row.updated_at,
      };
    }
  } catch (error) {
    console.error('❌ Error creating/updating sync settings:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getOrderStatistics() {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT 
        COUNT(*) as total_orders,
        COUNT(CASE 
          WHEN COALESCE(payment_amount, 0)::numeric > 0 THEN 1 
        END) as paid_orders,
        COUNT(CASE 
          WHEN COALESCE(payment_amount, 0)::numeric = 0 THEN 1 
        END) as unpaid_orders,
        COALESCE(SUM(COALESCE(total_to_pay_amount, 0)::numeric), 0) as total_revenue
      FROM commerce.orders
    `);
    
    const stats = result.rows[0];
    return {
      totalOrders: parseInt(stats.total_orders) || 0,
      paidOrders: parseInt(stats.paid_orders) || 0,
      unpaidOrders: parseInt(stats.unpaid_orders) || 0,
      totalRevenue: parseFloat(stats.total_revenue) || 0,
    };
  } catch (error) {
    console.error('❌ Error fetching order statistics:', error);
    return {
      totalOrders: 0,
      paidOrders: 0,
      unpaidOrders: 0,
      totalRevenue: 0,
    };
  } finally {
    client.release();
  }
}

export async function getOrdersChartData(days: number = 30) {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      WITH daily_stats AS (
        SELECT 
          DATE(order_date) as date,
          COUNT(*) as orders,
          COUNT(CASE WHEN COALESCE(payment_amount, 0) > 0 THEN 1 END) as paid_orders,
          COALESCE(SUM(COALESCE(total_to_pay_amount, 0)), 0) as revenue
        FROM commerce.orders
        WHERE order_date >= NOW() - INTERVAL '${days} days'
        GROUP BY DATE(order_date)
      )
      SELECT 
        TO_CHAR(date, 'YYYY-MM-DD') as date,
        orders,
        paid_orders,
        revenue
      FROM daily_stats
      ORDER BY date ASC
    `);
    
    return result.rows.map(row => ({
      date: row.date,
      orders: parseInt(row.orders) || 0,
      paidOrders: parseInt(row.paid_orders) || 0,
      revenue: parseFloat(row.revenue) || 0,
    }));
  } catch (error) {
    console.error('❌ Error fetching chart data:', error);
    return [];
  } finally {
    client.release();
  }
}

export async function getOrdersPeriodSummary(days: number = 30) {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      WITH period_stats AS (
        SELECT 
          -- Przychód całkowity za okres
          COALESCE(SUM(COALESCE(total_to_pay_amount, 0)), 0) as revenue_total,
          -- Zapłacono (suma payment_amount)
          COALESCE(SUM(COALESCE(payment_amount, 0)), 0) as paid_amount,
          -- Przychód z Allegro
          COALESCE(SUM(CASE WHEN source = 'ALLEGRO' THEN COALESCE(total_to_pay_amount, 0) ELSE 0 END), 0) as revenue_allegro,
          -- Przychód ze Shoper
          COALESCE(SUM(CASE WHEN source = 'SHOPER' THEN COALESCE(total_to_pay_amount, 0) ELSE 0 END), 0) as revenue_shoper
        FROM commerce.orders
        WHERE order_date >= NOW() - INTERVAL '${days} days'
      ),
      today_stats AS (
        SELECT 
          COALESCE(SUM(COALESCE(total_to_pay_amount, 0)), 0) as revenue_today
        FROM commerce.orders
        WHERE DATE(order_date) = CURRENT_DATE
      )
      SELECT 
        p.revenue_total,
        p.paid_amount,
        p.revenue_allegro,
        p.revenue_shoper,
        t.revenue_today
      FROM period_stats p, today_stats t
    `);
    
    const stats = result.rows[0] || {};
    return {
      revenueTotal: parseFloat(stats.revenue_total) || 0,
      paidAmount: parseFloat(stats.paid_amount) || 0,
      revenueAllegro: parseFloat(stats.revenue_allegro) || 0,
      revenueShoper: parseFloat(stats.revenue_shoper) || 0,
      revenueToday: parseFloat(stats.revenue_today) || 0,
    };
  } catch (error) {
    console.error('❌ Error fetching period summary:', error);
    return {
      revenueTotal: 0,
      paidAmount: 0,
      revenueAllegro: 0,
      revenueShoper: 0,
      revenueToday: 0,
    };
  } finally {
    client.release();
  }
}

export async function getTodayDetailedStats() {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      WITH today_stats AS (
        SELECT 
          -- Dzisiejszy przychód
          COALESCE(SUM(COALESCE(total_to_pay_amount, 0)), 0) as revenue_today,
          -- Zapłacono dzisiaj (suma payment_amount)
          COALESCE(SUM(COALESCE(payment_amount, 0)), 0) as paid_today,
          -- Pobrania (COD) - liczba i kwota
          COUNT(CASE WHEN payment_type = 'CASH_ON_DELIVERY' THEN 1 END) as cod_count,
          COALESCE(SUM(CASE WHEN payment_type = 'CASH_ON_DELIVERY' THEN COALESCE(total_to_pay_amount, 0) ELSE 0 END), 0) as cod_amount,
          -- Niezapłacono (poza pobraniami)
          COALESCE(SUM(CASE WHEN payment_type != 'CASH_ON_DELIVERY' AND COALESCE(payment_amount, 0) = 0 THEN COALESCE(total_to_pay_amount, 0) ELSE 0 END), 0) as unpaid_non_cod,
          -- Zapłacono Allegro
          COALESCE(SUM(CASE WHEN source = 'ALLEGRO' AND COALESCE(payment_amount, 0) > 0 THEN COALESCE(payment_amount, 0) ELSE 0 END), 0) as paid_allegro,
          -- Zapłacono Shoper
          COALESCE(SUM(CASE WHEN source = 'SHOPER' AND COALESCE(payment_amount, 0) > 0 THEN COALESCE(payment_amount, 0) ELSE 0 END), 0) as paid_shoper,
          -- Liczba zamówień ogółem
          COUNT(*) as orders_count,
          -- Liczba zamówień Allegro
          COUNT(CASE WHEN source = 'ALLEGRO' THEN 1 END) as orders_allegro,
          -- Liczba zamówień Shoper
          COUNT(CASE WHEN source = 'SHOPER' THEN 1 END) as orders_shoper
        FROM commerce.orders
        WHERE DATE(order_date) = CURRENT_DATE
      ),
      yesterday_stats AS (
        SELECT 
          COALESCE(SUM(COALESCE(total_to_pay_amount, 0)), 0) as revenue_yesterday
        FROM commerce.orders
        WHERE DATE(order_date) = CURRENT_DATE - INTERVAL '1 day'
      )
      SELECT 
        t.*,
        y.revenue_yesterday,
        TO_CHAR(CURRENT_DATE, 'Day') as day_name,
        TO_CHAR(CURRENT_DATE, 'DD.MM.YYYY') as date_formatted,
        CASE 
          WHEN y.revenue_yesterday > 0 THEN 
            ROUND(((t.revenue_today - y.revenue_yesterday) / y.revenue_yesterday * 100)::numeric, 1)
          ELSE 0
        END as revenue_change_percent
      FROM today_stats t, yesterday_stats y
    `);
    
    const stats = result.rows[0] || {};
    return {
      revenueToday: parseFloat(stats.revenue_today) || 0,
      paidToday: parseFloat(stats.paid_today) || 0,
      codCount: parseInt(stats.cod_count) || 0,
      codAmount: parseFloat(stats.cod_amount) || 0,
      unpaidNonCod: parseFloat(stats.unpaid_non_cod) || 0,
      paidAllegro: parseFloat(stats.paid_allegro) || 0,
      paidShoper: parseFloat(stats.paid_shoper) || 0,
      ordersCount: parseInt(stats.orders_count) || 0,
      ordersAllegro: parseInt(stats.orders_allegro) || 0,
      ordersShoper: parseInt(stats.orders_shoper) || 0,
      revenueYesterday: parseFloat(stats.revenue_yesterday) || 0,
      dayName: (stats.day_name || '').trim(),
      dateFormatted: stats.date_formatted || '',
      revenueChangePercent: parseFloat(stats.revenue_change_percent) || 0,
    };
  } catch (error) {
    console.error('❌ Error fetching today detailed stats:', error);
    return {
      revenueToday: 0,
      paidToday: 0,
      codCount: 0,
      codAmount: 0,
      unpaidNonCod: 0,
      paidAllegro: 0,
      paidShoper: 0,
      ordersCount: 0,
      ordersAllegro: 0,
      ordersShoper: 0,
      revenueYesterday: 0,
      dayName: '',
      dateFormatted: '',
      revenueChangePercent: 0,
    };
  } finally {
    client.release();
  }
}

export async function getAllegroConnectionFromPostgres(): Promise<any | null> {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      SELECT id, client_id, client_secret, access_token, refresh_token, token_expires_at, is_active, created_at, updated_at
      FROM allegro_connections
      ORDER BY created_at DESC
      LIMIT 1
    `);
    
    if (result.rows.length === 0) {
      return null;
    }
    
    const row = result.rows[0];
    return {
      id: row.id,
      clientId: row.client_id,
      clientSecret: row.client_secret,
      accessToken: row.access_token,
      refreshToken: row.refresh_token,
      tokenExpiresAt: row.token_expires_at,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  } catch (error) {
    console.error('❌ Error fetching Allegro connection from PostgreSQL:', error);
    return null;
  } finally {
    client.release();
  }
}

export async function saveAllegroConnectionToPostgres(connection: {
  clientId: string;
  clientSecret: string;
  accessToken?: string;
  refreshToken?: string;
  tokenExpiresAt?: Date;
  isActive?: boolean;
}): Promise<void> {
  const client = await pool.connect();
  try {
    // Check if connection exists
    const existingResult = await client.query(`
      SELECT id FROM allegro_connections ORDER BY created_at DESC LIMIT 1
    `);
    
    if (existingResult.rows.length > 0) {
      // Update existing connection
      await client.query(`
        UPDATE allegro_connections
        SET client_id = $1, client_secret = $2, access_token = $3, refresh_token = $4, 
            token_expires_at = $5, is_active = $6, updated_at = NOW()
        WHERE id = $7
      `, [
        connection.clientId,
        connection.clientSecret,
        connection.accessToken || null,
        connection.refreshToken || null,
        connection.tokenExpiresAt || null,
        connection.isActive !== undefined ? connection.isActive : true,
        existingResult.rows[0].id,
      ]);
      console.log('✅ Updated Allegro connection in PostgreSQL');
    } else {
      // Insert new connection
      await client.query(`
        INSERT INTO allegro_connections (
          client_id, client_secret, access_token, refresh_token, token_expires_at, is_active, created_at, updated_at
        ) VALUES ($1, $2, $3, $4, $5, $6, NOW(), NOW())
      `, [
        connection.clientId,
        connection.clientSecret,
        connection.accessToken || null,
        connection.refreshToken || null,
        connection.tokenExpiresAt || null,
        connection.isActive !== undefined ? connection.isActive : true,
      ]);
      console.log('✅ Inserted Allegro connection to PostgreSQL');
    }
  } catch (error) {
    console.error('❌ Error saving Allegro connection to PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function updateAllegroConnectionTokens(
  accessToken: string,
  refreshToken: string,
  expiresAt: Date
): Promise<void> {
  const client = await pool.connect();
  try {
    await client.query(`
      UPDATE allegro_connections
      SET access_token = $1, refresh_token = $2, token_expires_at = $3, is_active = true, updated_at = NOW()
      WHERE id = (SELECT id FROM allegro_connections ORDER BY created_at DESC LIMIT 1)
    `, [accessToken, refreshToken, expiresAt]);
    
    console.log('✅ Updated Allegro connection tokens in PostgreSQL');
  } catch (error) {
    console.error('❌ Error updating Allegro connection tokens in PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function renumberOrdersByDate(): Promise<void> {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      WITH numbered AS (
        SELECT 
          id,
          ROW_NUMBER() OVER (ORDER BY order_date ASC) as new_number,
          source
        FROM commerce.orders
      )
      UPDATE commerce.orders o
      SET 
        order_number = n.new_number
      FROM numbered n
      WHERE o.id = n.id
    `);
    
    // Update sequence to continue from max order_number
    const maxResult = await client.query(`
      SELECT COALESCE(MAX(order_number), 0) as max_number 
      FROM commerce.orders
    `);
    const maxNumber = maxResult.rows[0].max_number;
    
    await client.query(`
      SELECT setval('commerce.orders_order_number_seq', $1, true)
    `, [maxNumber]);
    
    console.log(`✅ Renumbered ${result.rowCount} orders by date, sequence set to ${maxNumber}`);
  } catch (error) {
    console.error('❌ Error renumbering orders by date:', error);
    throw error;
  } finally {
    client.release();
  }
}

// ========================================
// Auth Functions
// ========================================

export async function getUserByUsername(username: string) {
  const result = await pool.query(
    'SELECT * FROM users WHERE username = $1',
    [username]
  );
  return result.rows[0];
}

export async function getUserById(id: number) {
  const result = await pool.query(
    'SELECT * FROM users WHERE id = $1',
    [id]
  );
  return result.rows[0];
}

export async function getUserByEmail(email: string) {
  const result = await pool.query(
    'SELECT * FROM users WHERE email = $1',
    [email]
  );
  return result.rows[0];
}

export async function createUser(userData: { username: string; email: string; password: string; firstName?: string; lastName?: string }) {
  const result = await pool.query(
    `INSERT INTO users (username, email, password, first_name, last_name, is_active, created_at, updated_at)
     VALUES ($1, $2, $3, $4, $5, true, NOW(), NOW())
     RETURNING *`,
    [userData.username, userData.email, userData.password, userData.firstName || null, userData.lastName || null]
  );
  return result.rows[0];
}

export async function createPasswordResetToken(userId: number, token: string, expiresAt: Date) {
  const result = await pool.query(
    `INSERT INTO password_reset_tokens (user_id, token, expires_at, used, created_at)
     VALUES ($1, $2, $3, false, NOW())
     RETURNING *`,
    [userId, token, expiresAt]
  );
  return result.rows[0];
}

export async function getPasswordResetToken(token: string) {
  const result = await pool.query(
    `SELECT * FROM password_reset_tokens 
     WHERE token = $1 AND used = false AND expires_at > NOW()`,
    [token]
  );
  return result.rows[0];
}

export async function markTokenAsUsed(token: string) {
  await pool.query(
    'UPDATE password_reset_tokens SET used = true WHERE token = $1',
    [token]
  );
}

export async function updateUserPassword(userId: number, newPassword: string) {
  await pool.query(
    'UPDATE users SET password = $1, updated_at = NOW() WHERE id = $2',
    [newPassword, userId]
  );
}

export async function deleteExpiredTokens() {
  await pool.query(
    'DELETE FROM password_reset_tokens WHERE expires_at < NOW()'
  );
}

export async function getAllUsers() {
  const result = await pool.query(
    `SELECT id, username, email, first_name, last_name, role, permissions, is_active, created_at, updated_at 
     FROM users 
     ORDER BY created_at DESC`
  );
  return result.rows;
}

export async function updateUserRole(userId: number, role: string) {
  const result = await pool.query(
    `UPDATE users 
     SET role = $1, updated_at = NOW() 
     WHERE id = $2
     RETURNING id, username, email, first_name, last_name, role, permissions, is_active, created_at, updated_at`,
    [role, userId]
  );
  return result.rows[0];
}

export async function updateUserPermissions(userId: number, permissions: string[]) {
  const result = await pool.query(
    `UPDATE users 
     SET permissions = $1, updated_at = NOW() 
     WHERE id = $2
     RETURNING id, username, email, first_name, last_name, role, permissions, is_active, created_at, updated_at`,
    [JSON.stringify(permissions), userId]
  );
  return result.rows[0];
}

export async function updateUserStatus(userId: number, isActive: boolean) {
  const result = await pool.query(
    `UPDATE users 
     SET is_active = $1, updated_at = NOW() 
     WHERE id = $2
     RETURNING id, username, email, first_name, last_name, role, permissions, is_active, created_at, updated_at`,
    [isActive, userId]
  );
  return result.rows[0];
}

export async function deleteUser(userId: number) {
  await pool.query('DELETE FROM users WHERE id = $1', [userId]);
}
