import { pool } from './postgres';
import { createOdooClient } from './odoo-client';

interface OrderToSync {
  id: number;
  orderNumber: number;
  orderCode: string;
  source: string;
  sourceOrderId: string;
  orderDate: Date | null;
  status: string | null;
  buyerLogin: string | null;
  buyerEmail: string | null;
  buyerCompany: string | null;
  buyerFirstName: string | null;
  buyerLastName: string | null;
  buyerPhone: string | null;
  buyerAddress: string | null;
  buyerZip: string | null;
  buyerCity: string | null;
  buyerCountryCode: string | null;
  paymentId: string | null;
  paymentStatus: string | null;
  paymentProvider: string | null;
  paymentAmount: string | null;
  paymentCurrency: string | null;
  deliveryMethod: string | null;
  deliveryAmount: string | null;
  deliveryCurrency: string | null;
  totalToPayAmount: string | null;
  totalToPayCurrency: string | null;
  buyerNotes: string | null;
  invoiceRequired: boolean | null;
  trackingNumbers: string[] | null;
  items?: Array<{
    name: string;
    quantity: string;
    price: string;
  }>;
}

export async function addOrderToSyncQueue(
  orderCode: string,
  orderId: number,
  source: string,
  operation: 'create' | 'update' = 'create'
): Promise<void> {
  const client = await pool.connect();

  try {
    // Sprawdź czy zamówienie NIE jest właśnie przetwarzane lub już zakończone niedawno (< 2 min)
    const checkResult = await client.query(`
      SELECT status, processed_at FROM odoo_sync_queue
      WHERE order_code = $1
      LIMIT 1
    `, [orderCode]);

    if (checkResult.rows.length > 0) {
      const { status, processed_at } = checkResult.rows[0];
      
      // Jeśli zamówienie zostało zakończone w ciągu ostatnich 2 minut, nie dodawaj ponownie
      if (status === 'completed' && processed_at) {
        const completedAt = new Date(processed_at);
        const now = new Date();
        const minutesAgo = (now.getTime() - completedAt.getTime()) / 1000 / 60;
        if (minutesAgo < 2) {
          // console.log(`⏭️  Pominięto ${orderCode} - niedawno zakończone (${minutesAgo.toFixed(1)} min temu)`);
          return;
        }
      }
    }

    await client.query(`
      INSERT INTO odoo_sync_queue (
        order_code, order_id, source, operation, status, attempts
      ) VALUES ($1, $2, $3, $4, 'pending', 0)
      ON CONFLICT (order_code) DO UPDATE SET
        operation = EXCLUDED.operation,
        status = CASE
          WHEN odoo_sync_queue.status IN ('pending', 'failed') THEN 'pending'
          ELSE odoo_sync_queue.status
        END,
        attempts = CASE
          WHEN odoo_sync_queue.status IN ('pending', 'failed') THEN 0
          ELSE odoo_sync_queue.attempts
        END,
        scheduled_at = NOW(),
        updated_at = NOW(),
        odoo_order_id = CASE 
          WHEN EXCLUDED.operation = 'update' THEN odoo_sync_queue.odoo_order_id
          ELSE NULL
        END
      WHERE odoo_sync_queue.status NOT IN ('completed')
        OR odoo_sync_queue.processed_at < NOW() - INTERVAL '2 minutes'
    `, [orderCode, orderId, source, operation]);

    console.log(`📋 Added ${orderCode} to Odoo sync queue (${operation})`);
  } catch (error) {
    console.error(`❌ Failed to add ${orderCode} to sync queue:`, error);
  } finally {
    client.release();
  }
}

export async function addOrdersToSyncQueue(orderCodes: string[]): Promise<void> {
  const client = await pool.connect();

  try {
    for (const orderCode of orderCodes) {
      const orderResult = await client.query(`
        SELECT order_number, order_code, source
        FROM commerce.orders
        WHERE order_code = $1
      `, [orderCode]);

      if (orderResult.rows.length > 0) {
        const order = orderResult.rows[0];
        await addOrderToSyncQueue(order.order_code, order.order_number, order.source);
      }
    }
  } finally {
    client.release();
  }
}

export async function processSyncQueue(): Promise<{
  processed: number;
  succeeded: number;
  failed: number;
}> {
  console.log('🔧 processSyncQueue() started');
  const odooClient = await createOdooClient();
  console.log('🔧 createOdooClient() returned:', odooClient ? 'OdooClient instance' : 'null');
  
  if (!odooClient) {
    console.warn('⚠️  Odoo client not configured, skipping sync');
    return { processed: 0, succeeded: 0, failed: 0 };
  }

  const client = await pool.connect();
  let processed = 0;
  let succeeded = 0;
  let failed = 0;

  try {
    const configResult = await client.query(`
      SELECT auto_confirm_orders FROM odoo_config WHERE id = 1
    `);
    const autoConfirmOrders = configResult.rows[0]?.auto_confirm_orders || false;

    // Użyj SELECT FOR UPDATE SKIP LOCKED żeby zapobiec duplikatom
    const queueResult = await client.query(`
      SELECT * FROM odoo_sync_queue
      WHERE status = 'pending' AND attempts < max_attempts
      ORDER BY scheduled_at ASC
      LIMIT 50
      FOR UPDATE SKIP LOCKED
    `);

    for (const queueItem of queueResult.rows) {
      processed++;
      const startTime = Date.now();

      try {
        const orderResult = await client.query(`
          SELECT 
            o.id,
            o.order_number as "orderNumber",
            o.order_code as "orderCode",
            o.source,
            o.source_order_id as "sourceOrderId",
            o.order_date as "orderDate",
            o.status,
            o.buyer_login as "buyerLogin",
            o.buyer_email as "buyerEmail",
            o.buyer_company as "buyerCompany",
            o.buyer_first_name as "buyerFirstName",
            o.buyer_last_name as "buyerLastName",
            o.buyer_phone as "buyerPhone",
            o.buyer_address as "buyerAddress",
            o.buyer_zip as "buyerZip",
            o.buyer_city as "buyerCity",
            o.buyer_country_code as "buyerCountryCode",
            o.delivery_method as "deliveryMethod",
            o.delivery_amount as "deliveryPrice",
            o.invoice_required as "invoiceRequired",
            o.total_to_pay_amount as "totalToPayAmount",
            COALESCE(
              json_agg(
                json_build_object(
                  'name', oi.offer_name,
                  'quantity', oi.quantity::text,
                  'price', oi.unit_price::text
                )
                ORDER BY oi.id
              ) FILTER (WHERE oi.id IS NOT NULL),
              '[]'
            ) as items
          FROM commerce.orders o
          LEFT JOIN commerce.order_items oi ON o.id = oi.order_id
          WHERE o.order_number = $1
          GROUP BY o.id, o.order_number, o.order_code, o.source, o.source_order_id, 
                   o.order_date, o.status, o.buyer_login, o.buyer_email, o.buyer_company,
                   o.buyer_first_name, o.buyer_last_name, o.buyer_phone, o.buyer_address,
                   o.buyer_zip, o.buyer_city, o.buyer_country_code, o.delivery_method,
                   o.delivery_amount, o.invoice_required, o.total_to_pay_amount
        `, [queueItem.order_id]);

        if (orderResult.rows.length === 0) {
          throw new Error('Order not found');
        }

        const order = orderResult.rows[0] as OrderToSync;

        let odooOrderId: number;

        // PROSTA LOGIKA: 
        // - Jeśli zamówienie MA odoo_order_id → UPDATE
        // - Jeśli zamówienie NIE MA odoo_order_id → CREATE
        const hasOdooId = !!queueItem.odoo_order_id;
        const operation = hasOdooId ? 'update' : 'create';

        if (operation === 'create') {
          odooOrderId = await odooClient.createSaleOrder(order);
          
          // ZAPISZ odoo_order_id ZARAZ po utworzeniu, PRZED potwierdzeniem
          await client.query(`
            UPDATE odoo_sync_queue
            SET odoo_order_id = $1, updated_at = NOW()
            WHERE id = $2
          `, [odooOrderId, queueItem.id]);

          await client.query(`
            UPDATE commerce.orders
            SET odoo_order_id = $1, updated_at = NOW()
            WHERE order_code = $2
          `, [odooOrderId, queueItem.order_code]);
          
          if (autoConfirmOrders) {
            await odooClient.confirmSaleOrder(odooOrderId);
          }
        } else {
          if (!queueItem.odoo_order_id) {
            throw new Error('Odoo order ID not found for update operation');
          }

          // Pobierz historię zmian dla tego zamówienia
          const changesResult = await client.query(`
            SELECT 
              field_changed, old_value, new_value, detected_at
            FROM public.order_changes
            WHERE order_code = $1
            ORDER BY detected_at DESC
            LIMIT 20
          `, [queueItem.order_code]);

          // Przygotuj notatkę z historią zmian
          let note = `Updated from Alpma OMS at ${new Date().toISOString()}`;
          
          if (changesResult.rows.length > 0) {
            note += '\n\n=== Historia zmian ===\n';
            const fieldLabels: Record<string, string> = {
              status: 'Status zamówienia',
              payment_status: 'Status płatności',
              payment_amount: 'Kwota płatności',
              refund_amount: 'Kwota zwrotu',
              has_returns: 'Zwroty produktów',
              tracking_numbers: 'Numery przesyłek',
              buyer_address: 'Adres kupującego',
              buyer_city: 'Miasto',
              buyer_zip: 'Kod pocztowy',
            };

            for (const change of changesResult.rows) {
              const fieldName = fieldLabels[change.field_changed] || change.field_changed;
              const date = new Date(change.detected_at).toLocaleString('pl-PL');
              note += `\n[${date}] ${fieldName}: "${change.old_value}" → "${change.new_value}"`;
            }
          }

          await odooClient.updateSaleOrder(queueItem.odoo_order_id, {
            note: note,
          });
          odooOrderId = queueItem.odoo_order_id;
        }

        const duration = Date.now() - startTime;

        await client.query(`
          UPDATE odoo_sync_queue
          SET status = 'completed', processed_at = NOW(), odoo_order_id = $1, updated_at = NOW()
          WHERE id = $2
        `, [odooOrderId, queueItem.id]);

        await client.query(`
          UPDATE commerce.orders
          SET odoo_order_id = $1, updated_at = NOW()
          WHERE order_code = $2
        `, [odooOrderId, queueItem.order_code]);

        await client.query(`
          INSERT INTO odoo_sync_logs (
            queue_id, order_code, operation, status, message,
            request_payload, odoo_order_id, duration
          ) VALUES ($1, $2, $3, 'success', $4, $5, $6, $7)
        `, [
          queueItem.id,
          queueItem.order_code,
          queueItem.operation,
          `Successfully synced to Odoo order #${odooOrderId}`,
          JSON.stringify(order),
          odooOrderId,
          duration,
        ]);

        succeeded++;
        console.log(`✅ Synced ${queueItem.order_code} to Odoo (order #${odooOrderId}) in ${duration}ms`);
      } catch (error: unknown) {
        failed++;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        const duration = Date.now() - startTime;

        // Jeśli zamówienie zostało usunięte z Odoo, automatycznie zmień na CREATE
        const isDeletedInOdoo = errorMessage.includes('Record does not exist') || 
                                errorMessage.includes('has been deleted');
        
        if (isDeletedInOdoo) {
          console.log(`🔄 Zamówienie ${queueItem.order_code} usunięte z Odoo - zmieniam na CREATE`);
          await client.query(`
            UPDATE odoo_sync_queue
            SET 
              operation = 'create',
              odoo_order_id = NULL,
              attempts = 0,
              status = 'pending',
              last_error = NULL,
              updated_at = NOW()
            WHERE id = $1
          `, [queueItem.id]);
          
          // Wyczyść też odoo_order_id w commerce.orders
          await client.query(`
            UPDATE commerce.orders
            SET odoo_order_id = NULL
            WHERE order_code = $1
          `, [queueItem.order_code]);
          
          continue; // Pomiń logowanie błędu - to nie jest błąd, tylko automatyczna naprawa
        }

        await client.query(`
          UPDATE odoo_sync_queue
          SET 
            status = CASE 
              WHEN attempts + 1 >= max_attempts THEN 'failed'
              ELSE 'pending'
            END,
            attempts = attempts + 1,
            last_error = $1,
            updated_at = NOW()
          WHERE id = $2
        `, [errorMessage, queueItem.id]);

        await client.query(`
          INSERT INTO odoo_sync_logs (
            queue_id, order_code, operation, status, message, error_details, duration
          ) VALUES ($1, $2, $3, 'error', $4, $5, $6)
        `, [
          queueItem.id,
          queueItem.order_code,
          queueItem.operation,
          errorMessage,
          JSON.stringify({ error: errorMessage, attempt: queueItem.attempts + 1 }),
          duration,
        ]);

        console.error(`❌ Failed to sync ${queueItem.order_code} to Odoo (attempt ${queueItem.attempts + 1}):`, errorMessage);
      }
    }
  } catch (error) {
    console.error('❌ Error processing Odoo sync queue:', error);
  } finally {
    client.release();
  }

  if (processed > 0) {
    console.log(`📊 Odoo sync: ${succeeded} succeeded, ${failed} failed out of ${processed} processed`);
  }

  return { processed, succeeded, failed };
}

export async function syncRecentOrdersToOdoo(timeWindowMinutes: number = 10): Promise<void> {
  const client = await pool.connect();

  try {
    const result = await client.query(`
      SELECT order_number, order_code, source
      FROM commerce.orders
      WHERE (updated_at >= NOW() - INTERVAL '${timeWindowMinutes} minutes'
         OR created_at >= NOW() - INTERVAL '${timeWindowMinutes} minutes')
        AND order_code IS NOT NULL
      ORDER BY 
        CASE 
          WHEN updated_at > created_at THEN updated_at 
          ELSE created_at 
        END DESC
    `);

    console.log(`🔄 Found ${result.rows.length} orders updated in last ${timeWindowMinutes} minutes for Odoo sync`);

    for (const order of result.rows) {
      // PROSTA LOGIKA: Sprawdź czy zamówienie ma odoo_order_id w commerce.orders
      const odooCheck = await client.query(`
        SELECT odoo_order_id FROM commerce.orders
        WHERE order_code = $1
        LIMIT 1
      `, [order.order_code]);

      // Jeśli zamówienie ma odoo_order_id → UPDATE, jeśli nie → CREATE
      const operation = odooCheck.rows[0]?.odoo_order_id ? 'update' : 'create';

      await addOrderToSyncQueue(order.order_code, order.order_number, order.source, operation);
    }

    await processSyncQueue();
  } catch (error) {
    console.error('❌ Error syncing recent orders to Odoo:', error);
  } finally {
    client.release();
  }
}

export async function getOdooSyncStatus(): Promise<{
  pending: number;
  processing: number;
  completed: number;
  failed: number;
  lastSyncAt: string | null;
}> {
  const client = await pool.connect();

  try {
    const statsResult = await client.query(`
      SELECT 
        COUNT(*) FILTER (WHERE status = 'pending') as pending,
        COUNT(*) FILTER (WHERE status = 'processing') as processing,
        COUNT(*) FILTER (WHERE status = 'completed') as completed,
        COUNT(*) FILTER (WHERE status = 'failed') as failed,
        MAX(processed_at) as last_sync_at
      FROM odoo_sync_queue
    `);

    const stats = statsResult.rows[0];

    return {
      pending: parseInt(stats.pending || '0'),
      processing: parseInt(stats.processing || '0'),
      completed: parseInt(stats.completed || '0'),
      failed: parseInt(stats.failed || '0'),
      lastSyncAt: stats.last_sync_at,
    };
  } finally {
    client.release();
  }
}

export async function getOdooSyncLogs(limit: number = 100): Promise<any[]> {
  const client = await pool.connect();

  try {
    const result = await client.query(`
      SELECT * FROM odoo_sync_logs
      ORDER BY created_at DESC
      LIMIT $1
    `, [limit]);

    return result.rows;
  } finally {
    client.release();
  }
}

export async function retryFailedSync(queueId: number): Promise<void> {
  const client = await pool.connect();

  try {
    await client.query(`
      UPDATE odoo_sync_queue
      SET status = 'pending', attempts = 0, last_error = NULL, updated_at = NOW()
      WHERE id = $1
    `, [queueId]);

    console.log(`🔄 Reset queue item ${queueId} for retry`);
  } finally {
    client.release();
  }
}
