import xmlrpc from 'xmlrpc';

interface CommerceOrder {
  id: number;
  orderNumber: number;
  orderCode: string;
  source: string;
  sourceOrderId: string;
  orderDate: Date | null;
  status: string | null;
  buyerLogin: string | null;
  buyerEmail: string | null;
  buyerCompany: string | null;
  buyerFirstName: string | null;
  buyerLastName: string | null;
  buyerPhone: string | null;
  buyerAddress: string | null;
  buyerZip: string | null;
  buyerCity: string | null;
  buyerCountryCode: string | null;
  paymentId: string | null;
  paymentStatus: string | null;
  paymentProvider: string | null;
  paymentAmount: string | null;
  paymentCurrency: string | null;
  deliveryMethod: string | null;
  deliveryAmount: string | null;
  deliveryCurrency: string | null;
  totalToPayAmount: string | null;
  totalToPayCurrency: string | null;
  buyerNotes: string | null;
  invoiceRequired: boolean | null;
  trackingNumbers: string[] | null;
  items?: Array<{
    name: string;
    quantity: string;
    price: string;
  }>;
}

interface OdooConfig {
  url: string;
  database: string;
  username: string;
  apiKey: string;
}

interface OdooOrderLine {
  product_id: number | false;
  name: string;
  product_uom_qty: number;
  price_unit: number;
}

interface OdooOrderData {
  partner_id: number | false;
  origin: string;
  client_order_ref: string;
  date_order: string;
  amount_total: number;
  order_line: [number, number, OdooOrderLine][];
  note?: string;
  // Własne pola z Odoo Studio (prefix x_)
  x_oms_order_url?: string;           // Link do zamówienia w OMS
  x_has_returns?: boolean;            // Czy ma zwroty
  x_invoice_required?: boolean;       // Czy faktura wymagana
  x_tracking_numbers?: string;        // Numery przesyłek
  x_buyer_notes?: string;             // Uwagi kupującego
  x_last_sync_date?: string;          // Ostatnia synchronizacja
  x_payment_status?: string;          // Status płatności
  x_refund_amount?: number;           // Kwota zwrotu
}

export class OdooClient {
  private config: OdooConfig;
  private uid: number | null = null;
  private commonClient: xmlrpc.Client;
  private objectClient: xmlrpc.Client;

  constructor(config: OdooConfig) {
    // Automatyczna migracja na nową domenę Cloudflare Tunnel alpma.app
    if (config.url.includes('tunnel.pyjam.as')) {
      // Migracja ze starej domeny Cloudflare
      config.url = config.url.replace('tunnel.pyjam.as', 'alpma.app');
      console.log(`✅ Automatycznie zaktualizowano URL Odoo: stara domena → ${config.url}`);
    } else if (config.url.includes('100.99.76.111')) {
      // Podmiana lokalnego IP na Cloudflare Tunnel z nową domeną
      config.url = 'https://synchronizacja.alpma.app/odoo';
      console.log(`✅ Automatycznie użyto Cloudflare Tunnel: ${config.url}`);
    }
    
    this.config = config;
    
    const url = new URL(this.config.url);
    const basePath = url.pathname.endsWith('/') ? url.pathname.slice(0, -1) : url.pathname;
    const commonPath = `${basePath}/xmlrpc/2/common`;
    const objectPath = `${basePath}/xmlrpc/2/object`;
    const isHttps = url.protocol === 'https:';
    const defaultPort = isHttps ? 443 : 8069;
    
    const clientOptions = {
      host: url.hostname,
      port: parseInt(url.port) || defaultPort,
      path: commonPath,
    };

    const objectOptions = {
      host: url.hostname,
      port: parseInt(url.port) || defaultPort,
      path: objectPath,
    };
    
    // Use secure client for HTTPS URLs
    if (isHttps) {
      this.commonClient = xmlrpc.createSecureClient(clientOptions);
      this.objectClient = xmlrpc.createSecureClient(objectOptions);
    } else {
      this.commonClient = xmlrpc.createClient(clientOptions);
      this.objectClient = xmlrpc.createClient(objectOptions);
    }
  }

  async authenticate(): Promise<number> {
    return new Promise((resolve, reject) => {
      this.commonClient.methodCall(
        'authenticate',
        [this.config.database, this.config.username, this.config.apiKey, {}],
        (error, value) => {
          if (error) {
            const errorMsg = error && typeof error === 'object' && 'message' in error ? String(error.message) : String(error);
            reject(new Error(`Odoo authentication failed: ${errorMsg}`));
          } else if (!value) {
            reject(new Error('Odoo authentication failed: Invalid credentials'));
          } else {
            this.uid = value as number;
            resolve(this.uid);
          }
        }
      );
    });
  }

  async execute(model: string, method: string, params: any[]): Promise<any> {
    if (!this.uid) {
      await this.authenticate();
    }

    return new Promise((resolve, reject) => {
      this.objectClient.methodCall(
        'execute_kw',
        [
          this.config.database,
          this.uid,
          this.config.apiKey,
          model,
          method,
          params,
        ],
        (error, value) => {
          if (error) {
            const errorMsg = error && typeof error === 'object' && 'message' in error ? String(error.message) : String(error);
            reject(new Error(`Odoo execute failed: ${errorMsg}`));
          } else {
            resolve(value);
          }
        }
      );
    });
  }

  async findOrCreatePartner(
    email: string, 
    name?: string,
    phone?: string,
    street?: string,
    zip?: string,
    city?: string,
    countryCode?: string
  ): Promise<number> {
    try {
      const partners = await this.execute('res.partner', 'search_read', [
        [['email', '=', email]],
        ['id', 'name', 'email'],
      ]);

      if (partners && partners.length > 0) {
        console.log(`👤 Found existing partner: ${partners[0].name} (ID: ${partners[0].id})`);
        return partners[0].id;
      }

      const partnerData: any = {
        name: name || email,
        email: email,
        customer_rank: 1,
      };

      if (phone) partnerData.phone = phone;
      if (street) partnerData.street = street;
      if (zip) partnerData.zip = zip;
      if (city) partnerData.city = city;
      if (countryCode) {
        const countries = await this.execute('res.country', 'search_read', [
          [['code', '=', countryCode.toUpperCase()]],
          ['id'],
        ]);
        if (countries && countries.length > 0) {
          partnerData.country_id = countries[0].id;
        }
      }

      const partnerId = await this.execute('res.partner', 'create', [partnerData]);

      console.log(`👤 Created new partner: ${name || email} (ID: ${partnerId})`);
      return partnerId;
    } catch (error: unknown) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      throw new Error(`Failed to find or create partner: ${errorMessage}`);
    }
  }

  async findOrCreateProduct(productName: string, price: number): Promise<number> {
    try {
      const products = await this.execute('product.product', 'search_read', [
        [['name', '=', productName]],
        ['id', 'name'],
      ]);

      if (products && products.length > 0) {
        console.log(`📦 Found existing product: ${products[0].name} (ID: ${products[0].id})`);
        return products[0].id;
      }

      const productId = await this.execute('product.product', 'create', [
        {
          name: productName,
          type: 'consu',
          list_price: price,
          sale_ok: true,
          purchase_ok: false,
        },
      ]);

      console.log(`📦 Created new product: ${productName} (ID: ${productId}) - Price: ${price} PLN`);
      return productId;
    } catch (error: unknown) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      throw new Error(`Failed to find or create product: ${errorMessage}`);
    }
  }

  async createSaleOrder(order: CommerceOrder): Promise<number> {
    try {
      const customerName = [
        order.buyerFirstName,
        order.buyerLastName
      ].filter(Boolean).join(' ') || order.buyerLogin || order.buyerEmail || 'Unknown Customer';

      const partnerId = await this.findOrCreatePartner(
        order.buyerEmail || 'unknown@example.com',
        customerName,
        order.buyerPhone || undefined,
        order.buyerAddress || undefined,
        order.buyerZip || undefined,
        order.buyerCity || undefined,
        order.buyerCountryCode || undefined
      );

      const orderLines: [number, number, OdooOrderLine][] = [];

      if (order.items && order.items.length > 0) {
        for (const item of order.items) {
          const productPrice = parseFloat(item.price || '0');
          const productId = await this.findOrCreateProduct(
            item.name || 'Unknown Product',
            productPrice
          );

          orderLines.push([
            0,
            0,
            {
              product_id: productId,
              name: item.name || 'Unknown Product',
              product_uom_qty: parseInt(item.quantity || '1'),
              price_unit: productPrice,
            },
          ]);
        }
      }

      // Konwertuj datę zamówienia (może być string lub Date)
      const orderDate = order.orderDate ? new Date(order.orderDate) : new Date();
      const odooDateFormat = orderDate.toISOString()
        .replace('T', ' ')
        .replace(/\.\d{3}Z$/, '');
      
      console.log(`📅 [ODOO DATE DEBUG] ${order.orderCode}:`);
      console.log(`   Raw orderDate from DB: ${order.orderDate} (type: ${typeof order.orderDate})`);
      console.log(`   Converted to Date: ${orderDate.toISOString()}`);
      console.log(`   Final Odoo format: ${odooDateFormat}`);

      const orderData: OdooOrderData = {
        partner_id: partnerId,
        origin: order.source || 'ALPMA',
        client_order_ref: order.orderCode || '',
        date_order: odooDateFormat,
        amount_total: parseFloat(order.totalToPayAmount || '0'),
        order_line: orderLines,
        note: `Imported from Alpma OMS - ${order.source} Order #${order.orderNumber}`,
        // WŁASNE POLA WYŁĄCZONE - dodaj je w Odoo Studio, a potem odkomentuj poniższy kod:
        // x_oms_order_url: `https://${process.env.REPLIT_DOMAINS?.split(',')[0] || 'alpma-oms.replit.app'}/?order=${order.orderCode}`,
        // x_has_returns: false,
        // x_invoice_required: order.invoiceRequired || false,
        // x_tracking_numbers: order.trackingNumbers?.join(', ') || '',
        // x_buyer_notes: order.buyerNotes || '',
        // x_last_sync_date: new Date().toISOString().replace('T', ' ').replace(/\.\d{3}Z$/, ''),
        // x_payment_status: order.paymentStatus || 'PENDING',
        // x_refund_amount: 0,
      };

      console.log(`🔍 [ODOO CREATE] Wysyłam do Odoo:`, JSON.stringify({
        order_code: order.orderCode,
        date_order: orderData.date_order,
        partner_id: orderData.partner_id,
        amount_total: orderData.amount_total
      }, null, 2));

      const orderId = await this.execute('sale.order', 'create', [orderData]);

      // Sprawdź jaką datę Odoo faktycznie zapisało
      const createdOrder = await this.execute('sale.order', 'read', [
        [orderId],
        ['id', 'name', 'date_order', 'create_date']
      ]);
      
      console.log(`✅ Created Odoo sale order ${orderId} for ${order.orderCode}`);
      if (createdOrder && createdOrder[0]) {
        console.log(`🔍 [ODOO RESPONSE] Odoo zapisało:`, JSON.stringify({
          odoo_id: createdOrder[0].id,
          date_order: createdOrder[0].date_order,
          create_date: createdOrder[0].create_date
        }, null, 2));
      }
      return orderId;
    } catch (error: unknown) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      throw new Error(`Failed to create sale order: ${errorMessage}`);
    }
  }

  async confirmSaleOrder(odooOrderId: number): Promise<boolean> {
    try {
      const result = await this.execute('sale.order', 'action_confirm', [[odooOrderId]]);
      console.log(`✅ Confirmed Odoo sale order ${odooOrderId} (changed from quotation to sale order)`);
      return result;
    } catch (error: unknown) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      throw new Error(`Failed to confirm sale order: ${errorMessage}`);
    }
  }

  async updateSaleOrder(odooOrderId: number, updates: Partial<OdooOrderData>): Promise<boolean> {
    try {
      const result = await this.execute('sale.order', 'write', [
        [odooOrderId],
        updates,
      ]);

      console.log(`✅ Updated Odoo sale order ${odooOrderId}`);
      return result;
    } catch (error: unknown) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      throw new Error(`Failed to update sale order: ${errorMessage}`);
    }
  }

  async getSaleOrder(odooOrderId: number): Promise<any> {
    try {
      const orders = await this.execute('sale.order', 'read', [
        [odooOrderId],
        ['id', 'name', 'partner_id', 'amount_total', 'state'],
      ]);

      return orders && orders.length > 0 ? orders[0] : null;
    } catch (error: unknown) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      throw new Error(`Failed to get sale order: ${errorMessage}`);
    }
  }

  async testConnection(): Promise<{ success: boolean; version?: any; error?: string; details?: any }> {
    const details: any = {
      url: this.config.url,
      database: this.config.database,
      username: this.config.username,
      steps: []
    };

    try {
      // Step 1: Test version endpoint
      details.steps.push({ step: 'version', status: 'trying' });
      const version = await new Promise((resolve, reject) => {
        const timeout = setTimeout(() => {
          reject(new Error('Connection timeout (10s) - Odoo nie odpowiada'));
        }, 10000);

        this.commonClient.methodCall('version', [], (error, value) => {
          clearTimeout(timeout);
          if (error) reject(error);
          else resolve(value);
        });
      });
      details.steps.push({ step: 'version', status: 'success', value: version });

      // Step 2: Test authentication
      details.steps.push({ step: 'authenticate', status: 'trying' });
      await this.authenticate();
      details.steps.push({ step: 'authenticate', status: 'success', uid: this.uid });

      return { success: true, version, details };
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      details.steps.push({ 
        step: details.steps[details.steps.length - 1]?.step || 'unknown', 
        status: 'failed', 
        error: errorMessage 
      });

      return {
        success: false,
        error: errorMessage,
        details
      };
    }
  }
}

export async function createOdooClient(): Promise<OdooClient | null> {
  const apiKey = process.env.ODOO_API_KEY;

  if (!apiKey) {
    console.warn('⚠️  ODOO_API_KEY not configured');
    return null;
  }

  // Import pool to get config from database
  const { pool } = await import('./postgres.js');
  const client = await pool.connect();

  try {
    const result = await client.query(`
      SELECT url, database, username
      FROM odoo_config
      WHERE is_active = true
      LIMIT 1
    `);

    if (result.rows.length === 0) {
      console.warn('⚠️  No active Odoo configuration found');
      return null;
    }

    const config = result.rows[0];

    // Automatyczna migracja na nową domenę Cloudflare Tunnel
    let odooUrl = config.url;
    console.log(`🔍 DEBUG createOdooClient: Original URL = "${odooUrl}"`);
    console.log(`🔍 DEBUG createOdooClient: includes tunnel.pyjam.as = ${odooUrl.includes('tunnel.pyjam.as')}`);
    
    if (odooUrl.includes('tunnel.pyjam.as')) {
      odooUrl = odooUrl.replace('tunnel.pyjam.as', 'alpma.app');
      console.log(`✅ Automatycznie zaktualizowano URL Odoo: ${odooUrl}`);
    } else if (odooUrl.includes('100.99.76.111')) {
      odooUrl = 'https://synchronizacja.alpma.app/odoo';
      console.log(`✅ Automatycznie użyto Cloudflare Tunnel: ${odooUrl}`);
    }
    
    console.log(`🔍 DEBUG createOdooClient: Final URL = "${odooUrl}"`);

    return new OdooClient({
      url: odooUrl,
      database: config.database,
      username: config.username,
      apiKey,
    });
  } finally {
    client.release();
  }
}
