import axios from 'axios';
import fs from 'fs/promises';
import path from 'path';

const ALLEGRO_API_BASE = 'https://api.allegro.pl';
const ALLEGRO_AUTH_BASE = 'https://allegro.pl/auth/oauth';

let cachedToken: {
  token: string;
  expiresAt: Date;
} | null = null;

export async function getAllegroToken(): Promise<string> {
  const clientId = process.env.ALLEGRO_CLIENT_ID;
  const clientSecret = process.env.ALLEGRO_CLIENT_SECRET;

  if (!clientId || !clientSecret) {
    throw new Error('Allegro credentials not configured');
  }

  if (cachedToken && cachedToken.expiresAt > new Date()) {
    return cachedToken.token;
  }

  try {
    const response = await axios.post(
      `${ALLEGRO_AUTH_BASE}/token`,
      new URLSearchParams({
        grant_type: 'client_credentials',
      }),
      {
        auth: {
          username: clientId,
          password: clientSecret,
        },
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded',
        },
      }
    );

    const { access_token, expires_in } = response.data;
    const expiresAt = new Date(Date.now() + (expires_in - 60) * 1000);

    cachedToken = {
      token: access_token,
      expiresAt,
    };

    console.log('✅ Allegro token obtained, expires at:', expiresAt);
    return access_token;
  } catch (error: any) {
    console.error('❌ Failed to get Allegro token:', error.response?.data || error.message);
    throw new Error('Failed to authenticate with Allegro API');
  }
}

export async function searchProductByName(productName: string, userToken?: string): Promise<any> {
  const token = userToken || await getAllegroToken();

  try {
    const response = await axios.get(
      `${ALLEGRO_API_BASE}/sale/products`,
      {
        headers: {
          Authorization: `Bearer ${token}`,
          Accept: 'application/vnd.allegro.public.v1+json',
        },
        params: {
          phrase: productName,
          'language': 'pl-PL',
          limit: 1
        }
      }
    );

    return response.data;
  } catch (error: any) {
    console.error(`❌ Failed to search product "${productName}":`, error.response?.data || error.message);
    throw new Error(`Failed to search product: ${error.message}`);
  }
}

export async function fetchPaymentRefunds(orderId: string, userToken: string): Promise<any[]> {
  try {
    const response = await axios.get(
      `${ALLEGRO_API_BASE}/payments/refunds`,
      {
        headers: {
          Authorization: `Bearer ${userToken}`,
          Accept: 'application/vnd.allegro.public.v1+json',
        },
        params: {
          'order.id': orderId,
        }
      }
    );

    return response.data.refunds || [];
  } catch (error: any) {
    console.error(`❌ Failed to fetch payment refunds for order ${orderId}:`, error.response?.data || error.message);
    return [];
  }
}

export async function downloadProductImage(
  productName: string,
  externalId: string,
  userToken?: string
): Promise<string | null> {
  try {
    const searchResult = await searchProductByName(productName, userToken);
    const products = searchResult.products || [];

    if (products.length === 0) {
      console.log(`⚠️ No products found for "${productName}"`);
      return null;
    }

    const firstProduct = products[0];
    const images = firstProduct.images || [];

    if (images.length === 0) {
      console.log(`⚠️ No images found for product "${productName}"`);
      return null;
    }

    const firstImageUrl = images[0].url;
    
    const imageResponse = await axios.get(firstImageUrl, {
      responseType: 'arraybuffer',
    });

    const productsDir = path.join(process.cwd(), 'attached_assets', 'products');
    await fs.mkdir(productsDir, { recursive: true });

    const filename = `${externalId}_1.jpg`;
    const filepath = path.join(productsDir, filename);

    await fs.writeFile(filepath, imageResponse.data);
    console.log(`✅ Image saved: ${filename}`);

    return filename;
  } catch (error: any) {
    console.error(`❌ Failed to download image for product "${productName}":`, error.message);
    return null;
  }
}

export async function downloadProductImages(
  items: Array<{ productName: string; externalId: string }>,
  userToken?: string
): Promise<{ success: number; failed: number }> {
  let success = 0;
  let failed = 0;

  for (const item of items) {
    try {
      const result = await downloadProductImage(item.productName, item.externalId, userToken);
      if (result) {
        const { updateProductImageUrl } = await import('./postgres.js');
        await updateProductImageUrl(item.externalId, result);
        success++;
      } else {
        failed++;
      }
      await new Promise((resolve) => setTimeout(resolve, 200));
    } catch (error) {
      failed++;
    }
  }

  return { success, failed };
}

export async function downloadImageFromUrl(
  imageUrl: string,
  externalId: string
): Promise<string | null> {
  try {
    if (!imageUrl) {
      console.log(`⚠️ No image URL provided for product ${externalId}`);
      return null;
    }

    const imageResponse = await axios.get(imageUrl, {
      responseType: 'arraybuffer',
      timeout: 10000,
    });

    const productsDir = path.join(process.cwd(), 'attached_assets', 'products');
    await fs.mkdir(productsDir, { recursive: true });

    const filename = `${externalId}_1.jpg`;
    const filepath = path.join(productsDir, filename);

    await fs.writeFile(filepath, imageResponse.data);
    console.log(`✅ Image saved: ${filename}`);

    return filename;
  } catch (error: any) {
    console.error(`❌ Failed to download image for product ${externalId}:`, error.message);
    return null;
  }
}

export async function downloadProductImagesFromDatabase(
  items: Array<{ imageUrl: string | null; externalId: string }>
): Promise<{ success: number; failed: number; skipped: number }> {
  let success = 0;
  let failed = 0;
  let skipped = 0;

  const itemsWithUrls = items.filter(item => item.imageUrl);
  console.log(`📊 Processing ${itemsWithUrls.length}/${items.length} products with image URLs`);

  for (const item of itemsWithUrls) {
    try {
      const result = await downloadImageFromUrl(item.imageUrl!, item.externalId);
      if (result) {
        success++;
      } else {
        failed++;
      }
      
      await new Promise((resolve) => setTimeout(resolve, 100));
    } catch (error) {
      console.error(`❌ Error processing product ${item.externalId}:`, error);
      failed++;
    }
  }

  skipped = items.length - itemsWithUrls.length;
  if (skipped > 0) {
    console.log(`⏭️ Skipped ${skipped} products without image URLs`);
  }

  return { success, failed, skipped };
}
