# -*- coding: utf-8 -*-
"""Wizard do synchronizacji zamówień z Shoper"""

import logging
from datetime import datetime, timedelta
from odoo import models, fields, api
from odoo.exceptions import UserError

_logger = logging.getLogger(__name__)


class ShoperSyncWizard(models.TransientModel):
    """Wizard do kontrolowanej synchronizacji"""
    _name = 'shoper.sync.wizard'
    _description = 'Shoper Synchronization Wizard'
    
    sync_type = fields.Selection([
        ('new', 'New Orders Only'),
        ('recent', 'Recent Updates'),
        ('full', 'Full Synchronization'),
        ('custom', 'Custom Range')
    ], string='Sync Type', default='new', required=True)
    
    # Dla nowych zamówień
    pages_to_check = fields.Integer(
        string='Pages to Check',
        default=5,
        help='Number of last pages to check for new orders'
    )
    
    # Dla aktualizacji
    hours_back = fields.Integer(
        string='Hours Back',
        default=24,
        help='Update orders changed in last N hours'
    )
    
    # Dla custom range
    date_from = fields.Datetime(
        string='From Date',
        default=lambda self: fields.Datetime.now() - timedelta(days=7)
    )
    date_to = fields.Datetime(
        string='To Date',
        default=fields.Datetime.now
    )
    
    # Opcje
    update_existing = fields.Boolean(
        string='Update Existing Orders',
        default=True,
        help='Update orders that already exist in database'
    )
    create_sale_orders = fields.Boolean(
        string='Create Sale Orders',
        default=False,
        help='Automatically create sale orders for new orders'
    )
    
    # Statystyki
    last_sync_info = fields.Text(
        string='Last Sync Info',
        compute='_compute_last_sync_info'
    )
    
    @api.depends('sync_type')
    def _compute_last_sync_info(self):
        for wizard in self:
            OrderSync = self.env['shoper.order.sync']
            
            # Ostatnia synchronizacja
            last_sync = OrderSync.search([], order='last_sync_at desc', limit=1)
            
            # Statystyki
            total_orders = OrderSync.search_count([])
            paid_orders = OrderSync.search_count([('paid', '=', True)])
            recent_changes = self.env['shoper.order.change'].search_count([
                ('changed_at', '>=', fields.Datetime.now() - timedelta(hours=24))
            ])
            
            info = f"""
📊 Database Statistics:
• Total orders: {total_orders}
• Paid orders: {paid_orders}
• Changes (24h): {recent_changes}
"""
            
            if last_sync:
                info += f"""
• Last sync: {last_sync.last_sync_at}
• Last order ID: {last_sync.order_id}
"""
            
            wizard.last_sync_info = info
    
    def action_sync(self):
        """Wykonuje synchronizację"""
        self.ensure_one()
        
        # Sprawdź konfigurację
        config = self.env['shoper.config'].get_active_config()
        if not config:
            raise UserError('No active Shoper configuration found')
        
        OrderSync = self.env['shoper.order.sync']
        
        # Rozpocznij synchronizację
        _logger.info(f"🚀 Starting synchronization: {self.sync_type}")
        
        try:
            if self.sync_type == 'new':
                # Synchronizuj nowe zamówienia
                stats = OrderSync.sync_new_orders(limit_pages=self.pages_to_check)
                
            elif self.sync_type == 'recent':
                # Synchronizuj ostatnie zmiany
                stats = OrderSync.sync_recent_updates(hours_back=self.hours_back)
                
            elif self.sync_type == 'full':
                # Pełna synchronizacja
                stats_new = OrderSync.sync_new_orders(limit_pages=20)
                stats_update = OrderSync.sync_recent_updates(hours_back=72)
                
                stats = {
                    'new': stats_new.get('new', 0),
                    'updated': stats_new.get('updated', 0) + stats_update.get('updated', 0),
                    'unchanged': stats_new.get('unchanged', 0) + stats_update.get('unchanged', 0),
                    'errors': stats_new.get('errors', 0) + stats_update.get('errors', 0)
                }
                
            elif self.sync_type == 'custom':
                # Custom range - pobierz zamówienia z zakresu
                client = config._get_api_client()
                
                # Tu możemy dodać filtrowanie po datach
                # Na razie synchronizujemy wszystkie i filtrujemy lokalnie
                orders = OrderSync.search([
                    ('date_add', '>=', self.date_from),
                    ('date_add', '<=', self.date_to)
                ])
                
                stats = {'updated': 0, 'unchanged': 0, 'errors': 0}
                
                for order in orders:
                    try:
                        order_data = client.get_order_details(order.order_id)
                        if order_data:
                            status, record = OrderSync.sync_single_order(order_data)
                            stats[status] = stats.get(status, 0) + 1
                    except Exception as e:
                        _logger.error(f"Error syncing order #{order.order_id}: {e}")
                        stats['errors'] += 1
            
            # Podsumowanie
            message = f"""
✅ Synchronization Complete!

📥 New orders: {stats.get('new', 0)}
🔄 Updated: {stats.get('updated', 0)}
✓ Unchanged: {stats.get('unchanged', 0)}
❌ Errors: {stats.get('errors', 0)}
"""
            
            # Utwórz sale orders jeśli zaznaczono
            if self.create_sale_orders and stats.get('new', 0) > 0:
                new_orders = OrderSync.search([
                    ('sale_order_id', '=', False),
                    ('status', '!=', 'cancelled')
                ], limit=stats['new'])
                
                created = 0
                for order in new_orders:
                    try:
                        sale_order = self._create_sale_order(order)
                        if sale_order:
                            order.sale_order_id = sale_order.id
                            created += 1
                    except Exception as e:
                        _logger.error(f"Error creating sale order for #{order.order_id}: {e}")
                
                message += f"\n📋 Sale orders created: {created}"
            
            # Wyświetl podsumowanie
            return {
                'type': 'ir.actions.client',
                'tag': 'display_notification',
                'params': {
                    'title': 'Synchronization Complete',
                    'message': message,
                    'type': 'success',
                    'sticky': True,
                    'next': {
                        'type': 'ir.actions.act_window',
                        'res_model': 'shoper.order.sync',
                        'view_mode': 'tree,form',
                        'target': 'main',
                    }
                }
            }
            
        except Exception as e:
            _logger.error(f"Synchronization failed: {e}")
            raise UserError(f'Synchronization failed: {str(e)}')
    
    def _create_sale_order(self, shoper_order):
        """Tworzy sale.order z shoper.order.sync"""
        if not shoper_order.partner_id:
            return None
        
        # Przygotuj wartości
        vals = {
            'partner_id': shoper_order.partner_id.id,
            'date_order': shoper_order.date_add,
            'client_order_ref': shoper_order.order_number,
            'shoper_order_id': shoper_order.order_id,
        }
        
        # Utwórz zamówienie
        sale_order = self.env['sale.order'].create(vals)
        
        # Tu można dodać linie zamówienia jeśli mamy produkty
        # ...
        
        return sale_order
    
    @api.model
    def action_open_wizard(self):
        """Otwiera wizard"""
        return {
            'name': 'Synchronize Orders',
            'type': 'ir.actions.act_window',
            'res_model': 'shoper.sync.wizard',
            'view_mode': 'form',
            'target': 'new',
        }


class ShoperSyncScheduled(models.Model):
    """Rozszerzenie dla zaplanowanych synchronizacji"""
    _inherit = 'shoper.config'
    
    @api.model
    def cron_sync_orders(self):
        """Metoda dla crona - automatyczna synchronizacja"""
        config = self.get_active_config()
        if not config:
            _logger.warning("No active Shoper configuration for scheduled sync")
            return
        
        if not config.auto_sync:
            return
        
        try:
            OrderSync = self.env['shoper.order.sync']
            
            # Synchronizuj nowe zamówienia
            stats_new = OrderSync.sync_new_orders(limit_pages=3)
            
            # Synchronizuj ostatnie zmiany
            stats_update = OrderSync.sync_recent_updates(hours_back=6)
            
            total_changes = (
                stats_new.get('new', 0) + 
                stats_update.get('updated', 0)
            )
            
            if total_changes > 0:
                _logger.info(
                    f"🔄 Scheduled sync completed: "
                    f"{stats_new.get('new', 0)} new, "
                    f"{stats_update.get('updated', 0)} updated"
                )
                
                # Wyślij email z podsumowaniem jeśli skonfigurowano
                if config.sync_notification_email:
                    self._send_sync_notification(stats_new, stats_update)
                    
        except Exception as e:
            _logger.error(f"Scheduled sync failed: {e}")
    
    def _send_sync_notification(self, stats_new, stats_update):
        """Wysyła powiadomienie o synchronizacji"""
        # Tu można dodać wysyłkę emaila z podsumowaniem
        pass
