#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Skrypt diagnostyczny do testowania API Shoper
Sprawdza połączenie i dostępność zamówień
"""

import requests
import json
from datetime import datetime, timedelta

# =============================================================================
# KONFIGURACJA - UZUPEŁNIJ DANYMI Z ODOO
# =============================================================================

SHOP_URL = "https://alpmeb.pl"  # Twój URL sklepu
ACCESS_TOKEN = "TWÓJ_TOKEN_API"  # Token z Shoper API

# =============================================================================

class ShoperAPITester:
    def __init__(self, shop_url, access_token):
        self.shop_url = shop_url.rstrip('/')
        self.access_token = access_token
        self.api_base_url = f"{self.shop_url}/webapi/rest"
        self.headers = {
            'Authorization': f'Bearer {self.access_token}',
            'Content-Type': 'application/json'
        }
    
    def test_connection(self):
        """Test 1: Podstawowe połączenie"""
        print("\n" + "="*70)
        print("TEST 1: Połączenie z API")
        print("="*70)
        
        url = f"{self.api_base_url}/orders"
        params = {'limit': 1}
        
        try:
            response = requests.get(url, headers=self.headers, params=params, timeout=10)
            print(f"URL: {url}")
            print(f"Status Code: {response.status_code}")
            
            if response.status_code == 200:
                data = response.json()
                print("✅ Połączenie OK!")
                print(f"Struktura odpowiedzi: {list(data.keys())}")
                return True
            else:
                print(f"❌ Błąd: {response.status_code}")
                print(f"Odpowiedź: {response.text}")
                return False
        except Exception as e:
            print(f"❌ Wyjątek: {e}")
            return False
    
    def test_orders_count(self):
        """Test 2: Liczba zamówień"""
        print("\n" + "="*70)
        print("TEST 2: Liczba zamówień w sklepie")
        print("="*70)
        
        url = f"{self.api_base_url}/orders"
        params = {'limit': 1}
        
        try:
            response = requests.get(url, headers=self.headers, params=params, timeout=10)
            
            if response.status_code == 200:
                data = response.json()
                count = data.get('count', 0)
                print(f"✅ Całkowita liczba zamówień: {count}")
                
                if count == 0:
                    print("⚠️  UWAGA: Brak zamówień w sklepie!")
                    print("   Sprawdź czy zamówienia faktycznie istnieją w Shoper")
                
                return count
            else:
                print(f"❌ Błąd: {response.status_code}")
                return 0
        except Exception as e:
            print(f"❌ Wyjątek: {e}")
            return 0
    
    def test_orders_recent(self):
        """Test 3: Ostatnie zamówienia"""
        print("\n" + "="*70)
        print("TEST 3: Ostatnie zamówienia (10 najnowszych)")
        print("="*70)
        
        url = f"{self.api_base_url}/orders"
        params = {
            'limit': 10,
            'order': '-date'  # Sortuj po dacie malejąco
        }
        
        try:
            response = requests.get(url, headers=self.headers, params=params, timeout=10)
            
            if response.status_code == 200:
                data = response.json()
                orders = data.get('list', [])
                
                print(f"✅ Znaleziono {len(orders)} zamówień")
                
                if orders:
                    print("\nPierwsze 5 zamówień:")
                    for i, order in enumerate(orders[:5], 1):
                        order_id = order.get('order_id')
                        date = order.get('date', 'brak')
                        updated = order.get('updated_at', 'brak')
                        email = order.get('email', 'brak')
                        total = order.get('sum', 0)
                        
                        print(f"\n{i}. Zamówienie #{order_id}")
                        print(f"   Data utworzenia: {date}")
                        print(f"   Ostatnia aktualizacja: {updated}")
                        print(f"   Email: {email}")
                        print(f"   Suma: {total}")
                else:
                    print("❌ Brak zamówień do wyświetlenia")
                
                return orders
            else:
                print(f"❌ Błąd: {response.status_code}")
                return []
        except Exception as e:
            print(f"❌ Wyjątek: {e}")
            return []
    
    def test_orders_with_date_filter(self):
        """Test 4: Zamówienia z filtrem daty"""
        print("\n" + "="*70)
        print("TEST 4: Zamówienia z ostatnich 30 dni")
        print("="*70)
        
        # Data 30 dni wstecz
        date_from = (datetime.now() - timedelta(days=30)).strftime('%Y-%m-%d 00:00:00')
        
        url = f"{self.api_base_url}/orders"
        params = {
            'limit': 10,
            'date[>=]': date_from,
            'order': '-date'
        }
        
        print(f"Filtr: date >= {date_from}")
        
        try:
            response = requests.get(url, headers=self.headers, params=params, timeout=10)
            
            if response.status_code == 200:
                data = response.json()
                count = data.get('count', 0)
                orders = data.get('list', [])
                
                print(f"✅ Znaleziono {count} zamówień z ostatnich 30 dni")
                print(f"   Zwrócono: {len(orders)} zamówień na tej stronie")
                
                if orders:
                    print("\nPrzykładowe zamówienie:")
                    order = orders[0]
                    print(f"   ID: {order.get('order_id')}")
                    print(f"   Data: {order.get('date')}")
                    print(f"   Updated: {order.get('updated_at')}")
                
                return count
            else:
                print(f"❌ Błąd: {response.status_code}")
                print(f"   URL: {url}")
                print(f"   Params: {params}")
                return 0
        except Exception as e:
            print(f"❌ Wyjątek: {e}")
            return 0
    
    def test_orders_with_updated_filter(self):
        """Test 5: Zamówienia zaktualizowane ostatnio"""
        print("\n" + "="*70)
        print("TEST 5: Zamówienia zaktualizowane w ostatnich 7 dniach")
        print("="*70)
        
        # Data 7 dni wstecz
        date_from = (datetime.now() - timedelta(days=7)).strftime('%Y-%m-%d 00:00:00')
        
        url = f"{self.api_base_url}/orders"
        params = {
            'limit': 10,
            'updated_at[>=]': date_from,
            'order': '-updated_at'
        }
        
        print(f"Filtr: updated_at >= {date_from}")
        
        try:
            response = requests.get(url, headers=self.headers, params=params, timeout=10)
            
            if response.status_code == 200:
                data = response.json()
                count = data.get('count', 0)
                orders = data.get('list', [])
                
                print(f"✅ Znaleziono {count} zamówień zaktualizowanych w ostatnich 7 dniach")
                print(f"   Zwrócono: {len(orders)} zamówień na tej stronie")
                
                if count == 0:
                    print("\n⚠️  UWAGA: To może być przyczyna problemu!")
                    print("   Odoo używa filtra updated_at do synchronizacji")
                    print("   Brak zamówień z updated_at oznacza brak synchronizacji")
                
                return count
            else:
                print(f"❌ Błąd: {response.status_code}")
                return 0
        except Exception as e:
            print(f"❌ Wyjątek: {e}")
            return 0
    
    def test_order_products(self, order_id):
        """Test 6: Produkty zamówienia"""
        print("\n" + "="*70)
        print(f"TEST 6: Produkty zamówienia #{order_id}")
        print("="*70)
        
        # Test nowego endpointa (poprawionego)
        url = f"{self.api_base_url}/order-products"
        filters = json.dumps({"order_id": order_id})
        params = {
            'filters': filters,
            'limit': 50
        }
        
        print(f"Endpoint: {url}")
        print(f"Filters: {filters}")
        
        try:
            response = requests.get(url, headers=self.headers, params=params, timeout=10)
            
            if response.status_code == 200:
                data = response.json()
                products = data.get('list', [])
                
                print(f"✅ Znaleziono {len(products)} produktów")
                
                if products:
                    print("\nPierwszy produkt:")
                    product = products[0]
                    print(f"   Nazwa: {product.get('name')}")
                    print(f"   Ilość: {product.get('quantity')}")
                    print(f"   Cena: {product.get('price')}")
                
                return products
            else:
                print(f"❌ Błąd: {response.status_code}")
                print(f"   Odpowiedź: {response.text}")
                return []
        except Exception as e:
            print(f"❌ Wyjątek: {e}")
            return []
    
    def run_all_tests(self):
        """Uruchom wszystkie testy"""
        print("\n" + "="*70)
        print("DIAGNOSTYKA API SHOPER")
        print("="*70)
        print(f"Shop URL: {self.shop_url}")
        print(f"API Base: {self.api_base_url}")
        print("="*70)
        
        # Test 1: Połączenie
        if not self.test_connection():
            print("\n❌ BŁĄD: Nie można połączyć się z API!")
            print("Sprawdź:")
            print("  1. URL sklepu (czy zawiera https://)")
            print("  2. Token API (czy jest aktualny)")
            print("  3. Połączenie internetowe")
            return
        
        # Test 2: Liczba zamówień
        total_orders = self.test_orders_count()
        
        if total_orders == 0:
            print("\n❌ PROBLEM: Brak zamówień w sklepie!")
            print("To wyjaśnia dlaczego synchronizacja zwraca 'No orders'")
            return
        
        # Test 3: Ostatnie zamówienia
        recent_orders = self.test_orders_recent()
        
        # Test 4: Zamówienia z ostatnich 30 dni
        orders_30d = self.test_orders_with_date_filter()
        
        # Test 5: Zamówienia zaktualizowane ostatnio
        updated_7d = self.test_orders_with_updated_filter()
        
        # Test 6: Produkty (jeśli są zamówienia)
        if recent_orders:
            first_order_id = recent_orders[0].get('order_id')
            if first_order_id:
                self.test_order_products(first_order_id)
        
        # Podsumowanie
        print("\n" + "="*70)
        print("PODSUMOWANIE")
        print("="*70)
        print(f"Całkowita liczba zamówień: {total_orders}")
        print(f"Zamówienia z ostatnich 30 dni: {orders_30d}")
        print(f"Zamówienia zaktualizowane (7 dni): {updated_7d}")
        print("")
        
        if total_orders == 0:
            print("❌ WNIOSEK: Brak zamówień w sklepie - to normalne że sync zwraca 'No orders'")
        elif orders_30d == 0:
            print("⚠️  WNIOSEK: Są zamówienia, ale żadne z ostatnich 30 dni")
            print("   Sprawdź pole 'Import From Date' w konfiguracji Odoo")
        elif updated_7d == 0:
            print("⚠️  WNIOSEK: Są zamówienia, ale żadne nie było aktualizowane ostatnio")
            print("   To może blokować incremental sync w Odoo")
        else:
            print("✅ WNIOSEK: API działa prawidłowo, są zamówienia do synchronizacji")
            print("   Problem może być w konfiguracji Odoo (filtry, daty)")
        
        print("="*70)


if __name__ == "__main__":
    print("""
    ╔═══════════════════════════════════════════════════════════════╗
    ║         SHOPER API - NARZĘDZIE DIAGNOSTYCZNE                 ║
    ╚═══════════════════════════════════════════════════════════════╝
    
    UWAGA: Przed uruchomieniem:
    1. Otwórz ten plik w edytorze
    2. Uzupełnij SHOP_URL i ACCESS_TOKEN na początku pliku
    3. Zapisz i uruchom ponownie
    """)
    
    # Sprawdź czy dane są uzupełnione
    if ACCESS_TOKEN == "TWÓJ_TOKEN_API":
        print("❌ BŁĄD: Musisz uzupełnić ACCESS_TOKEN w pliku!")
        print("   Znajdź linię: ACCESS_TOKEN = \"TWÓJ_TOKEN_API\"")
        print("   I zamień TWÓJ_TOKEN_API na rzeczywisty token z Shoper")
        exit(1)
    
    # Uruchom testy
    tester = ShoperAPITester(SHOP_URL, ACCESS_TOKEN)
    tester.run_all_tests()
    
    print("\n✅ Diagnostyka zakończona!")
    print("Jeśli masz pytania, udostępnij wyniki powyżej.\n")
