# -*- coding: utf-8 -*-
"""
Quick Check - Sprawdzenie konfiguracji Shoper w Odoo
Uruchom przez Odoo shell lub jako skrypt Python

Użycie z Odoo shell:
    odoo-bin shell -d your_database
    >>> from odoo_shoper_integration.quick_check import run_check
    >>> run_check(env)
"""

def run_check(env):
    """
    Sprawdza konfigurację Shoper i diagnozuje problemy
    
    Args:
        env: Odoo environment object
    """
    print("\n" + "="*70)
    print("QUICK CHECK - Konfiguracja Shoper Integration")
    print("="*70)
    
    # Znajdź aktywną konfigurację
    ShoperConfig = env['shoper.config']
    configs = ShoperConfig.search([('active', '=', True)])
    
    if not configs:
        print("❌ BŁĄD: Brak aktywnej konfiguracji!")
        print("   Rozwiązanie: Utwórz konfigurację w Shoper → Configuration")
        return
    
    config = configs[0]
    print(f"✅ Znaleziono aktywną konfigurację: {config.shop_url}")
    print("")
    
    # Sprawdzenia
    issues = []
    warnings = []
    
    # 1. Shop URL
    print("1. Shop URL:")
    print(f"   Wartość: {config.shop_url}")
    if not config.shop_url.startswith('https://'):
        issues.append("Shop URL nie zawiera 'https://'")
        print("   ❌ Brak 'https://' - to może powodować problemy")
    else:
        print("   ✅ OK")
    print("")
    
    # 2. API Token
    print("2. API Access Token:")
    if config.access_token:
        token_len = len(config.access_token)
        print(f"   Długość: {token_len} znaków")
        if token_len < 30:
            issues.append("Token API wygląda na zbyt krótki")
            print("   ⚠️  Token wygląda podejrzanie krótki")
        else:
            print("   ✅ OK")
    else:
        issues.append("Brak tokena API!")
        print("   ❌ BRAK TOKENA!")
    print("")
    
    # 3. Import From Date - NAJCZĘSTSZY PROBLEM!
    print("3. Import From Date:")
    if config.import_from_date:
        print(f"   Wartość: {config.import_from_date}")
        from datetime import datetime
        now = datetime.now()
        if config.import_from_date > now:
            issues.append("Import From Date jest w PRZYSZŁOŚCI!")
            print("   ❌ DATA W PRZYSZŁOŚCI - to blokuje import!")
        elif (now - config.import_from_date).days < 7:
            warnings.append("Import From Date jest bardzo niedawna")
            print(f"   ⚠️  Data tylko {(now - config.import_from_date).days} dni wstecz")
            print("      Starsze zamówienia nie będą importowane")
        else:
            print("   ✅ OK")
    else:
        print("   Wartość: [PUSTE]")
        print("   ✅ OK - importuje wszystkie zamówienia")
    print("")
    
    # 4. Last Order Update
    print("4. Last Order Update:")
    if config.last_order_update:
        print(f"   Wartość: {config.last_order_update}")
        from datetime import datetime
        now = datetime.now()
        if config.last_order_update > now:
            issues.append("Last Order Update jest w PRZYSZŁOŚCI!")
            print("   ❌ DATA W PRZYSZŁOŚCI - to blokuje sync!")
        else:
            print("   ✅ OK - incremental sync będzie działać")
    else:
        print("   Wartość: [PUSTE]")
        print("   ℹ️  Pierwszy sync będzie FULL (importuje wszystkie)")
    print("")
    
    # 5. Max Orders Per Sync
    print("5. Max Orders Per Sync:")
    print(f"   Wartość: {config.max_orders_per_sync}")
    if config.max_orders_per_sync < 10:
        warnings.append("Max Orders Per Sync jest bardzo niskie")
        print("   ⚠️  Bardzo niska wartość - zwiększ do 50-100")
    elif config.max_orders_per_sync > 200:
        warnings.append("Max Orders Per Sync jest bardzo wysokie")
        print("   ⚠️  Bardzo wysoka wartość - może powodować timeout")
    else:
        print("   ✅ OK")
    print("")
    
    # 6. Statystyki
    print("6. Statystyki:")
    print(f"   Ostatnia synchronizacja: {config.last_sync_date or 'NIGDY'}")
    print(f"   Całkowicie zaimportowano: {config.total_orders_imported}")
    print(f"   Ostatnio zaimportowano: {config.last_import_count}")
    print(f"   Ostatnio zaktualizowano: {config.last_update_count}")
    
    if config.total_orders_imported == 0:
        warnings.append("Żadne zamówienia nie zostały jeszcze zaimportowane")
        print("   ⚠️  Brak zaimportowanych zamówień")
    else:
        print("   ✅ Zamówienia były importowane")
    print("")
    
    # 7. Istniejące zamówienia Shoper w bazie
    print("7. Zamówienia Shoper w bazie Odoo:")
    SaleOrder = env['sale.order']
    shoper_orders = SaleOrder.search([('is_shoper_order', '=', True)])
    print(f"   Liczba: {len(shoper_orders)}")
    
    if shoper_orders:
        # Najstarsze i najnowsze
        oldest = min(shoper_orders, key=lambda o: o.shoper_date or '1970-01-01')
        newest = max(shoper_orders, key=lambda o: o.shoper_date or '1970-01-01')
        print(f"   Najstarsze: {oldest.shoper_date} (#{oldest.shoper_order_id})")
        print(f"   Najnowsze: {newest.shoper_date} (#{newest.shoper_order_id})")
        print("   ✅ OK")
    else:
        print("   ⚠️  Brak zamówień Shoper w bazie")
    print("")
    
    # Podsumowanie
    print("="*70)
    print("PODSUMOWANIE")
    print("="*70)
    
    if issues:
        print("\n❌ ZNALEZIONO KRYTYCZNE PROBLEMY:")
        for i, issue in enumerate(issues, 1):
            print(f"   {i}. {issue}")
    
    if warnings:
        print("\n⚠️  OSTRZEŻENIA:")
        for i, warning in enumerate(warnings, 1):
            print(f"   {i}. {warning}")
    
    if not issues and not warnings:
        print("\n✅ Wszystko wygląda dobrze!")
        print("   Jeśli sync nadal nie działa, sprawdź:")
        print("   - Czy token API jest aktualny (nie wygasł)")
        print("   - Czy w Shoper faktycznie są zamówienia")
        print("   - Logi Odoo pod kątem błędów API")
    
    print("\n" + "="*70)
    
    # Sugestie akcji
    if issues or warnings:
        print("\nSUGEROWANE AKCJE:")
        print("")
        
        if "Import From Date" in str(issues) or any("Import From Date" in w for w in warnings):
            print("1. WYCZYŚĆ Import From Date:")
            print("   - Otwórz Shoper → Configuration")
            print("   - Usuń datę z pola 'Import From Date'")
            print("   - Zapisz i uruchom sync ponownie")
            print("")
        
        if "Last Order Update" in str(issues):
            print("2. WYCZYŚĆ Last Order Update:")
            print("   - Włącz tryb developer w Odoo")
            print("   - Otwórz konfigurację")
            print("   - Wyczyść pole 'Last Order Update'")
            print("   - Zapisz")
            print("")
        
        if "Token" in str(issues):
            print("3. WYGENERUJ NOWY TOKEN:")
            print("   - Zaloguj się do panelu Shoper")
            print("   - Settings → API")
            print("   - Wygeneruj nowy token")
            print("   - Skopiuj do konfiguracji Odoo")
            print("")
    
    print("="*70 + "\n")


if __name__ == "__main__":
    """
    Uruchamianie z Odoo shell:
    
    docker exec -it alpsys-odoo-web-1 odoo shell -d postgres
    
    Następnie w shellu:
    >>> import sys
    >>> sys.path.append('/mnt/extra-addons/odoo_shoper_integration')
    >>> from quick_check import run_check
    >>> run_check(env)
    """
    print("Ten skrypt należy uruchomić z Odoo shell!")
    print("Instrukcje znajdują się w komentarzach na początku pliku.")
