# -*- coding: utf-8 -*-
"""
Rozszerzone modele synchronizacji - dodanie automatycznego tworzenia Sale Orders
"""

import logging
import json
from datetime import datetime
from odoo import models, fields, api
from odoo.exceptions import UserError

_logger = logging.getLogger(__name__)

class ShoperOrderSync(models.Model):
    _inherit = 'shoper.order.sync'
    
    # Dodajemy pole do linkowania z sale.order
    sale_order_id = fields.Many2one(
        'sale.order',
        string='Sale Order',
        ondelete='set null',
        readonly=True
    )
    
    def create_sale_order(self):
        """Tworzy zamówienie sprzedaży na podstawie danych z Shopera"""
        self.ensure_one()
        
        if self.sale_order_id:
            raise UserError(f'Sale Order already exists: {self.sale_order_id.name}')
        
        # Znajdź lub utwórz partnera
        partner = self._get_or_create_partner()
        
        # Przygotuj dane Sale Order
        sale_order_vals = {
            'partner_id': partner.id,
            'date_order': self.date_add or fields.Datetime.now(),
            'state': 'sale' if self.paid else 'draft',
            'origin': f'Shoper #{self.order_id}',
            'client_order_ref': self.order_number,
            'note': self._prepare_order_note(),
        }
        
        # Utwórz Sale Order
        sale_order = self.env['sale.order'].create(sale_order_vals)
        _logger.info(f"Created Sale Order {sale_order.name} for Shoper order #{self.order_id}")
        
        # Dodaj produkty
        self._create_sale_order_lines(sale_order)
        
        # Zapisz referencję
        self.sale_order_id = sale_order.id
        
        # Potwierdź zamówienie jeśli opłacone
        if self.paid:
            sale_order.action_confirm()
        
        return sale_order
    
    def _get_or_create_partner(self):
        """Znajduje lub tworzy partnera dla zamówienia"""
        if self.partner_id:
            return self.partner_id
        
        # Szukaj po emailu
        if self.email:
            partner = self.env['res.partner'].search([('email', '=', self.email)], limit=1)
            if partner:
                self.partner_id = partner
                return partner
        
        # Utwórz nowego partnera
        partner_vals = {
            'name': f"{self.firstname or ''} {self.surname or ''}".strip() or self.email or 'Unknown Customer',
            'email': self.email,
            'phone': self.phone,
            'customer_rank': 1,
            'is_company': bool(self.company_name),
        }
        
        if self.company_name:
            partner_vals['name'] = self.company_name
            partner_vals['vat'] = self.nip
        
        partner = self.env['res.partner'].create(partner_vals)
        self.partner_id = partner
        _logger.info(f"Created new partner: {partner.name}")
        
        return partner
    
    def _create_sale_order_lines(self, sale_order):
        """Tworzy linie zamówienia sprzedaży"""
        # Znajdź produkt domyślny (usługa)
        service = self.env['product.product'].search([
            ('name', '=', 'Shoper Order'),
            ('type', '=', 'service')
        ], limit=1)
        
        if not service:
            # Utwórz produkt jeśli nie istnieje
            service = self.env['product.product'].create({
                'name': 'Shoper Order',
                'type': 'service',
                'list_price': 0,
                'sale_ok': True,
                'purchase_ok': False,
            })
        
        # Utwórz linię zamówienia
        line_vals = {
            'order_id': sale_order.id,
            'product_id': service.id,
            'name': f'Order from Shoper #{self.order_id}',
            'product_uom_qty': 1,
            'price_unit': self.sum or 0,
        }
        
        self.env['sale.order.line'].create(line_vals)
        _logger.info(f"Created order line for {self.sum} PLN")
    
    def _prepare_order_note(self):
        """Przygotowuje notatkę do zamówienia"""
        notes = []
        
        if self.status:
            notes.append(f"Status: {self.status}")
        if self.tracking_number:
            notes.append(f"Tracking: {self.tracking_number}")
        notes.append(f"Shoper ID: {self.order_id}")
        notes.append(f"Order Number: {self.order_number}")
        
        return '\n'.join(notes)
    
    @api.model
    def create(self, vals):
        """Override create aby automatycznie tworzyć Sale Order"""
        record = super(ShoperOrderSync, self).create(vals)
        
        # Sprawdź czy automatycznie tworzyć SO
        config = self.env['shoper.config'].get_active_config()
        if config and config.auto_create_sale_orders:
            try:
                record.create_sale_order()
            except Exception as e:
                _logger.error(f"Failed to auto-create SO for order {record.order_id}: {e}")
        
        return record
    
    def action_create_sale_order(self):
        """Akcja do ręcznego tworzenia Sale Order"""
        self.ensure_one()
        sale_order = self.create_sale_order()
        
        # Otwórz utworzone zamówienie
        return {
            'type': 'ir.actions.act_window',
            'res_model': 'sale.order',
            'res_id': sale_order.id,
            'view_mode': 'form',
            'target': 'current',
        }
    
    @api.model
    def batch_create_sale_orders(self, limit=100):
        """Tworzy Sale Orders dla wszystkich zamówień które ich nie mają"""
        orders_without_so = self.search([
            ('sale_order_id', '=', False)
        ], limit=limit)
        
        created = 0
        errors = []
        
        for order in orders_without_so:
            try:
                order.create_sale_order()
                created += 1
                if created % 10 == 0:
                    self.env.cr.commit()
                    _logger.info(f"Progress: {created} Sale Orders created")
            except Exception as e:
                errors.append(f"Order {order.order_id}: {str(e)}")
                _logger.error(f"Failed to create SO for order {order.order_id}: {e}")
        
        self.env.cr.commit()
        
        message = f"Created {created} Sale Orders."
        if errors:
            message += f"\n\nErrors ({len(errors)}):\n" + "\n".join(errors[:5])
        
        _logger.info(message)
        return {'created': created, 'errors': len(errors)}


class ShoperConfig(models.Model):
    _inherit = 'shoper.config'
    
    auto_create_sale_orders = fields.Boolean(
        string='Auto Create Sale Orders',
        default=True,
        help='Automatically create Sale Orders when importing from Shoper'
    )
    
    def action_create_all_sale_orders(self):
        """Tworzy Sale Orders dla wszystkich zamówień"""
        self.ensure_one()
        
        result = self.env['shoper.order.sync'].batch_create_sale_orders(limit=1000)
        
        return {
            'type': 'ir.actions.client',
            'tag': 'display_notification',
            'params': {
                'title': 'Batch Creation Complete',
                'message': f"Created {result['created']} Sale Orders. Errors: {result['errors']}",
                'type': 'success' if result['errors'] == 0 else 'warning',
                'sticky': True,
            }
        }
