# -*- coding: utf-8 -*-
"""
Ulepszona synchronizacja zamówień dla modułu Odoo
- Obsługa pola updated_at zgodna z API Shoper  
- Szczegółowe logowanie
- Śledzenie zmian (statusy, płatności, przesyłki)
- Incrementalna synchronizacja
"""

import logging
import json
from datetime import datetime, timedelta
from odoo import models, fields, api
from odoo.exceptions import UserError

_logger = logging.getLogger(__name__)

# Dodatkowy logger dla zmian
changes_logger = logging.getLogger('shoper.changes')
changes_logger.setLevel(logging.INFO)


class ShoperOrderSync(models.Model):
    """Model do synchronizacji zamówień z Shoper"""
    _name = 'shoper.order.sync'
    _description = 'Shoper Order Synchronization'
    _order = 'order_id desc'
    _rec_name = 'order_number'
    
    # Mapowanie statusów
    STATUS_MAPPING = {
        1: 'new',
        2: 'processing', 
        3: 'completed',
        4: 'cancelled',
        5: 'return',
        6: 'complaint',
        7: 'waiting_for_payment',
        8: 'shipped',
        9: 'ready_for_pickup'
    }
    
    # Podstawowe pola
    order_id = fields.Integer(
        string='Shoper Order ID',
        required=True,
        index=True,
        readonly=True
    )
    order_number = fields.Char(
        string='Order Number',
        required=True,
        index=True,
        readonly=True
    )
    
    # Status i płatności
    status_id = fields.Integer(
        string='Status ID',
        readonly=True
    )
    status = fields.Selection([
        ('new', 'New'),
        ('processing', 'Processing'),
        ('completed', 'Completed'),
        ('cancelled', 'Cancelled'),
        ('return', 'Return'),
        ('complaint', 'Complaint'),
        ('waiting_for_payment', 'Waiting for Payment'),
        ('shipped', 'Shipped'),
        ('ready_for_pickup', 'Ready for Pickup')
    ], string='Status', compute='_compute_status', store=True)
    
    payment_id = fields.Integer(
        string='Payment Method ID',
        readonly=True
    )
    delivery_id = fields.Integer(
        string='Delivery Method ID',
        readonly=True
    )
    paid = fields.Boolean(
        string='Paid',
        default=False,
        readonly=True
    )
    
    # Kwoty
    sum = fields.Float(
        string='Total Amount',
        digits=(10, 2),
        readonly=True
    )
    currency_id = fields.Many2one(
        'res.currency',
        string='Currency',
        default=lambda self: self.env.company.currency_id
    )
    
    # Dane klienta
    partner_id = fields.Many2one(
        'res.partner',
        string='Customer',
        ondelete='restrict'
    )
    email = fields.Char(
        string='Email',
        readonly=True
    )
    firstname = fields.Char(
        string='First Name',
        readonly=True
    )
    surname = fields.Char(
        string='Last Name',
        readonly=True
    )
    company_name = fields.Char(
        string='Company Name',
        readonly=True
    )
    phone = fields.Char(
        string='Phone',
        readonly=True
    )
    nip = fields.Char(
        string='NIP/VAT',
        readonly=True
    )
    
    # Daty z API
    date_add = fields.Datetime(
        string='Created Date',
        readonly=True
    )
    date_confirm = fields.Datetime(
        string='Confirmation Date',
        readonly=True
    )
    date_delivery = fields.Datetime(
        string='Delivery Date',
        readonly=True
    )
    
    # KLUCZOWE POLE - data ostatniej modyfikacji w Shoper (date_edit z API)
    shoper_updated_at = fields.Datetime(
        string='Shoper Updated At',
        readonly=True,
        help='Last modification date in Shoper (date_edit from API)'
    )
    
    # Tracking
    tracking_number = fields.Char(
        string='Tracking Number',
        readonly=True
    )
    
    # Metadane synchronizacji
    imported_at = fields.Datetime(
        string='Imported At',
        default=fields.Datetime.now,
        readonly=True
    )
    last_sync_at = fields.Datetime(
        string='Last Sync At',
        default=fields.Datetime.now,
        readonly=True
    )
    sync_count = fields.Integer(
        string='Sync Count',
        default=1,
        readonly=True
    )
    
    # Surowe dane
    raw_data = fields.Text(
        string='Raw Data (JSON)',
        readonly=True
    )
    
    # Powiązanie z sale.order
    sale_order_id = fields.Many2one(
        'sale.order',
        string='Sale Order',
        ondelete='set null'
    )
    
    # Historia zmian
    change_ids = fields.One2many(
        'shoper.order.change',
        'order_sync_id',
        string='Changes History'
    )
    
    _sql_constraints = [
        ('order_id_uniq', 'unique(order_id)', 'Shoper Order ID must be unique!'),
    ]
    
    @api.depends('status_id')
    def _compute_status(self):
        """Oblicza status na podstawie status_id"""
        for record in self:
            record.status = self.STATUS_MAPPING.get(record.status_id, 'new')
    
    @api.model
    def sync_single_order(self, order_data):
        """
        Synchronizuje pojedyncze zamówienie
        Zwraca: (status, record) gdzie status to 'new', 'updated', 'unchanged'
        """
        order_id = int(order_data.get('order_id', 0))
        
        if not order_id:
            _logger.warning("Order without ID, skipping")
            return 'error', None
        
        try:
            # Sprawdź czy istnieje
            existing = self.search([('order_id', '=', order_id)], limit=1)
            
            # Parsuj daty
            def parse_date(date_str):
                if not date_str or date_str == '0000-00-00 00:00:00':
                    return False
                try:
                    return datetime.strptime(date_str, '%Y-%m-%d %H:%M:%S')
                except:
                    return False
            
            # Przygotuj dane
            vals = {
                'order_id': order_id,
                'order_number': order_data.get('code', ''),
                'status_id': int(order_data.get('status_id', 0)),
                'payment_id': int(order_data.get('payment_id', 0)),
                'delivery_id': int(order_data.get('delivery_id', 0)),
                'paid': bool(order_data.get('paid') in [1, '1', True]),
                'sum': float(order_data.get('sum', 0)),
                'email': order_data.get('email', ''),
                'firstname': order_data.get('firstname', ''),
                'surname': order_data.get('surname', ''),
                'company_name': order_data.get('company_name', ''),
                'phone': order_data.get('phone', ''),
                'nip': order_data.get('nip', ''),
                'tracking_number': order_data.get('tracking_number', ''),
                'date_add': parse_date(order_data.get('date_add')),
                'date_confirm': parse_date(order_data.get('date_confirm')),
                'date_delivery': parse_date(order_data.get('date_delivery')),
                'shoper_updated_at': parse_date(order_data.get('date_edit')),  # KLUCZOWE!
                'raw_data': json.dumps(order_data, ensure_ascii=False),
            }
            
            if not existing:
                # Nowe zamówienie
                vals['imported_at'] = fields.Datetime.now()
                vals['last_sync_at'] = fields.Datetime.now()
                vals['sync_count'] = 1
                
                # Znajdź lub utwórz partnera
                partner = self._find_or_create_partner(order_data)
                if partner:
                    vals['partner_id'] = partner.id
                
                # Utwórz rekord
                new_order = self.create(vals)
                
                _logger.info(f"📦 New order #{order_id}: {vals['email']}, {vals['sum']:.2f} PLN")
                changes_logger.info(f"ORDER #{order_id} | Created: {vals['order_number']}")
                
                # Zapisz do historii
                self.env['shoper.order.change'].create({
                    'order_sync_id': new_order.id,
                    'field_name': 'order',
                    'old_value': '',
                    'new_value': f"#{vals['order_number']}",
                    'change_type': 'create'
                })
                
                return 'new', new_order
                
            else:
                # Sprawdź zmiany
                changes = self._detect_changes(existing, vals)
                
                if changes:
                    # Aktualizuj
                    vals['last_sync_at'] = fields.Datetime.now()
                    vals['sync_count'] = existing.sync_count + 1
                    
                    existing.write(vals)
                    
                    # Zapisz zmiany do historii
                    for change in changes:
                        self.env['shoper.order.change'].create({
                            'order_sync_id': existing.id,
                            'field_name': change['field'],
                            'old_value': str(change['old']),
                            'new_value': str(change['new']),
                            'change_type': 'update'
                        })
                        
                        changes_logger.info(
                            f"ORDER #{order_id} | {change['field']}: "
                            f"{change['old']} → {change['new']}"
                        )
                    
                    _logger.info(f"🔄 Updated order #{order_id} - {len(changes)} changes")
                    
                    return 'updated', existing
                    
                else:
                    # Bez zmian - tylko aktualizuj last_sync_at
                    existing.write({'last_sync_at': fields.Datetime.now()})
                    return 'unchanged', existing
            
        except Exception as e:
            _logger.error(f"❌ Error syncing order #{order_id}: {e}")
            return 'error', None
    
    def _detect_changes(self, record, new_vals):
        """Wykrywa zmiany w zamówieniu"""
        changes = []
        
        # Pola do monitorowania
        fields_to_check = {
            'status_id': 'Status',
            'paid': 'Payment',
            'sum': 'Amount',
            'tracking_number': 'Tracking',
            'delivery_id': 'Delivery',
            'payment_id': 'Payment Method'
        }
        
        for field, name in fields_to_check.items():
            old_val = getattr(record, field)
            new_val = new_vals.get(field)
            
            # Normalizacja typów
            if field == 'paid':
                old_val = bool(old_val)
                new_val = bool(new_val)
            elif field == 'sum':
                old_val = float(old_val or 0)
                new_val = float(new_val or 0)
            elif field in ['status_id', 'delivery_id', 'payment_id']:
                old_val = int(old_val or 0)
                new_val = int(new_val or 0)
            
            if old_val != new_val:
                changes.append({
                    'field': field,
                    'name': name,
                    'old': old_val,
                    'new': new_val
                })
        
        return changes
    
    def _find_or_create_partner(self, order_data):
        """Znajduje lub tworzy partnera"""
        email = order_data.get('email')
        if not email:
            return None
        
        # Szukaj po emailu
        partner = self.env['res.partner'].search([('email', '=', email)], limit=1)
        
        if not partner:
            # Utwórz nowego
            partner_vals = {
                'name': f"{order_data.get('firstname', '')} {order_data.get('surname', '')}".strip() or email,
                'email': email,
                'phone': order_data.get('phone', ''),
                'vat': order_data.get('nip', ''),
                'is_company': bool(order_data.get('company_name')),
                'customer_rank': 1,
            }
            
            if order_data.get('company_name'):
                partner_vals['name'] = order_data['company_name']
            
            partner = self.env['res.partner'].create(partner_vals)
        
        return partner
    
    @api.model
    def sync_new_orders(self, limit_pages=5):
        """Synchronizuje nowe zamówienia"""
        config = self.env['shoper.config'].get_active_config()
        if not config:
            raise UserError('No active Shoper configuration found')
        
        client = config._get_api_client()
        
        # Pobierz ostatnie ID
        last_order = self.search([], order='order_id desc', limit=1)
        last_id = last_order.order_id if last_order else 0
        
        _logger.info(f"📍 Last ID in database: {last_id}")
        
        # Pobierz całkowitą liczbę
        total_count = client.get_orders_count()
        _logger.info(f"📊 Orders in shop: {total_count}")
        
        if total_count == 0:
            return {'status': 'success', 'new': 0, 'updated': 0}
        
        # Oblicz strony
        total_pages = (total_count // 50) + (1 if total_count % 50 > 0 else 0)
        pages_to_check = min(limit_pages, total_pages)
        
        new_orders = []
        stats = {'new': 0, 'updated': 0, 'unchanged': 0, 'errors': 0}
        
        # Sprawdź ostatnie strony
        for i in range(pages_to_check):
            page = total_pages - i
            if page < 1:
                break
            
            orders = client.get_orders_page(limit=50, page=page)
            
            if not orders:
                continue
            
            # Filtruj tylko nowe
            for order in orders:
                order_id = int(order.get('order_id', 0))
                if order_id > last_id:
                    # Pobierz pełne dane
                    full_order = client.get_order_details(order_id)
                    if full_order:
                        status, record = self.sync_single_order(full_order)
                        stats[status] = stats.get(status, 0) + 1
        
        _logger.info(f"✅ Sync completed: {stats}")
        
        return stats
    
    @api.model
    def sync_recent_updates(self, hours_back=24):
        """Synchronizuje ostatnie zmiany"""
        config = self.env['shoper.config'].get_active_config()
        if not config:
            raise UserError('No active Shoper configuration found')
        
        client = config._get_api_client()
        
        # Znajdź zamówienia do sprawdzenia
        date_from = fields.Datetime.now() - timedelta(hours=hours_back)
        orders_to_check = self.search([
            '|',
            ('last_sync_at', '<', date_from),
            ('shoper_updated_at', '>', date_from)
        ], limit=100)
        
        _logger.info(f"📋 Checking {len(orders_to_check)} orders for updates")
        
        stats = {'updated': 0, 'unchanged': 0, 'errors': 0}
        
        for order in orders_to_check:
            try:
                # Pobierz aktualne dane
                order_data = client.get_order_details(order.order_id)
                if order_data:
                    status, record = self.sync_single_order(order_data)
                    stats[status] = stats.get(status, 0) + 1
            except Exception as e:
                _logger.error(f"Error updating order #{order.order_id}: {e}")
                stats['errors'] += 1
        
        _logger.info(f"✅ Updates completed: {stats}")
        
        return stats
    
    def action_sync_single(self):
        """Akcja do ręcznej synchronizacji pojedynczego zamówienia"""
        self.ensure_one()
        
        config = self.env['shoper.config'].get_active_config()
        if not config:
            raise UserError('No active Shoper configuration found')
        
        client = config._get_api_client()
        
        # Pobierz dane z API
        order_data = client.get_order_details(self.order_id)
        if not order_data:
            raise UserError(f'Could not fetch order #{self.order_id} from Shoper')
        
        # Synchronizuj
        status, record = self.sync_single_order(order_data)
        
        if status == 'updated':
            message = f'Order updated successfully ({len(self.change_ids)} changes)'
        elif status == 'unchanged':
            message = 'Order is up to date'
        else:
            message = f'Sync status: {status}'
        
        return {
            'type': 'ir.actions.client',
            'tag': 'display_notification',
            'params': {
                'title': 'Synchronization Complete',
                'message': message,
                'type': 'success',
                'sticky': False,
            }
        }


class ShoperOrderChange(models.Model):
    """Historia zmian zamówień"""
    _name = 'shoper.order.change'
    _description = 'Shoper Order Change History'
    _order = 'changed_at desc'
    
    order_sync_id = fields.Many2one(
        'shoper.order.sync',
        string='Order',
        required=True,
        ondelete='cascade'
    )
    changed_at = fields.Datetime(
        string='Changed At',
        default=fields.Datetime.now,
        required=True
    )
    field_name = fields.Char(
        string='Field',
        required=True
    )
    old_value = fields.Text(
        string='Old Value'
    )
    new_value = fields.Text(
        string='New Value'
    )
    change_type = fields.Selection([
        ('create', 'Created'),
        ('update', 'Updated'),
        ('delete', 'Deleted')
    ], string='Type', default='update')
    
    def name_get(self):
        result = []
        for record in self:
            name = f"{record.field_name}: {record.old_value} → {record.new_value}"
            result.append((record.id, name))
        return result


    
    @api.model
    def create_from_shoper(self, order_data, config):
        """Tworzy nowe zamówienie z danych Shopera"""
        
        # Parsuj daty
        def parse_date(date_str):
            if not date_str or date_str == '0000-00-00 00:00:00':
                return False
            try:
                from datetime import datetime
                return datetime.strptime(date_str, '%Y-%m-%d %H:%M:%S')
            except:
                return False
        
        # Przygotuj dane
        vals = {
            'order_id': int(order_data.get('order_id', 0)),
            'order_number': order_data.get('code', ''),
            'status_id': int(order_data.get('status_id', 0)),
            'payment_id': int(order_data.get('payment_id', 0)),
            'delivery_id': int(order_data.get('delivery_id', 0)),
            'paid': bool(order_data.get('paid') in [1, '1', True]),
            'sum': float(order_data.get('sum', 0)),
            'email': order_data.get('email', ''),
            'firstname': order_data.get('firstname', ''),
            'surname': order_data.get('surname', ''),
            'company_name': order_data.get('company_name', ''),
            'phone': order_data.get('phone', ''),
            'nip': order_data.get('nip', ''),
            'tracking_number': order_data.get('tracking_number', ''),
            'date_add': parse_date(order_data.get('date_add')),
            'date_confirm': parse_date(order_data.get('date_confirm')),
            'date_delivery': parse_date(order_data.get('date_delivery')),
            'shoper_updated_at': parse_date(order_data.get('date_edit')),
            'raw_data': json.dumps(order_data, ensure_ascii=False),
        }
        
        # Znajdź lub utwórz partnera
        partner = self._find_or_create_partner(order_data)
        if partner:
            vals['partner_id'] = partner.id
        
        # Utwórz rekord
        new_order = self.create(vals)
        
        # Jeśli włączone, utwórz Sale Order
        if config.auto_create_sale_orders:
            try:
                new_order.create_sale_order()
            except Exception as e:
                _logger.error(f"Failed to create SO for order {new_order.order_id}: {e}")
        
        return new_order
    
    def create_sale_order(self):
        """Tworzy Sale Order z zamówienia Shoper"""
        self.ensure_one()
        
        if self.sale_order_id:
            return self.sale_order_id
        
        # Znajdź lub utwórz partnera
        if not self.partner_id:
            self.partner_id = self._find_or_create_partner({
                'email': self.email,
                'firstname': self.firstname,
                'surname': self.surname,
                'company_name': self.company_name,
                'phone': self.phone,
                'nip': self.nip,
            })
        
        # Przygotuj dane SO
        sale_order_vals = {
            'partner_id': self.partner_id.id,
            'partner_invoice_id': self.partner_id.id,
            'partner_shipping_id': self.partner_id.id,
            'date_order': self.date_add or fields.Datetime.now(),
            'state': 'sale' if self.paid else 'draft',
            'origin': f'Shoper #{self.order_id}',
            'client_order_ref': self.order_number,
        }
        
        # Utwórz SO
        sale_order = self.env['sale.order'].create(sale_order_vals)
        
        # Dodaj linię z produktem
        product = self.env['product.product'].search([
            ('name', '=', 'Shoper Order'),
            ('type', '=', 'service')
        ], limit=1)
        
        if not product:
            product = self.env['product.product'].create({
                'name': 'Shoper Order',
                'type': 'service',
                'list_price': 0,
                'sale_ok': True,
                'purchase_ok': False,
            })
        
        # Utwórz linię
        self.env['sale.order.line'].create({
            'order_id': sale_order.id,
            'product_id': product.id,
            'name': f'Zamówienie Shoper #{self.order_id}',
            'product_uom_qty': 1,
            'price_unit': self.sum,
        })
        
        # Zapisz referencję
        self.sale_order_id = sale_order.id
        
        # Potwierdź jeśli opłacone
        if self.paid and sale_order.state == 'draft':
            sale_order.action_confirm()
        
        _logger.info(f"Created Sale Order {sale_order.name} for Shoper #{self.order_id}")
        
        return sale_order
    
    def action_create_all_missing_so(self):
        """Tworzy brakujące Sale Orders"""
        orders_without_so = self.search([
            ('sale_order_id', '=', False)
        ])
        
        created = 0
        for order in orders_without_so:
            try:
                order.create_sale_order()
                created += 1
                if created % 10 == 0:
                    self.env.cr.commit()
                    _logger.info(f"Created {created} Sale Orders...")
            except Exception as e:
                _logger.error(f"Failed to create SO for {order.order_id}: {e}")
        
        return {
            'type': 'ir.actions.client',
            'tag': 'display_notification',
            'params': {
                'title': 'Sale Orders Created',
                'message': f'Successfully created {created} Sale Orders',
                'type': 'success',
                'sticky': True,
            }
        }
