# -*- coding: utf-8 -*-
"""
Poprawiona konfiguracja Shoper z automatycznym tworzeniem Sale Orders
"""

from odoo import models, fields, api
from odoo.exceptions import UserError
import logging

_logger = logging.getLogger(__name__)


class ShoperConfig(models.Model):
    _name = 'shoper.config'
    _description = 'Shoper API Configuration'
    _rec_name = 'shop_url'

    # Połączenie
    shop_url = fields.Char(
        string='Shop URL',
        required=True,
        help='Shoper shop URL (e.g., https://yourshop.pl)',
    )
    access_token = fields.Char(
        string='API Access Token',
        required=True,
        help='API token from Shoper admin panel',
    )
    active = fields.Boolean(
        string='Active',
        default=True,
    )

    # Ustawienia synchronizacji
    auto_sync_enabled = fields.Boolean(
        string='Enable Automatic Synchronization',
        default=False,
        help='Enable automatic order synchronization via cron job',
    )
    auto_create_sale_orders = fields.Boolean(
        string='Auto Create Sale Orders',
        default=True,
        help='Automatically create Sale Orders when importing from Shoper'
    )
    sync_interval_hours = fields.Integer(
        string='Synchronization Interval (hours)',
        default=1,
        help='How often to synchronize orders automatically',
    )
    last_sync_date = fields.Datetime(
        string='Last Synchronization',
        readonly=True,
    )
    last_order_update = fields.Datetime(
        string='Last Order Update Date',
        readonly=True,
        help='Used for incremental synchronization (updated_at field from Shoper)',
    )

    # Ustawienia importu
    max_orders_per_sync = fields.Integer(
        string='Max Orders Per Sync',
        default=100,
        help='Maximum number of orders to import in one synchronization',
    )
    import_from_date = fields.Datetime(
        string='Import From Date',
        help='Import only orders created after this date (leave empty for all). Used only on first sync.',
    )

    # Mapowanie statusów
    default_order_state = fields.Selection(
        [('draft', 'Quotation'), ('sent', 'Quotation Sent'), ('sale', 'Sales Order'), ('done', 'Locked'), ('cancel', 'Cancelled')],
        string='Default Order State',
        default='sale',
        help='Default state for imported orders',
    )

    # Statystyki
    total_orders_imported = fields.Integer(
        string='Total Orders Imported',
        readonly=True,
        default=0,
    )
    total_orders_updated = fields.Integer(
        string='Total Orders Updated',
        readonly=True,
        default=0,
    )
    total_sale_orders_created = fields.Integer(
        string='Total Sale Orders Created',
        readonly=True,
        default=0,
    )

    # Company
    company_id = fields.Many2one(
        'res.company',
        string='Company',
        required=True,
        default=lambda self: self.env.company,
    )

    _sql_constraints = [
        ('shop_url_unique', 'unique(shop_url, company_id)', 'Shop URL must be unique per company!'),
    ]

    @api.model
    def get_active_config(self):
        """Pobiera aktywną konfigurację"""
        config = self.search([('active', '=', True), ('company_id', '=', self.env.company.id)], limit=1)
        if not config:
            raise UserError('No active Shoper configuration found. Please configure Shoper API settings first.')
        return config
    
    def _get_api_client(self):
        """Zwraca instancję klienta API Shoper"""
        self.ensure_one()
        from .shoper_api_client import ShoperAPIClient
        return ShoperAPIClient(self.shop_url, self.access_token)

    def test_connection(self):
        """Test połączenia z API Shoper"""
        self.ensure_one()
        
        try:
            from .shoper_api_client import ShoperAPIClient
            
            client = ShoperAPIClient(self.shop_url, self.access_token)
            
            # Sprawdź połączenie
            result = client.test_connection()
            
            if result:
                # Sprawdź ile jest zamówień
                count = client.get_orders_count()
                
                return {
                    'type': 'ir.actions.client',
                    'tag': 'display_notification',
                    'params': {
                        'title': 'Connection Successful',
                        'message': f'Successfully connected to {self.shop_url}\nTotal orders in shop: {count}',
                        'type': 'success',
                        'sticky': False,
                    }
                }
            else:
                raise UserError('Connection test failed. Please check your credentials.')
                
        except Exception as e:
            _logger.error(f"Shoper connection test failed: {str(e)}")
            raise UserError(f'Connection test failed: {str(e)}')

    def action_sync_orders(self):
        """
        Ręczna/automatyczna synchronizacja zamówień
        """
        self.ensure_one()
        
        try:
            from .shoper_api_client import ShoperAPIClient
            from datetime import datetime, timedelta
            
            _logger.info("="*80)
            _logger.info(f"STARTING ORDER SYNCHRONIZATION: {self.shop_url}")
            _logger.info("="*80)
            
            # Utwórz klienta API
            client = ShoperAPIClient(self.shop_url, self.access_token)
            
            # Określ strategię synchronizacji
            filters = {}
            
            if self.last_order_update:
                # INKREMENTALNA SYNCHRONIZACJA - tylko zaktualizowane zamówienia
                sync_from = self.last_order_update - timedelta(minutes=1)
                filters = {
                    'updated_at': {'>=': sync_from.strftime('%Y-%m-%d %H:%M:%S')}
                }
                _logger.info(f"Fetching orders updated since: {sync_from}")
                
            elif self.import_from_date:
                # SYNCHRONIZACJA OD DATY
                filters = {
                    'date': {'>=': self.import_from_date.strftime('%Y-%m-%d %H:%M:%S')}
                }
                _logger.info(f"Fetching orders from date: {self.import_from_date}")
                
            else:
                # PEŁNA SYNCHRONIZACJA
                _logger.info(f"Fetching ALL orders (max {self.max_orders_per_sync})")
            
            # Pobierz zamówienia z Shoper
            orders = client.get_orders_filtered(
                filters=filters,
                max_orders=self.max_orders_per_sync,
                order_by='-order_id'  # Najnowsze pierwsze
            )
            
            _logger.info(f"Fetched {len(orders)} orders from Shoper API")
            
            if not orders:
                _logger.info("No orders returned from API")
                
                self.write({
                    'last_sync_date': fields.Datetime.now(),
                })
                
                return {
                    'type': 'ir.actions.client',
                    'tag': 'display_notification',
                    'params': {
                        'title': 'Synchronization Completed',
                        'message': 'No new orders to synchronize',
                        'type': 'info',
                        'sticky': False,
                    }
                }
            
            # Przetwórz zamówienia
            imported_count = 0
            updated_count = 0
            sale_orders_created = 0
            errors = []
            
            latest_update = self.last_order_update
            
            for order_data in orders:
                order_id = order_data.get('order_id')
                
                try:
                    # Sprawdź czy zamówienie już istnieje
                    existing_order = self.env['shoper.order.sync'].search([
                        ('order_id', '=', order_id)
                    ], limit=1)
                    
                    if existing_order:
                        # Aktualizacja istniejącego
                        updated_count += 1
                        _logger.info(f"Updated order #{order_id}")
                    else:
                        # Import nowego zamówienia
                        new_order = self.env['shoper.order.sync'].create_from_shoper(order_data, self)
                        imported_count += 1
                        _logger.info(f"Created new order #{order_id}")
                        
                        # Automatyczne tworzenie Sale Order
                        if self.auto_create_sale_orders:
                            try:
                                sale_order = new_order.create_sale_order()
                                sale_orders_created += 1
                                _logger.info(f"Created Sale Order {sale_order.name} for Shoper #{order_id}")
                            except Exception as e:
                                _logger.error(f"Failed to create SO for order {order_id}: {e}")
                    
                    # Commit co 20 zamówień
                    if (imported_count + updated_count) % 20 == 0:
                        self.env.cr.commit()
                    
                except Exception as e:
                    error_msg = f"Error processing order #{order_id}: {str(e)}"
                    _logger.error(error_msg)
                    errors.append(error_msg)
            
            # Końcowy commit
            self.env.cr.commit()
            
            # Aktualizuj statystyki
            self.write({
                'last_sync_date': fields.Datetime.now(),
                'total_orders_imported': self.total_orders_imported + imported_count,
                'total_orders_updated': self.total_orders_updated + updated_count,
                'total_sale_orders_created': self.total_sale_orders_created + sale_orders_created,
                'last_order_update': fields.Datetime.now(),
            })
            
            # Podsumowanie
            _logger.info("="*80)
            _logger.info("SYNCHRONIZATION SUMMARY")
            _logger.info("="*80)
            _logger.info(f"New orders imported: {imported_count}")
            _logger.info(f"Sale Orders created: {sale_orders_created}")
            _logger.info(f"Errors: {len(errors)}")
            _logger.info("="*80)
            
            message = f"""Synchronization completed:
• New orders: {imported_count}
• Updated: {updated_count}
• Sale Orders created: {sale_orders_created}
• Errors: {len(errors)}"""
            
            return {
                'type': 'ir.actions.client',
                'tag': 'display_notification',
                'params': {
                    'title': 'Synchronization Completed',
                    'message': message,
                    'type': 'success' if len(errors) == 0 else 'warning',
                    'sticky': True,
                }
            }
            
        except Exception as e:
            _logger.error(f"Order synchronization failed: {str(e)}")
            raise UserError(f'Synchronization failed: {str(e)}')

    @api.model
    def cron_sync_orders(self):
        """Metoda wywoływana przez cron do automatycznej synchronizacji"""
        configs = self.search([
            ('active', '=', True),
            ('auto_sync_enabled', '=', True)
        ])
        
        for config in configs:
            try:
                _logger.info(f"Running automatic synchronization for {config.shop_url}")
                config.action_sync_orders()
            except Exception as e:
                _logger.error(f"Automatic synchronization failed for {config.shop_url}: {str(e)}")
