# -*- coding: utf-8 -*-
"""
Klient API Shopera dla Odoo - Wersja DEBUG
"""

import requests
from datetime import datetime
import time
import logging
import json

_logger = logging.getLogger(__name__)


class ShoperAPIClient:
    """Klient do komunikacji z API Shoper"""
    
    def __init__(self, shop_url, access_token):
        """
        Inicjalizacja klienta API
        
        Args:
            shop_url: URL sklepu Shoper (np. https://alpmeb.pl)
            access_token: Token dostępu do API
        """
        self.shop_url = shop_url.rstrip('/')
        self.access_token = access_token
        self.api_base_url = f"{self.shop_url}/webapi/rest"
        self.headers = {
            'Authorization': f'Bearer {self.access_token}',
            'Content-Type': 'application/json'
        }
        _logger.info(f"ShoperAPIClient initialized for {self.shop_url}")
        
    def _make_request(self, method, endpoint, params=None, data=None):
        """Wykonuje zapytanie do API"""
        url = f"{self.api_base_url}/{endpoint}"
        
        _logger.debug(f"API Request: {method} {url}")
        _logger.debug(f"Params: {params}")
        
        try:
            response = requests.request(
                method=method,
                url=url,
                headers=self.headers,
                params=params,
                json=data,
                timeout=30
            )
            
            _logger.debug(f"API Response Status: {response.status_code}")
            
            if response.status_code == 200:
                result = response.json()
                _logger.debug(f"API Response (first 500 chars): {str(result)[:500]}")
                return result
            else:
                _logger.error(f"Shoper API error {response.status_code}: {endpoint}")
                _logger.error(f"Response text: {response.text[:1000]}")
                return None
                
        except Exception as e:
            _logger.error(f"Shoper API connection error: {e}")
            return None
    
    def test_connection(self):
        """Testuje połączenie z API"""
        result = self._make_request('GET', 'orders', params={'limit': 1})
        return result is not None
    
    def get_orders_count(self, filters=None):
        """
        Pobiera liczbę zamówień (opcjonalnie z filtrami)
        """
        params = {'limit': 1}
        
        # Dodaj filtry do query params
        if filters:
            params.update(self._prepare_filter_params(filters))
        
        result = self._make_request('GET', 'orders', params=params)
        if result and 'count' in result:
            count = result['count']
            return int(count) if isinstance(count, str) else count
        return 0
    
    def _prepare_filter_params(self, filters):
        """
        Przygotowuje parametry filtrów dla API Shoper
        """
        params = {}
        
        for field, value in filters.items():
            if isinstance(value, dict):
                # Dla operatorów porównania
                for operator, val in value.items():
                    params[f'{field}[{operator}]'] = val
            else:
                # Dla prostych wartości
                params[field] = value
        
        _logger.debug(f"Prepared filter params: {params}")
        return params
    
    def get_orders_page(self, limit=50, page=1, filters=None, order_by=None):
        """
        Pobiera stronę zamówień z opcjonalnymi filtrami
        """
        params = {
            'limit': min(limit, 50),
            'page': page
        }
        
        # Dodaj filtry
        if filters:
            params.update(self._prepare_filter_params(filters))
        
        # Dodaj sortowanie
        if order_by:
            params['order'] = order_by
        
        _logger.info(f"Fetching orders page {page} with params: {params}")
        
        result = self._make_request('GET', 'orders', params=params)
        
        if not result:
            _logger.warning("No result from API")
            return []
            
        if 'list' not in result:
            _logger.warning(f"No 'list' in API response. Keys: {result.keys()}")
            return []
        
        orders = result['list']
        _logger.info(f"Got {len(orders)} orders from API")
        
        # Konwertuj ID na int
        for order in orders:
            if 'order_id' in order and isinstance(order['order_id'], str):
                try:
                    order['order_id'] = int(order['order_id'])
                except (ValueError, TypeError):
                    pass
        
        return orders
    
    def get_order_details(self, order_id):
        """Pobiera szczegóły zamówienia"""
        result = self._make_request('GET', f'orders/{order_id}')
        if result and 'order_id' in result and isinstance(result['order_id'], str):
            try:
                result['order_id'] = int(result['order_id'])
            except (ValueError, TypeError):
                pass
        return result
    
    def get_order_products(self, order_id):
        """
        Pobiera produkty z zamówienia
        """
        # Przygotuj filtr JSON
        filters_json = json.dumps({"order_id": order_id})
        params = {
            'filters': filters_json,
            'limit': 50  
        }
        
        result = self._make_request('GET', 'order-products', params=params)
        
        if not result:
            return []
        
        products = result.get('list', [])
        
        # Konwertuj ID produktów
        for product in products:
            if 'product_id' in product and isinstance(product['product_id'], str):
                try:
                    product['product_id'] = int(product['product_id'])
                except (ValueError, TypeError):
                    pass
        
        return products
    
    def get_orders_filtered(self, filters=None, max_orders=100, order_by=None):
        """
        Pobiera zamówienia z filtrami - WERSJA Z DEBUGIEM
        """
        all_orders = []
        page = 1
        batch_size = 50
        
        _logger.info("="*60)
        _logger.info(f"get_orders_filtered called:")
        _logger.info(f"  filters: {filters}")
        _logger.info(f"  max_orders: {max_orders}")
        _logger.info(f"  order_by: {order_by}")
        
        # Najpierw pobierz liczbę dostępnych zamówień
        total_count = self.get_orders_count(filters)
        _logger.info(f"Total orders matching filters: {total_count}")
        
        if total_count == 0:
            _logger.warning("No orders match the filters")
            return []
        
        # Jeśli nie ma filtrów, pobierz ostatnie zamówienia
        if not filters:
            _logger.info("No filters provided, fetching recent orders")
            # Ustaw domyślne sortowanie
            if not order_by:
                order_by = '-order_id'  # Najnowsze pierwsze
        
        while len(all_orders) < max_orders:
            _logger.info(f"Fetching page {page} (have {len(all_orders)} orders so far)...")
            
            # Pobierz stronę zamówień
            orders = self.get_orders_page(
                limit=batch_size,
                page=page,
                filters=filters,
                order_by=order_by
            )
            
            if not orders:
                _logger.info("No more orders to fetch")
                break
            
            _logger.info(f"Processing {len(orders)} orders from page {page}")
            
            # Przetwórz każde zamówienie
            for idx, order in enumerate(orders):
                order_id = order.get('order_id')
                _logger.debug(f"Processing order {idx+1}/{len(orders)}: ID={order_id}")
                
                # Pomiń jeśli puste (bez emaila i sumy)
                if not order.get('email') and not order.get('sum'):
                    _logger.debug(f"Skipping empty order: {order_id}")
                    continue
                
                # Dla pierwszego zamówienia pobierz szczegóły
                if len(all_orders) < 10:  # Pobierz szczegóły tylko dla pierwszych 10
                    _logger.debug(f"Fetching full details for order {order_id}")
                    full_order = self.get_order_details(order_id)
                    
                    if full_order:
                        # Pobierz produkty
                        products = self.get_order_products(order_id)
                        full_order['products'] = products
                        all_orders.append(full_order)
                        _logger.info(f"✓ Added order {order_id} with {len(products)} products")
                    else:
                        # Użyj podstawowych danych
                        order['products'] = []
                        all_orders.append(order)
                        _logger.warning(f"Using basic data for order {order_id}")
                else:
                    # Dla pozostałych użyj tylko podstawowych danych
                    order['products'] = []
                    all_orders.append(order)
                
                # Sprawdź limit
                if len(all_orders) >= max_orders:
                    _logger.info(f"Reached max orders limit: {max_orders}")
                    break
            
            # Jeśli na stronie było mniej niż batch_size, to koniec
            if len(orders) < batch_size:
                _logger.info("Fetched last page (less than batch_size orders)")
                break
            
            # Następna strona
            page += 1
            
            # Limit stron dla bezpieczeństwa
            if page > 10:
                _logger.warning("Reached page limit (10), stopping")
                break
            
            # Krótkie opóźnienie między requestami
            time.sleep(0.3)
        
        _logger.info(f"FINAL: Total orders fetched: {len(all_orders)}")
        _logger.info("="*60)
        return all_orders
