# -*- coding: utf-8 -*-

from odoo import models, fields, api
from odoo.exceptions import UserError
import logging
import json

_logger = logging.getLogger(__name__)


class SaleOrder(models.Model):
    _inherit = 'sale.order'

    # Pola Shoper
    shoper_order_id = fields.Integer(
        string='Shoper Order ID',
        readonly=True,
        index=True,
        help='Original order ID from Shoper',
    )
    shoper_status_id = fields.Integer(
        string='Shoper Status ID',
        readonly=True,
    )
    shoper_status_name = fields.Char(
        string='Shoper Status',
        readonly=True,
    )
    shoper_date = fields.Datetime(
        string='Shoper Order Date',
        readonly=True,
    )
    shoper_updated_at = fields.Datetime(
        string='Shoper Last Update',
        readonly=True,
        index=True,
        help='Last update date from Shoper (used for synchronization)',
    )
    shoper_payment_method = fields.Char(
        string='Shoper Payment Method',
        readonly=True,
    )
    shoper_delivery_method = fields.Char(
        string='Shoper Delivery Method',
        readonly=True,
    )
    shoper_paid = fields.Boolean(
        string='Shoper Paid Status',
        readonly=True,
        help='Payment status from Shoper',
    )
    is_shoper_order = fields.Boolean(
        string='Shoper Order',
        default=False,
        readonly=True,
        index=True,
    )
    shoper_raw_data = fields.Text(
        string='Shoper Raw Data',
        readonly=True,
        help='Raw JSON data from Shoper API',
    )

    _sql_constraints = [
        ('shoper_order_id_unique', 'unique(shoper_order_id, company_id)', 
         'Shoper Order ID must be unique per company!'),
    ]

    @api.model
    def create_from_shoper(self, order_data, config):
        """
        Tworzy zamówienie w Odoo na podstawie danych z Shoper
        
        Args:
            order_data: Dane zamówienia z API Shoper
            config: Rekord shoper.config
        
        Returns:
            Utworzony rekord sale.order
        """
        try:
            # Przygotuj dane klienta
            partner_vals = self._prepare_partner_from_shoper(order_data)
            
            # Znajdź lub utwórz partnera
            partner = self.env['res.partner'].find_or_create_from_shoper(partner_vals)
            
            # Przygotuj dane zamówienia
            order_vals = {
                'partner_id': partner.id,
                'company_id': config.company_id.id,
                'date_order': order_data.get('date') or fields.Datetime.now(),
                'state': config.default_order_state,
                'origin': f"Shoper #{order_data.get('order_id')}",
                
                # Pola Shoper
                'is_shoper_order': True,
                'shoper_order_id': order_data.get('order_id'),
                'shoper_status_id': order_data.get('status_id'),
                'shoper_status_name': self._get_status_name(order_data),
                'shoper_date': order_data.get('date'),
                'shoper_updated_at': order_data.get('updated_at'),
                'shoper_payment_method': self._get_payment_name(order_data),
                'shoper_delivery_method': self._get_delivery_name(order_data),
                'shoper_paid': bool(order_data.get('paid') in [1, '1', True]),
                'shoper_raw_data': json.dumps(order_data, ensure_ascii=False),
            }
            
            # Dodaj notatki jeśli są
            notes = []
            if order_data.get('notes'):
                notes.append(f"Customer notes: {order_data['notes']}")
            if order_data.get('notes_pub'):
                notes.append(f"Public notes: {order_data['notes_pub']}")
            if notes:
                order_vals['note'] = '\n'.join(notes)
            
            # Utwórz zamówienie
            order = self.create(order_vals)
            
            _logger.info(f"Created Odoo order {order.name} from Shoper #{order_data.get('order_id')}")
            
            # Dodaj produkty
            if order_data.get('products'):
                self._create_order_lines_from_shoper(order, order_data['products'])
            
            # Dodaj koszty dostawy jeśli są
            delivery_cost = self._get_delivery_cost(order_data)
            if delivery_cost:
                self._add_delivery_cost(order, order_data, delivery_cost)
            
            return order
            
        except Exception as e:
            _logger.error(f"Error creating order from Shoper: {str(e)}", exc_info=True)
            raise UserError(f"Failed to create order from Shoper: {str(e)}")

    def update_from_shoper(self, order_data):
        """
        Aktualizuje zamówienie na podstawie danych z Shoper
        
        Args:
            order_data: Zaktualizowane dane zamówienia z API Shoper
        
        Returns:
            bool: True jeśli coś się zmieniło, False jeśli brak zmian
        """
        self.ensure_one()
        
        try:
            # Pobierz aktualne wartości
            old_status_id = self.shoper_status_id
            old_paid = self.shoper_paid
            old_updated_at = self.shoper_updated_at
            
            # Pobierz nowe wartości
            new_status_id = order_data.get('status_id')
            new_paid = bool(order_data.get('paid') in [1, '1', True])
            new_updated_at = order_data.get('updated_at')
            
            # Sprawdź czy coś się zmieniło
            changes = []
            
            if old_status_id != new_status_id:
                old_name = self.shoper_status_name or f"Status {old_status_id}"
                new_name = self._get_status_name(order_data) or f"Status {new_status_id}"
                changes.append(f"status: {old_name} → {new_name}")
            
            if old_paid != new_paid:
                changes.append(f"paid: {old_paid} → {new_paid}")
            
            if old_updated_at != new_updated_at:
                changes.append(f"updated_at: {old_updated_at} → {new_updated_at}")
            
            # Jeśli nie ma zmian, zakończ
            if not changes:
                _logger.debug(f"No changes for Shoper order #{order_data.get('order_id')}")
                return False
            
            # Przygotuj wartości do aktualizacji
            vals = {
                'shoper_status_id': new_status_id,
                'shoper_status_name': self._get_status_name(order_data),
                'shoper_updated_at': new_updated_at,
                'shoper_paid': new_paid,
                'shoper_raw_data': json.dumps(order_data, ensure_ascii=False),
            }
            
            # Zapisz zmiany
            self.write(vals)
            
            # Loguj zmiany
            changes_str = ", ".join(changes)
            _logger.info(f"Updated Odoo order {self.name} (Shoper #{self.shoper_order_id}): {changes_str}")
            
            # Dodaj notatkę o zmianach
            self.message_post(
                body=f"Shoper order updated:\n• " + "\n• ".join(changes),
                subject="Shoper Sync Update"
            )
            
            return True
            
        except Exception as e:
            _logger.error(f"Error updating order from Shoper: {str(e)}", exc_info=True)
            raise UserError(f"Failed to update order from Shoper: {str(e)}")

    @api.model
    def _prepare_partner_from_shoper(self, order_data):
        """Przygotowuje dane partnera z danych zamówienia Shoper"""
        # Pobierz adres - preferuj delivery, fallback na billing
        address = order_data.get('delivery_address') or order_data.get('billing_address') or {}
        
        # Jeśli brak danych adresowych, użyj pól na głównym poziomie zamówienia
        if not address:
            address = order_data
        
        return {
            'name': f"{address.get('firstname', '')} {address.get('lastname', '')}".strip() or order_data.get('email', 'Unknown Customer'),
            'email': order_data.get('email'),
            'phone': address.get('phone'),
            'street': address.get('street1') or address.get('street'),
            'street2': address.get('street2'),
            'city': address.get('city'),
            'zip': address.get('postcode') or address.get('zip'),
            'country_code': address.get('country_code'),
            'company_name': address.get('company'),
            'vat': address.get('tax_identification_number') or address.get('nip'),
            'shoper_customer_id': order_data.get('user_id'),
        }

    @api.model
    def _create_order_lines_from_shoper(self, order, products):
        """Tworzy linie zamówienia z produktów Shoper"""
        for product_data in products:
            try:
                # Znajdź produkt w Odoo (po kodzie, EAN, nazwie)
                product = self._find_product_from_shoper(product_data)
                
                if not product:
                    # Utwórz produkt jeśli nie istnieje
                    product = self._create_product_from_shoper(product_data)
                
                # Przygotuj dane linii
                line_vals = {
                    'order_id': order.id,
                    'product_id': product.id if product else False,
                    'name': product_data.get('name', 'Unknown Product'),
                    'product_uom_qty': float(product_data.get('quantity', 1)),
                    'price_unit': float(product_data.get('price_gross', 0.0)),
                }
                
                # Dodaj SKU do opisu jeśli nie ma produktu
                if not product and product_data.get('code'):
                    line_vals['name'] += f" [SKU: {product_data.get('code')}]"
                
                # Utwórz linię
                self.env['sale.order.line'].create(line_vals)
                
            except Exception as e:
                _logger.warning(f"Failed to create order line for product {product_data.get('name')}: {str(e)}")

    @api.model
    def _find_product_from_shoper(self, product_data):
        """Znajduje produkt w Odoo na podstawie danych z Shoper"""
        Product = self.env['product.product']
        
        # 1. Szukaj po kodzie/SKU
        code = product_data.get('code') or product_data.get('sku')
        if code:
            product = Product.search([('default_code', '=', code)], limit=1)
            if product:
                _logger.debug(f"Found product by code: {code}")
                return product
        
        # 2. Szukaj po EAN
        ean = product_data.get('ean')
        if ean:
            product = Product.search([('barcode', '=', ean)], limit=1)
            if product:
                _logger.debug(f"Found product by EAN: {ean}")
                return product
        
        # 3. Szukaj po nazwie (ostatnia opcja, mniej pewna)
        name = product_data.get('name')
        if name:
            product = Product.search([('name', '=', name)], limit=1)
            if product:
                _logger.debug(f"Found product by name: {name}")
                return product
        
        _logger.debug(f"Product not found: {name} (code: {code}, ean: {ean})")
        return None

    @api.model
    def _create_product_from_shoper(self, product_data):
        """Tworzy nowy produkt w Odoo na podstawie danych z Shoper"""
        try:
            product_vals = {
                'name': product_data.get('name', 'Unnamed Product'),
                'default_code': product_data.get('code') or product_data.get('sku'),
                'barcode': product_data.get('ean'),
                'list_price': float(product_data.get('price_gross', 0.0)),
                'standard_price': float(product_data.get('price_net', 0.0)) if product_data.get('price_net') else 0.0,
                'type': 'product',
                'invoice_policy': 'order',
                'purchase_ok': False,
                'sale_ok': True,
            }
            
            product = self.env['product.product'].create(product_vals)
            _logger.info(f"Created new product: {product.name} (code: {product.default_code})")
            return product
            
        except Exception as e:
            _logger.error(f"Failed to create product: {str(e)}", exc_info=True)
            return None

    @api.model
    def _get_delivery_cost(self, order_data):
        """Pobiera koszt dostawy z danych zamówienia"""
        # Sprawdź różne możliwe miejsca gdzie może być koszt dostawy
        delivery_cost = order_data.get('delivery_cost') or order_data.get('delivery_price')
        
        if delivery_cost:
            try:
                return float(delivery_cost)
            except (ValueError, TypeError):
                return 0.0
        
        # Jeśli jest obiekt delivery, sprawdź tam
        delivery = order_data.get('delivery')
        if isinstance(delivery, dict):
            delivery_cost = delivery.get('cost') or delivery.get('price')
            if delivery_cost:
                try:
                    return float(delivery_cost)
                except (ValueError, TypeError):
                    pass
        
        return 0.0

    @api.model
    def _add_delivery_cost(self, order, order_data, delivery_cost):
        """Dodaje koszty dostawy jako linię zamówienia"""
        if not delivery_cost or delivery_cost <= 0:
            return
        
        try:
            # Znajdź lub utwórz produkt dostawy
            delivery_product = self.env.ref('odoo_shoper_integration.product_delivery_shoper', raise_if_not_found=False)
            
            if not delivery_product:
                # Utwórz produkt dostawy jeśli nie istnieje
                delivery_product = self.env['product.product'].create({
                    'name': 'Delivery Cost (Shoper)',
                    'type': 'service',
                    'list_price': 0.0,
                    'invoice_policy': 'order',
                    'sale_ok': True,
                    'purchase_ok': False,
                })
            
            # Nazwa dostawy
            delivery_name = self._get_delivery_name(order_data) or 'Delivery'
            
            # Dodaj linię z kosztem dostawy
            self.env['sale.order.line'].create({
                'order_id': order.id,
                'product_id': delivery_product.id,
                'name': delivery_name,
                'product_uom_qty': 1,
                'price_unit': delivery_cost,
            })
            
            _logger.debug(f"Added delivery cost: {delivery_cost} PLN ({delivery_name})")
            
        except Exception as e:
            _logger.warning(f"Failed to add delivery cost: {str(e)}")

    @staticmethod
    def _get_status_name(order_data):
        """Pobiera nazwę statusu z danych zamówienia"""
        status = order_data.get('status')
        if isinstance(status, dict):
            return status.get('status_name') or status.get('name')
        return None

    @staticmethod
    def _get_payment_name(order_data):
        """Pobiera nazwę metody płatności z danych zamówienia"""
        payment = order_data.get('payment')
        if isinstance(payment, dict):
            return payment.get('name')
        return None

    @staticmethod
    def _get_delivery_name(order_data):
        """Pobiera nazwę metody dostawy z danych zamówienia"""
        delivery = order_data.get('delivery')
        if isinstance(delivery, dict):
            return delivery.get('name')
        return None
