# -*- coding: utf-8 -*-

from odoo import models, fields, api
import logging

_logger = logging.getLogger(__name__)


class ResPartner(models.Model):
    _inherit = 'res.partner'

    shoper_customer_id = fields.Integer(
        string='Shoper Customer ID',
        readonly=True,
        index=True,
        help='Customer ID from Shoper',
    )

    @api.model
    def find_or_create_from_shoper(self, partner_data):
        """
        Znajduje lub tworzy partnera na podstawie danych z Shoper
        
        Args:
            partner_data: Słownik z danymi partnera
        
        Returns:
            Rekord res.partner
        """
        # Szukaj po email
        email = partner_data.get('email')
        if email:
            partner = self.search([('email', '=', email)], limit=1)
            if partner:
                # Zaktualizuj dane jeśli są nowe
                self._update_partner_if_needed(partner, partner_data)
                return partner
        
        # Szukaj po telefonie
        phone = partner_data.get('phone')
        if phone:
            partner = self.search([('phone', '=', phone)], limit=1)
            if partner:
                self._update_partner_if_needed(partner, partner_data)
                return partner
        
        # Utwórz nowego partnera
        return self._create_partner_from_shoper(partner_data)

    @api.model
    def _create_partner_from_shoper(self, partner_data):
        """Tworzy nowego partnera z danych Shoper"""
        # Przygotuj dane adresowe
        vals = {
            'name': partner_data.get('name', 'Unknown'),
            'email': partner_data.get('email'),
            'phone': partner_data.get('phone'),
            'street': partner_data.get('street'),
            'street2': partner_data.get('street2'),
            'city': partner_data.get('city'),
            'zip': partner_data.get('zip'),
            'is_company': bool(partner_data.get('company_name')),
        }
        
        # Firma
        if partner_data.get('company_name'):
            vals['name'] = partner_data['company_name']
            vals['company_name'] = partner_data['company_name']
        
        # VAT
        if partner_data.get('vat'):
            vals['vat'] = partner_data['vat']
        
        # Kraj
        country_code = partner_data.get('country_code')
        if country_code:
            country = self.env['res.country'].search([('code', '=', country_code.upper())], limit=1)
            if country:
                vals['country_id'] = country.id
        
        # Utwórz partnera
        partner = self.create(vals)
        _logger.info(f"Created new partner: {partner.name} (email: {partner.email})")
        
        return partner

    @api.model
    def _update_partner_if_needed(self, partner, partner_data):
        """Aktualizuje partnera jeśli dane się zmieniły"""
        vals = {}
        
        # Sprawdź czy telefon się zmienił
        if partner_data.get('phone') and partner_data['phone'] != partner.phone:
            vals['phone'] = partner_data['phone']
        
        # Sprawdź czy adres się zmienił
        if partner_data.get('street') and partner_data['street'] != partner.street:
            vals['street'] = partner_data['street']
        
        if partner_data.get('city') and partner_data['city'] != partner.city:
            vals['city'] = partner_data['city']
        
        if partner_data.get('zip') and partner_data['zip'] != partner.zip:
            vals['zip'] = partner_data['zip']
        
        # Aktualizuj jeśli są zmiany
        if vals:
            partner.write(vals)
            _logger.info(f"Updated partner: {partner.name}")
