#!/bin/bash

# Shoper Integration Installation Script for Odoo 18
# This script helps with quick installation and setup

set -e  # Exit on error

echo "========================================"
echo "Shoper Integration - Installation Script"
echo "========================================"
echo ""

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Function to print colored output
print_success() {
    echo -e "${GREEN}✓ $1${NC}"
}

print_error() {
    echo -e "${RED}✗ $1${NC}"
}

print_warning() {
    echo -e "${YELLOW}⚠ $1${NC}"
}

print_info() {
    echo -e "ℹ $1"
}

# Check if running as root
if [ "$EUID" -eq 0 ]; then 
    print_warning "Running as root. This might cause permission issues."
    read -p "Continue anyway? (y/n) " -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        exit 1
    fi
fi

# Step 1: Find Odoo installation
echo ""
print_info "Step 1: Locating Odoo installation..."

ODOO_PATHS=(
    "/opt/odoo/odoo"
    "/usr/lib/python3/dist-packages/odoo"
    "/opt/odoo18"
    "$HOME/odoo"
)

ODOO_PATH=""
for path in "${ODOO_PATHS[@]}"; do
    if [ -d "$path/addons" ]; then
        ODOO_PATH="$path"
        break
    fi
done

if [ -z "$ODOO_PATH" ]; then
    print_error "Could not find Odoo installation automatically."
    read -p "Enter Odoo installation path: " ODOO_PATH
    
    if [ ! -d "$ODOO_PATH/addons" ]; then
        print_error "Invalid Odoo path. Addons directory not found."
        exit 1
    fi
fi

print_success "Found Odoo at: $ODOO_PATH"

# Step 2: Locate addons directory
ADDONS_PATH="$ODOO_PATH/addons"

# Check for custom addons path
if [ -f "/etc/odoo/odoo.conf" ]; then
    CUSTOM_ADDONS=$(grep "addons_path" /etc/odoo/odoo.conf | cut -d'=' -f2 | cut -d',' -f1 | xargs)
    if [ ! -z "$CUSTOM_ADDONS" ] && [ -d "$CUSTOM_ADDONS" ]; then
        print_warning "Found custom addons path in config: $CUSTOM_ADDONS"
        read -p "Use custom addons path? (y/n) " -n 1 -r
        echo
        if [[ $REPLY =~ ^[Yy]$ ]]; then
            ADDONS_PATH="$CUSTOM_ADDONS"
        fi
    fi
fi

print_success "Using addons path: $ADDONS_PATH"

# Step 3: Copy module files
echo ""
print_info "Step 2: Copying module files..."

MODULE_SRC="$(pwd)"
MODULE_DST="$ADDONS_PATH/odoo_shoper_integration"

if [ -d "$MODULE_DST" ]; then
    print_warning "Module already exists at $MODULE_DST"
    read -p "Overwrite? (y/n) " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        rm -rf "$MODULE_DST"
    else
        print_error "Installation cancelled."
        exit 1
    fi
fi

cp -r "$MODULE_SRC" "$MODULE_DST"
print_success "Module files copied to $MODULE_DST"

# Step 4: Install Python dependencies
echo ""
print_info "Step 3: Installing Python dependencies..."

REQUIREMENTS_FILE="$MODULE_DST/requirements.txt"

if [ -f "$REQUIREMENTS_FILE" ]; then
    # Try to find pip for Odoo's Python
    PIP_CMD="pip3"
    
    if command -v pip3 &> /dev/null; then
        print_info "Using system pip3"
    elif [ -f "/opt/odoo/venv/bin/pip" ]; then
        PIP_CMD="/opt/odoo/venv/bin/pip"
        print_info "Using Odoo virtualenv pip"
    fi
    
    $PIP_CMD install -r "$REQUIREMENTS_FILE"
    print_success "Dependencies installed"
else
    print_warning "requirements.txt not found, skipping dependencies"
fi

# Step 5: Set permissions
echo ""
print_info "Step 4: Setting file permissions..."

# Find Odoo user
ODOO_USER="odoo"
if ! id "$ODOO_USER" &>/dev/null; then
    ODOO_USER=$(ls -l "$ODOO_PATH" | head -n 2 | tail -n 1 | awk '{print $3}')
fi

if id "$ODOO_USER" &>/dev/null; then
    chown -R "$ODOO_USER:$ODOO_USER" "$MODULE_DST" 2>/dev/null || print_warning "Could not change ownership. Run with sudo if needed."
    print_success "Permissions set for user: $ODOO_USER"
else
    print_warning "Could not determine Odoo user. Please set permissions manually."
fi

# Step 6: Restart Odoo
echo ""
print_info "Step 5: Restarting Odoo service..."

RESTART_NEEDED=true

if systemctl is-active --quiet odoo; then
    read -p "Restart Odoo service now? (y/n) " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        sudo systemctl restart odoo
        print_success "Odoo service restarted"
        RESTART_NEEDED=false
    fi
elif systemctl is-active --quiet odoo18; then
    read -p "Restart Odoo18 service now? (y/n) " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        sudo systemctl restart odoo18
        print_success "Odoo18 service restarted"
        RESTART_NEEDED=false
    fi
fi

if $RESTART_NEEDED; then
    print_warning "Please restart Odoo manually"
fi

# Step 7: Summary
echo ""
echo "========================================"
print_success "Installation completed!"
echo "========================================"
echo ""
print_info "Next steps:"
echo "1. Go to Odoo Apps menu"
echo "2. Click 'Update Apps List'"
echo "3. Search for 'Shoper Integration'"
echo "4. Click Install"
echo "5. Configure Shoper API in Settings"
echo ""
print_info "Module location: $MODULE_DST"
print_info "Documentation: $MODULE_DST/README.md"
print_info "Examples: $MODULE_DST/EXAMPLES.md"
echo ""
print_warning "Remember to update the apps list in Odoo!"
echo ""
