# 🔥 HOTFIX v1.1.1 - Krytyczna Poprawka Endpointa Produktów

**Data:** 2025-10-05  
**Wersja:** 1.1.1  
**Priorytet:** KRYTYCZNY ⚠️

## 🐛 Problem

Moduł nie mógł poprawnie pobierać zamówień z produktami ze sklepu Shoper z powodu **błędnego endpointa API**.

### Objaw
- Zamówienia nie są importowane
- Logi pokazują błędy 404 lub puste produkty
- Synchronizacja kończy się niepowodzeniem

### Przyczyna
Kod używał nieistniejącego endpointa API Shoper:
```python
# ❌ BŁĘDNY ENDPOINT (nie istnieje w API Shoper!)
GET /webapi/rest/orders/{id}/products
```

Według oficjalnej dokumentacji API Shoper, produkty zamówienia są dostępne pod:
```python
# ✅ PRAWIDŁOWY ENDPOINT
GET /webapi/rest/order-products?filters={"order_id":123}
```

## ✅ Rozwiązanie

### Co zostało naprawione?

**Plik:** `models/shoper_api_client.py`  
**Metoda:** `get_order_products()`

#### Przed (v1.1.0):
```python
def get_order_products(self, order_id):
    """Pobiera produkty z zamówienia"""
    result = self._make_request('GET', f'orders/{order_id}/products')
    products = result.get('list', []) if result else []
    # ...
```

#### Po (v1.1.1):
```python
def get_order_products(self, order_id):
    """Pobiera produkty z zamówienia"""
    # Przygotuj filtr JSON
    filters_json = json.dumps({"order_id": order_id})
    params = {
        'filters': filters_json,
        'limit': 50
    }
    
    result = self._make_request('GET', 'order-products', params=params)
    products = result.get('list', []) if result else []
    # ...
```

### Zmieniony Workflow
```
┌─────────────────────────────────────────────────────┐
│  Przed (v1.1.0) - BŁĄD                              │
├─────────────────────────────────────────────────────┤
│  1. GET /orders/{id}              ✓ OK              │
│  2. GET /orders/{id}/products     ✗ 404 NOT FOUND   │
│  3. Order imported WITHOUT products                 │
└─────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────┐
│  Po (v1.1.1) - NAPRAWIONE                           │
├─────────────────────────────────────────────────────┤
│  1. GET /orders/{id}                      ✓ OK      │
│  2. GET /order-products?filters={...}     ✓ OK      │
│  3. Order imported WITH products          ✓ OK      │
└─────────────────────────────────────────────────────┘
```

## 📦 Jak Zastosować Poprawkę

### Opcja 1: Automatyczna (Zalecana)

1. **Zatrzymaj Odoo:**
   ```bash
   docker-compose stop odoo-web
   # lub
   sudo systemctl stop odoo
   ```

2. **Zaktualizuj plik:**
   ```bash
   cd /path/to/odoo/addons/odoo_shoper_integration
   git pull  # jeśli używasz git
   # lub skopiuj nowy plik models/shoper_api_client.py
   ```

3. **Uruchom Odoo:**
   ```bash
   docker-compose up -d odoo-web
   # lub
   sudo systemctl start odoo
   ```

4. **Zaktualizuj moduł w Odoo:**
   - Apps → Odoo Shoper Integration → Upgrade
   - Lub: `odoo-bin -u odoo_shoper_integration -d your_database`

### Opcja 2: Ręczna

1. **Otwórz plik:**
   ```
   /path/to/odoo/addons/odoo_shoper_integration/models/shoper_api_client.py
   ```

2. **Znajdź metodę `get_order_products`** (około linii 190)

3. **Zamień całą metodę** na nową wersję (patrz sekcja "Po" powyżej)

4. **Dodaj import json** na początku pliku (jeśli nie ma):
   ```python
   import json
   ```

5. **Zrestartuj Odoo**

## 🧪 Test Po Poprawce

### 1. Test Połączenia
```
Odoo > Shoper > Configuration
→ Kliknij "Test Connection"
→ Powinno pokazać liczbę zamówień
```

### 2. Test Synchronizacji
```
Odoo > Shoper > Configuration
→ Kliknij "Synchronize Orders Now"
→ Obserwuj logi
```

### 3. Sprawdź Logi
```bash
tail -f /var/log/odoo/odoo.log | grep -i "product"
```

Powinien wyświetlać:
```
✓ Found X products for order 12345
✓ Added order 12345 (..., 5 products)
```

A NIE:
```
✗ Shoper API error 404: orders/12345/products
✗ No products found for order 12345
```

### 4. Sprawdź Zamówienia
```
Odoo > Shoper > Orders
→ Otwórz zamówienie
→ Zakładka "Order Lines"
→ Powinny być produkty!
```

## 📊 Rezultaty

### Przed Poprawką:
- ❌ Zamówienia bez produktów
- ❌ Błędy 404 w logach
- ❌ Synchronizacja niepełna

### Po Poprawce:
- ✅ Zamówienia z pełną listą produktów
- ✅ Prawidłowe API calls
- ✅ Kompletna synchronizacja

## 🔍 Weryfikacja Wersji

Sprawdź czy masz poprawioną wersję:

```python
# W pliku shoper_api_client.py, na początku powinno być:
"""
POPRAWKA v1.1.1: Naprawiono endpoint dla produktów zamówienia
"""
```

Lub w Odoo:
```
Apps > Odoo Shoper Integration
Version: 18.0.1.1.1
```

## 📝 Changelog

### [1.1.1] - 2025-10-05

**CRITICAL FIX:**
- Fixed `get_order_products()` to use correct API endpoint
- Changed from `orders/{id}/products` to `order-products` with filter
- Added proper JSON filter formatting
- Added detailed logging for product fetching
- Updated module version to 1.1.1

**Impact:**
- Orders now import WITH products (previously without)
- No more 404 errors for product endpoints
- Complete order data synchronization

## 🆘 Support

### Jeśli Nadal Nie Działa:

1. **Sprawdź logi szczegółowo:**
   ```bash
   tail -f /var/log/odoo/odoo.log | grep -E "shoper|order|product" > debug.log
   ```

2. **Test API ręcznie:**
   ```bash
   curl -H "Authorization: Bearer YOUR_TOKEN" \
        "https://alpmeb.pl/webapi/rest/order-products?filters={\"order_id\":12345}"
   ```

3. **Wyczyść cache Odoo:**
   ```bash
   rm -rf ~/.local/share/Odoo/sessions/*
   ```

4. **Sprawdź uprawnienia API:**
   - W Shoper: Settings > API
   - Upewnij się że token ma scope: `orders_read`

## 📚 Dokumentacja

- **API Shoper:** https://developers.shoper.pl/developers/api/resources/orders
- **Endpoint order-products:** https://developers.shoper.pl/developers/api/resources/order-products

## ⚠️ Ważne Notatki

1. Ta poprawka jest **kompatybilna wstecz** - nie wymaga zmian w bazie danych
2. Istniejące zamówienia bez produktów **NIE** zostaną automatycznie uzupełnione
3. Aby uzupełnić stare zamówienia - usuń je w Odoo i uruchom ponowną synchronizację
4. Nowe zamówienia będą importowane poprawnie od razu

## 🎯 Następne Kroki

Po zastosowaniu poprawki:

1. ✅ Uruchom test synchronizacji
2. ✅ Zweryfikuj kilka zamówień
3. ✅ Włącz automatyczną synchronizację
4. ✅ Monitoruj logi przez pierwsze 24h

---

**Status:** ✅ ROZWIĄZANE  
**Wersja:** 1.1.1  
**Data:** 2025-10-05  
**Priorytet:** KRYTYCZNY - natychmiastowe wdrożenie zalecane
