# Shoper Integration - Usage Examples

## Example 1: Basic Setup and First Sync

```python
# 1. Create configuration
config = env['shoper.config'].create({
    'shop_url': 'https://alpmeb.pl',
    'access_token': '8178507ff9b80d67a10fd27bec40567a6d417b28b6e5255063b9f9830eb5d2df',
    'active': True,
    'max_orders_per_sync': 50,
    'default_order_state': 'draft',
})

# 2. Test connection
config.test_connection()

# 3. Run first sync
config.action_sync_orders()

# 4. Check imported orders
orders = env['sale.order'].search([('is_shoper_order', '=', True)])
print(f"Imported {len(orders)} orders")
```

## Example 2: Incremental Sync (Import Only New Orders)

```python
# Configuration with incremental sync
config = env['shoper.config'].search([('active', '=', True)], limit=1)

# First sync - imports all orders
config.action_sync_orders()

# Wait some time...

# Second sync - imports only new/updated orders since last sync
config.action_sync_orders()
# Uses config.last_order_update automatically for filtering
```

## Example 3: Import Orders from Specific Date

```python
from datetime import datetime, timedelta

# Import only orders from last 30 days
config = env['shoper.config'].search([('active', '=', True)], limit=1)
config.import_from_date = datetime.now() - timedelta(days=30)
config.action_sync_orders()
```

## Example 4: Automatic Synchronization Setup

```python
# Enable automatic sync every 2 hours
config = env['shoper.config'].search([('active', '=', True)], limit=1)
config.write({
    'auto_sync_enabled': True,
    'sync_interval_hours': 2,
    'max_orders_per_sync': 100,
})

# The cron job will now run automatically
```

## Example 5: Find Order by Shoper ID

```python
# Find Odoo order by original Shoper order ID
shoper_order_id = 23046
order = env['sale.order'].search([
    ('shoper_order_id', '=', shoper_order_id)
], limit=1)

if order:
    print(f"Order found: {order.name}")
    print(f"Partner: {order.partner_id.name}")
    print(f"Total: {order.amount_total}")
    print(f"Shoper Status: {order.shoper_status_name}")
else:
    print("Order not found in Odoo")
```

## Example 6: Get Shoper Orders by Status

```python
# Get all orders with specific Shoper status
status_name = "Completed"
orders = env['sale.order'].search([
    ('shoper_status_name', '=', status_name)
])

print(f"Found {len(orders)} orders with status '{status_name}'")
for order in orders:
    print(f"  - {order.name}: {order.partner_id.name} ({order.amount_total} {order.currency_id.name})")
```

## Example 7: Update Order from Shoper

```python
from models.shoper_api_client import ShoperAPIClient

# Get config and create API client
config = env['shoper.config'].search([('active', '=', True)], limit=1)
client = ShoperAPIClient(config.shop_url, config.access_token)

# Fetch updated order data from Shoper
shoper_order_id = 23046
order_data = client.get_order_details(shoper_order_id)

if order_data:
    # Find order in Odoo
    order = env['sale.order'].search([
        ('shoper_order_id', '=', shoper_order_id)
    ], limit=1)
    
    if order:
        # Update order with new data
        order.update_from_shoper(order_data)
        print(f"Order {order.name} updated successfully")
```

## Example 8: Check Synchronization Statistics

```python
# Get configuration and check stats
config = env['shoper.config'].search([('active', '=', True)], limit=1)

print("=== Synchronization Statistics ===")
print(f"Last sync: {config.last_sync_date}")
print(f"Last order update: {config.last_order_update}")
print(f"Total orders imported: {config.total_orders_imported}")
print(f"Last import count: {config.last_import_count}")

# Count Shoper orders in Odoo
orders_count = env['sale.order'].search_count([
    ('is_shoper_order', '=', True)
])
print(f"Shoper orders in Odoo: {orders_count}")
```

## Example 9: Custom Product Mapping

```python
# Before sync, pre-map products
shoper_code = "PROD-123"
odoo_product = env['product.product'].search([
    ('default_code', '=', 'INTERNAL-123')
], limit=1)

if odoo_product:
    # Update Odoo product code to match Shoper
    odoo_product.default_code = shoper_code
    print(f"Mapped {odoo_product.name} to Shoper code {shoper_code}")

# Now sync will match this product automatically
config = env['shoper.config'].search([('active', '=', True)], limit=1)
config.action_sync_orders()
```

## Example 10: Batch Processing with Filters

```python
from datetime import datetime

# Process orders from specific date range
start_date = datetime(2024, 1, 1)
end_date = datetime(2024, 1, 31)

config = env['shoper.config'].search([('active', '=', True)], limit=1)

# Set date filter
config.import_from_date = start_date
config.max_orders_per_sync = 200

# Run sync
config.action_sync_orders()

# Find imported orders in date range
orders = env['sale.order'].search([
    ('is_shoper_order', '=', True),
    ('shoper_date', '>=', start_date),
    ('shoper_date', '<=', end_date),
])

print(f"Processed {len(orders)} orders from {start_date.date()} to {end_date.date()}")
```

## Example 11: Error Handling and Logging

```python
import logging

_logger = logging.getLogger(__name__)

def safe_sync_orders(config_id):
    """Safely synchronize orders with error handling"""
    try:
        config = env['shoper.config'].browse(config_id)
        
        if not config.exists():
            _logger.error(f"Config {config_id} not found")
            return False
        
        if not config.active:
            _logger.warning(f"Config {config_id} is not active")
            return False
        
        # Run sync
        _logger.info(f"Starting sync for {config.shop_url}")
        result = config.action_sync_orders()
        _logger.info(f"Sync completed: {config.last_import_count} orders")
        
        return True
        
    except Exception as e:
        _logger.error(f"Sync failed: {str(e)}", exc_info=True)
        return False

# Use it
config = env['shoper.config'].search([('active', '=', True)], limit=1)
safe_sync_orders(config.id)
```

## Example 12: Monitoring and Alerts

```python
from datetime import datetime, timedelta

def check_sync_health():
    """Check if synchronization is working properly"""
    config = env['shoper.config'].search([('active', '=', True)], limit=1)
    
    if not config:
        return "No active configuration"
    
    # Check if sync is recent
    if config.last_sync_date:
        hours_since_sync = (datetime.now() - config.last_sync_date).total_seconds() / 3600
        
        if hours_since_sync > config.sync_interval_hours * 2:
            return f"WARNING: Last sync was {hours_since_sync:.1f} hours ago"
    
    # Check if orders are being imported
    if config.last_import_count == 0:
        recent_orders = env['sale.order'].search_count([
            ('is_shoper_order', '=', True),
            ('create_date', '>=', datetime.now() - timedelta(days=1)),
        ])
        
        if recent_orders == 0:
            return "WARNING: No orders imported in last 24 hours"
    
    return "OK"

# Monitor
status = check_sync_health()
print(f"Sync Health: {status}")
```

## Example 13: API Client Direct Usage

```python
from models.shoper_api_client import ShoperAPIClient

# Create API client
client = ShoperAPIClient(
    'https://alpmeb.pl',
    '8178507ff9b80d67a10fd27bec40567a6d417b28b6e5255063b9f9830eb5d2df'
)

# Test connection
if client.test_connection():
    print("Connected to Shoper API")
    
    # Get orders count
    total = client.get_orders_count()
    print(f"Total orders in Shoper: {total}")
    
    # Fetch first page
    orders = client.get_orders_page(limit=10, page=1)
    print(f"Fetched {len(orders)} orders")
    
    # Get order details
    if orders:
        order_id = orders[0]['order_id']
        details = client.get_order_details(order_id)
        products = client.get_order_products(order_id)
        print(f"Order {order_id} has {len(products)} products")
```

## Example 14: Bulk Update Status

```python
# Update all Shoper orders to specific state
orders = env['sale.order'].search([
    ('is_shoper_order', '=', True),
    ('state', '=', 'draft'),
])

for order in orders:
    # Confirm orders based on Shoper status
    if order.shoper_status_id in [2, 3]:  # Example: status 2 or 3 = confirmed
        order.action_confirm()
        print(f"Confirmed order {order.name}")
```

## Example 15: Export Report

```python
import csv
from datetime import datetime

# Generate report of Shoper orders
orders = env['sale.order'].search([
    ('is_shoper_order', '=', True),
])

# Export to CSV
filename = f'/tmp/shoper_orders_{datetime.now().strftime("%Y%m%d")}.csv'
with open(filename, 'w', newline='', encoding='utf-8') as csvfile:
    writer = csv.writer(csvfile)
    
    # Header
    writer.writerow([
        'Odoo Order', 'Shoper ID', 'Date', 'Customer', 'Total', 
        'Status', 'Payment', 'Delivery'
    ])
    
    # Data
    for order in orders:
        writer.writerow([
            order.name,
            order.shoper_order_id,
            order.date_order,
            order.partner_id.name,
            order.amount_total,
            order.shoper_status_name,
            order.shoper_payment_method,
            order.shoper_delivery_method,
        ])

print(f"Report exported to {filename}")
```

## Tips and Best Practices

### 1. Start Small
Begin with recent orders (last 7-30 days) to test the integration.

### 2. Monitor Regularly
Check sync statistics and logs daily for the first week.

### 3. Product Mapping
Pre-create products in Odoo with matching codes to avoid duplicates.

### 4. Customer Deduplication
Review imported customers and merge duplicates after initial import.

### 5. Status Mapping
Document which Shoper statuses map to which Odoo states.

### 6. Backup Before Bulk Operations
Always backup before running bulk updates or imports.

### 7. Use Staging Environment
Test synchronization in staging before production.

### 8. API Rate Limits
Respect Shoper API rate limits - use reasonable sync intervals.

### 9. Error Handling
Always use try-except blocks for production code.

### 10. Logging
Enable detailed logging during initial setup and troubleshooting.
