# Changelog

All notable changes to the Shoper Integration module will be documented in this file.

## [1.1.1] - 2025-10-05 🔥 CRITICAL HOTFIX

### 🚨 Critical Fix - Order Products Endpoint

#### Fixed
- **CRITICAL**: Fixed order products API endpoint - was using non-existent `orders/{id}/products`
- **Corrected**: Changed to proper `order-products` endpoint with JSON filter
- **Impact**: Orders now import WITH products (previously imported without products)
- **Status**: This was causing orders to fail importing completely or importing without product lines

#### Changed
- **API Client**: `get_order_products()` method completely rewritten
- **Endpoint**: Now uses `GET /order-products?filters={"order_id":123}`
- **Logging**: Added detailed debug logs for product fetching
- **Filter Format**: Proper JSON filter formatting for Shoper API

#### Technical Details

```python
# BEFORE v1.1.1 (INCORRECT):
def get_order_products(self, order_id):
    result = self._make_request('GET', f'orders/{order_id}/products')
    # This endpoint DOES NOT EXIST in Shoper API!
    # Result: 404 errors or empty product lists

# AFTER v1.1.1 (CORRECT):
def get_order_products(self, order_id):
    filters_json = json.dumps({"order_id": order_id})
    params = {'filters': filters_json, 'limit': 50}
    result = self._make_request('GET', 'order-products', params=params)
    # This is the CORRECT Shoper API endpoint
    # Result: Products properly fetched and imported
```

#### Impact Assessment
- **Before Fix**: Orders imported WITHOUT products (broken)
- **After Fix**: Orders imported WITH products (working)
- **Backward Compatible**: Yes - no database changes needed
- **Existing Orders**: Will NOT be automatically fixed (re-import needed)

#### Upgrade Instructions
1. **Stop Odoo** service
2. **Replace** `models/shoper_api_client.py` file
3. **Restart Odoo** service
4. **Update module** in Odoo (Apps → Upgrade)
5. **Test** synchronization
6. *Optional*: Delete and re-import old orders to get products

#### Verification
After upgrade, check logs for:
```
✓ "Found X products for order {id}"
✗ "Shoper API error 404: orders/{id}/products" (should NOT appear)
```

### Breaking Changes
- None - fully backward compatible

### Migration
- No database migration needed
- No configuration changes needed
- Existing orders with missing products can be re-imported

---

## [1.1.0] - 2025-10-02

### 🚀 Major Improvements - Fixed Incremental Synchronization

#### Fixed
- **Critical**: Fixed incremental synchronization using `updated_at` field
- **API Client**: Corrected Shoper API filter format (query parameters instead of JSON)
- **Update Detection**: Implemented proper change detection in `update_from_shoper()`
- **Sync Strategy**: Added smart sync mode selection (INCREMENTAL/FULL_FROM_DATE/FULL_ALL)

#### Added
- **New Field**: `shoper_paid` - Boolean field for payment status tracking
- **New Field**: `total_orders_updated` - Statistics for updated orders
- **New Field**: `last_update_count` - Count of orders updated in last sync
- **Order Sorting**: Orders now sorted by `updated_at` DESC for better sync
- **Change Logging**: Detailed logging of what changed in each order
- **Chatter Integration**: Order updates now posted to message/chatter
- **Sync Buffer**: Added 1-minute buffer to avoid missing edge-case updates
- **Enhanced Logging**: Comprehensive sync summary with mode and statistics

#### Improved
- **API Client**: Enhanced `get_orders_filtered()` with proper filter handling
- **API Client**: Added `get_orders_by_updated_at()` dedicated method
- **API Client**: Improved `_prepare_filter_params()` for Shoper API format
- **Sync Logic**: Better handling of first sync vs incremental sync
- **Performance**: Commit every 20 orders during sync (was: end of sync)
- **Error Handling**: Better error messages with full exception info
- **Statistics**: More detailed sync statistics in config

#### Changed
- **Breaking**: API filter format changed from JSON to query parameters
- **update_from_shoper()**: Now returns boolean indicating if changes were made
- **Sync Summary**: Enhanced notification with sync mode and detailed counts
- **View Updates**: Added `shoper_paid` field to form and tree views
- **Filters**: Added "Paid/Unpaid" filters for Shoper orders

### Technical Details

#### API Client Changes
```python
# OLD (incorrect):
filters = {'updated_at': {'>=': '2024-01-01'}}
params['filters'] = json.dumps(filters)

# NEW (correct):
filters = {'updated_at': {'>=': '2024-01-01'}}
params['updated_at[>=]'] = '2024-01-01'  # Shoper API format
```

#### Synchronization Strategy
1. **INCREMENTAL**: Uses `last_order_update` with 1-minute buffer
2. **FULL_FROM_DATE**: Uses `import_from_date` (first sync with date filter)
3. **FULL_ALL**: No filters (first sync, all orders)

#### Performance Improvements
- Commits every 20 orders (reduces memory usage)
- Better progress logging
- Skip unchanged orders early
- Optimized change detection

### Migration Guide

#### Upgrading from 1.0.0

1. **Backup your database** before upgrading
2. Update module files
3. Upgrade module in Odoo Apps
4. Run database update to add new fields:
   - `shoper_paid` on `sale.order`
   - `total_orders_updated` on `shoper.config`
   - `last_update_count` on `shoper.config`
5. **Important**: First sync after upgrade will re-check recent orders
6. Monitor logs for the first few syncs

#### No Data Loss
- All existing orders preserved
- `last_order_update` field preserved
- Next sync will continue from where it left off

### Testing Recommendations

After upgrade, verify:
1. ✅ First sync detects and updates changed orders
2. ✅ `shoper_paid` field populated correctly
3. ✅ Change detection works (modify order in Shoper, re-sync)
4. ✅ Statistics updated correctly
5. ✅ Logs show sync mode (INCREMENTAL/FULL)

---

## [1.0.0] - 2025-01-15

### Added
- Initial release of Shoper Integration module for Odoo 18
- Automatic order synchronization from Shoper to Odoo
- Incremental synchronization using `updated_at` field
- Manual synchronization via button
- Scheduled synchronization via cron job
- Customer creation and update from order data
- Product matching by code, EAN, and name
- Automatic product creation if no match found
- Delivery cost handling as separate order line
- Shoper status tracking in Odoo
- Payment and delivery method information
- Raw JSON data storage for debugging
- Multi-company support
- Configuration UI with test connection
- API client with error handling and retry logic
- Comprehensive logging
- README and installation guide
- Usage examples and documentation

### Features
- **Order Import**: Full order data including customer, products, totals
- **Customer Management**: Automatic partner creation/update with address
- **Product Mapping**: Smart matching and automatic creation
- **Status Sync**: Shoper order status visible in Odoo
- **Incremental Sync**: Only fetch changed orders
- **Batch Processing**: Configurable batch sizes and limits
- **Error Recovery**: Graceful error handling with detailed logs
- **API Integration**: Complete Shoper REST API client
- **Scheduled Jobs**: Automatic synchronization via cron
- **Manual Trigger**: On-demand sync button

### Technical
- Python 3.10+ compatible
- Odoo 18.0 compatible
- RESTful API integration
- Bearer token authentication
- JSON data handling
- PostgreSQL optimized queries
- Proper indexing on key fields
- Transaction safety
- UTF-8 encoding support

### Security
- Access rights for sales managers and users
- Secure token storage
- SQL injection prevention
- XSS protection in views

### Documentation
- README.md - Main documentation
- INSTALL.md - Installation guide
- EXAMPLES.md - Usage examples
- Inline code documentation
- User-friendly error messages

---

## [Unreleased]

### Planned Features
- Product stock synchronization
- Order status update to Shoper
- Invoice synchronization
- Shipping label generation
- Advanced product mapping rules
- Custom field mapping
- Webhook support for real-time sync
- Multi-language support
- Enhanced reporting
- Order notes synchronization
- Attachment handling
- Delivery tracking integration

### Known Issues
- None reported yet

### Notes
- Based on Shoper API documentation version 2024
- Tested with Shoper shops on Polish market
- Requires Shoper API access (paid plan)
