# Module Structure - Shoper Integration for Odoo 18

## Directory Structure

```
odoo_shoper_integration/
├── __init__.py                 # Main module initialization
├── __manifest__.py             # Module manifest with metadata
├── README.md                   # Main documentation
├── INSTALL.md                  # Installation guide
├── EXAMPLES.md                 # Usage examples
├── CHANGELOG.md                # Version history
├── LICENSE                     # LGPL-3 license
├── requirements.txt            # Python dependencies
├── install.sh                  # Installation script
│
├── models/                     # Python models
│   ├── __init__.py
│   ├── shoper_config.py        # Configuration model
│   ├── sale_order.py           # Extended sale.order
│   ├── res_partner.py          # Extended res.partner
│   └── shoper_api_client.py    # API client library
│
├── views/                      # XML views
│   ├── shoper_config_views.xml # Configuration views
│   ├── sale_order_views.xml    # Order views
│   └── shoper_menu.xml         # Menu structure
│
├── wizards/                    # Wizard models (future)
│   └── __init__.py
│
├── security/                   # Access rights
│   └── ir.model.access.csv     # Model access rules
│
├── data/                       # Data files
│   └── ir_cron.xml             # Scheduled actions
│
└── static/                     # Static assets
    └── description/
        ├── index.html          # Module description
        └── icon.png            # Module icon (add manually)
```

## Key Components

### 1. Models

#### shoper_config.py
- **Purpose**: Store Shoper API configuration
- **Key Fields**:
  - `shop_url`: Shoper shop URL
  - `access_token`: API token
  - `auto_sync_enabled`: Enable automatic sync
  - `sync_interval_hours`: Sync frequency
  - `last_sync_date`: Last synchronization timestamp
  - `last_order_update`: Last order update (for incremental sync)
  - `max_orders_per_sync`: Batch size limit
  - `total_orders_imported`: Statistics counter

- **Key Methods**:
  - `test_connection()`: Test API connectivity
  - `action_sync_orders()`: Manual synchronization
  - `cron_sync_orders()`: Automatic synchronization (called by cron)
  - `get_active_config()`: Get active configuration

#### sale_order.py
- **Purpose**: Extend sale.order with Shoper data
- **Key Fields**:
  - `shoper_order_id`: Original Shoper order ID (unique)
  - `shoper_status_id`: Shoper status ID
  - `shoper_status_name`: Shoper status name
  - `shoper_date`: Order date from Shoper
  - `shoper_updated_at`: Last update timestamp
  - `shoper_payment_method`: Payment method name
  - `shoper_delivery_method`: Delivery method name
  - `is_shoper_order`: Flag for Shoper orders
  - `shoper_raw_data`: Raw JSON from API

- **Key Methods**:
  - `create_from_shoper()`: Create order from Shoper data
  - `update_from_shoper()`: Update existing order
  - `_prepare_partner_from_shoper()`: Extract partner data
  - `_create_order_lines_from_shoper()`: Create order lines
  - `_find_product_from_shoper()`: Match products
  - `_create_product_from_shoper()`: Create missing products
  - `_add_delivery_cost()`: Add delivery line

#### res_partner.py
- **Purpose**: Extend res.partner with Shoper data
- **Key Fields**:
  - `shoper_customer_id`: Shoper customer ID

- **Key Methods**:
  - `find_or_create_from_shoper()`: Find or create partner
  - `_create_partner_from_shoper()`: Create new partner
  - `_update_partner_if_needed()`: Update partner data

#### shoper_api_client.py
- **Purpose**: Shoper REST API client library
- **Key Methods**:
  - `test_connection()`: Test API access
  - `get_orders_count()`: Get total orders count
  - `get_orders_page()`: Fetch orders page
  - `get_order_details()`: Get order details
  - `get_order_products()`: Get order products
  - `get_orders_filtered()`: Fetch with filters
  - `_make_request()`: HTTP request wrapper

### 2. Views

#### shoper_config_views.xml
- Form view for configuration
- Tree view for multiple configs
- Action for opening configurations

#### sale_order_views.xml
- Extended form view with Shoper tab
- Extended tree view with Shoper fields
- Search filters for Shoper orders
- Action for Shoper orders list

#### shoper_menu.xml
- Main Shoper menu
- Configuration submenu
- Orders submenu
- Link in Sales settings

### 3. Security

#### ir.model.access.csv
- Manager: Full access (CRUD)
- User: Read-only access

### 4. Data

#### ir_cron.xml
- Scheduled action for automatic sync
- Runs every hour by default
- Calls `shoper.config.cron_sync_orders()`

### 5. Static Files

#### static/description/
- Module icon (icon.png)
- HTML description (index.html)

## Data Flow

### Order Import Flow

```
1. Trigger (Manual/Cron)
   ↓
2. Get Active Config
   ↓
3. Create API Client
   ↓
4. Fetch Orders from Shoper
   - Apply filters (updated_at, date, etc.)
   - Paginate through results
   - Skip existing orders
   ↓
5. For Each Order:
   a. Get Full Order Details
   b. Get Order Products
   c. Find/Create Customer
   d. Match/Create Products
   e. Create Sale Order
   f. Create Order Lines
   g. Add Delivery Cost
   ↓
6. Update Statistics
   - last_sync_date
   - last_order_update
   - total_orders_imported
```

### Incremental Sync Flow

```
First Sync:
- last_order_update = NULL
- Imports all orders (or from import_from_date)
- Sets last_order_update to latest order's updated_at

Next Syncs:
- Uses last_order_update as filter
- Only fetches orders with updated_at > last_order_update
- Updates last_order_update to new latest
```

## API Integration

### Shoper API Endpoints Used

1. **GET /orders**
   - List orders
   - Supports pagination
   - Supports filters (JSON)
   - Returns order list + count

2. **GET /orders/{id}**
   - Get order details
   - Returns full order data

3. **GET /orders/{id}/products**
   - Get order products
   - Returns product list

### Authentication

- Method: Bearer Token
- Header: `Authorization: Bearer {token}`
- Token from Shoper admin panel

### Data Mapping

#### Shoper → Odoo Order

| Shoper Field | Odoo Field | Notes |
|--------------|------------|-------|
| order_id | shoper_order_id | Unique identifier |
| date | date_order | Order date |
| updated_at | shoper_updated_at | For incremental sync |
| status_id | shoper_status_id | Status ID |
| status.name | shoper_status_name | Status name |
| sum | amount_total | Order total |
| email | partner_id.email | Customer email |

#### Shoper → Odoo Partner

| Shoper Field | Odoo Field |
|--------------|------------|
| delivery_address.firstname | name (part) |
| delivery_address.lastname | name (part) |
| email | email |
| delivery_address.phone | phone |
| delivery_address.street1 | street |
| delivery_address.city | city |
| delivery_address.postcode | zip |
| delivery_address.country_code | country_id |

#### Shoper → Odoo Product

| Shoper Field | Odoo Field | Priority |
|--------------|------------|----------|
| code / sku | default_code | 1 (match) |
| ean | barcode | 2 (match) |
| name | name | 3 (match) |
| price_gross | list_price | New product |

## Configuration

### Environment Variables (Optional)

Can be set in Odoo config file:

```ini
[options]
# Shoper specific settings (if needed)
shoper_api_timeout = 30
shoper_batch_size = 50
```

### Module Dependencies

Required Odoo modules:
- base
- sale_management
- stock
- account

Python packages:
- requests >= 2.28.0

## Troubleshooting

### Common Issues

1. **Import fails with 404 error**
   - Check Shop URL format
   - Verify API token is valid
   - Ensure API access is enabled in Shoper

2. **Orders not importing**
   - Check filters (import_from_date)
   - Verify last_order_update is not blocking
   - Check max_orders_per_sync limit

3. **Product duplicates**
   - Improve product matching (update codes)
   - Pre-create products with matching codes
   - Use product templates

4. **Customer duplicates**
   - Check email matching logic
   - Manually merge duplicates
   - Update customer data in Odoo

### Debug Mode

Enable debug logging in Odoo config:

```ini
[options]
log_level = debug
log_handler = odoo.addons.odoo_shoper_integration:DEBUG
```

### Test Mode

Test in Odoo shell:

```python
# Start Odoo shell
./odoo-bin shell -d your_database

# In shell
config = env['shoper.config'].search([('active', '=', True)], limit=1)
config.action_sync_orders()
```

## Performance

### Optimization Tips

1. **Batch Size**: Keep max_orders_per_sync between 50-100
2. **Sync Interval**: 1-4 hours depending on order volume
3. **Incremental Sync**: Always enabled (automatic)
4. **Indexing**: All key fields are indexed
5. **Pagination**: API client handles pagination automatically

### Resource Usage

- Memory: ~50MB per 100 orders
- CPU: Light (mostly API I/O)
- Database: ~5-10 KB per order with products
- Network: Depends on order size

## Future Enhancements

See CHANGELOG.md for planned features.

Priority improvements:
1. Webhook support for real-time sync
2. Status sync back to Shoper
3. Stock synchronization
4. Invoice synchronization
5. Advanced mapping rules
6. Multi-language support

## Support

For technical questions:
- Check logs: `/var/log/odoo/odoo.log`
- Review documentation: README.md, EXAMPLES.md
- Contact Odoo partner

For Shoper API issues:
- Check Shoper API documentation
- Contact Shoper support
- Verify API plan includes necessary features
