# API Usage Examples - Shoper Integration v1.1.0

## 📚 Python API Examples

This document shows how to use the Shoper API client programmatically in Odoo.

---

## 🔧 Basic Usage

### Example 1: Initialize Client

```python
from odoo.addons.odoo_shoper_integration.models.shoper_api_client import ShoperAPIClient

# Create client
client = ShoperAPIClient(
    shop_url='https://yourshop.pl',
    access_token='your_api_token_here'
)

# Test connection
if client.test_connection():
    print("Connected successfully!")
else:
    print("Connection failed")
```

### Example 2: Get Order Count

```python
# Get total orders
total = client.get_orders_count()
print(f"Total orders in shop: {total}")

# Get orders matching filter
filters = {'status_id': 2}  # Status "In progress"
count = client.get_orders_count(filters)
print(f"Orders in progress: {count}")
```

---

## 📥 Fetching Orders

### Example 3: Get Single Page

```python
# Get first page (50 orders)
orders = client.get_orders_page(limit=50, page=1)
print(f"Fetched {len(orders)} orders")

for order in orders:
    print(f"Order #{order['order_id']}: {order.get('email')}")
```

### Example 4: Get Orders with Filters

```python
from datetime import datetime, timedelta

# Get orders from last 7 days
date_from = datetime.now() - timedelta(days=7)
filters = {
    'date': {'>=': date_from.strftime('%Y-%m-%d %H:%M:%S')}
}

orders = client.get_orders_page(limit=50, page=1, filters=filters)
print(f"Orders from last 7 days: {len(orders)}")
```

### Example 5: Get Updated Orders (Incremental Sync)

```python
from datetime import datetime

# Get orders updated since specific date
from_date = datetime(2025, 10, 1, 0, 0, 0)

orders = client.get_orders_by_updated_at(
    from_date=from_date,
    max_orders=100
)

print(f"Orders updated since {from_date}: {len(orders)}")

for order in orders:
    print(f"Order #{order['order_id']} updated at: {order.get('updated_at')}")
```

---

## 🔍 Advanced Filtering

### Example 6: Multiple Filters

```python
# Complex filter: paid orders from last month
from datetime import datetime, timedelta

date_from = datetime.now() - timedelta(days=30)

filters = {
    'date': {'>=': date_from.strftime('%Y-%m-%d %H:%M:%S')},
    'paid': 1,
    'status_id': {'!=': 4}  # Not cancelled
}

orders = client.get_orders_filtered(
    filters=filters,
    max_orders=200,
    order_by='-date'  # Newest first
)

print(f"Paid non-cancelled orders: {len(orders)}")
```

### Example 7: Operator Examples

```python
# Shoper API supports these operators in filters:
# >=, <=, !=, <, >, =

filters = {
    'sum': {'>': 100.0},           # Total > 100
    'status_id': {'!=': 4},         # Not cancelled
    'date': {'>=': '2025-01-01 00:00:00'}  # From 2025
}

orders = client.get_orders_filtered(filters=filters, max_orders=50)
```

---

## 📦 Order Details

### Example 8: Get Full Order Data

```python
# Get order with products
order_id = 23456

# Get basic order data
order = client.get_order_details(order_id)
print(f"Order #{order_id}:")
print(f"  Customer: {order.get('email')}")
print(f"  Total: {order.get('sum')} PLN")
print(f"  Status: {order.get('status_id')}")

# Get order products
products = client.get_order_products(order_id)
print(f"  Products: {len(products)}")
for product in products:
    print(f"    - {product.get('name')}: {product.get('quantity')}x {product.get('price_gross')} PLN")
```

---

## 🔄 Pagination

### Example 9: Paginate Through All Orders

```python
def fetch_all_orders(client, max_orders=1000):
    """Fetch all orders with pagination"""
    all_orders = []
    page = 1
    batch_size = 50
    
    while len(all_orders) < max_orders:
        print(f"Fetching page {page}...")
        
        orders = client.get_orders_page(limit=batch_size, page=page)
        
        if not orders:
            print("No more orders")
            break
        
        all_orders.extend(orders)
        print(f"Total so far: {len(all_orders)}")
        
        if len(orders) < batch_size:
            print("Last page")
            break
        
        page += 1
    
    return all_orders

# Usage
orders = fetch_all_orders(client, max_orders=500)
print(f"Fetched {len(orders)} orders total")
```

---

## 🔗 Integration with Odoo

### Example 10: Sync from Python Code

```python
# In Odoo shell or custom module

# Get active configuration
config = env['shoper.config'].get_active_config()

# Run synchronization
config.action_sync_orders()

# Or in custom code:
from odoo.addons.odoo_shoper_integration.models.shoper_api_client import ShoperAPIClient

client = ShoperAPIClient(config.shop_url, config.access_token)
orders = client.get_orders_by_updated_at(
    from_date=config.last_order_update,
    max_orders=config.max_orders_per_sync
)

for order_data in orders:
    # Check if exists
    existing = env['sale.order'].search([
        ('shoper_order_id', '=', order_data['order_id'])
    ], limit=1)
    
    if existing:
        # Update
        existing.update_from_shoper(order_data)
    else:
        # Create
        env['sale.order'].create_from_shoper(order_data, config)
```

### Example 11: Custom Sync Logic

```python
def sync_specific_orders(env, order_ids):
    """Sync specific orders by ID"""
    config = env['shoper.config'].get_active_config()
    client = ShoperAPIClient(config.shop_url, config.access_token)
    
    for order_id in order_ids:
        print(f"Syncing order #{order_id}...")
        
        # Fetch from Shoper
        order_data = client.get_order_details(order_id)
        if not order_data:
            print(f"  Order not found in Shoper")
            continue
        
        # Get products
        products = client.get_order_products(order_id)
        order_data['products'] = products
        
        # Check if exists in Odoo
        existing = env['sale.order'].search([
            ('shoper_order_id', '=', order_id)
        ], limit=1)
        
        if existing:
            existing.update_from_shoper(order_data)
            print(f"  Updated: {existing.name}")
        else:
            new_order = env['sale.order'].create_from_shoper(order_data, config)
            print(f"  Created: {new_order.name}")
    
    print("Done!")

# Usage
sync_specific_orders(env, [23450, 23451, 23452])
```

---

## 📊 Statistics and Monitoring

### Example 12: Get Sync Statistics

```python
# Get configuration stats
config = env['shoper.config'].get_active_config()

print(f"Sync Statistics:")
print(f"  Last sync: {config.last_sync_date}")
print(f"  Last order update: {config.last_order_update}")
print(f"  Total imported: {config.total_orders_imported}")
print(f"  Total updated: {config.total_orders_updated}")
print(f"  Last import count: {config.last_import_count}")
print(f"  Last update count: {config.last_update_count}")
```

### Example 13: Monitor Recent Changes

```python
from datetime import datetime, timedelta

# Get orders changed in last 24 hours
date_from = datetime.now() - timedelta(hours=24)

changed_orders = env['sale.order'].search([
    ('is_shoper_order', '=', True),
    ('shoper_updated_at', '>=', date_from)
], order='shoper_updated_at desc')

print(f"Orders changed in last 24h: {len(changed_orders)}")

for order in changed_orders:
    print(f"  {order.name} (Shoper #{order.shoper_order_id}):")
    print(f"    Status: {order.shoper_status_name}")
    print(f"    Paid: {order.shoper_paid}")
    print(f"    Updated: {order.shoper_updated_at}")
```

---

## 🧪 Testing and Debugging

### Example 14: Test API Filters

```python
# Test different filter formats
test_filters = [
    {'updated_at': {'>=': '2025-10-01 00:00:00'}},
    {'status_id': 2},
    {'paid': 1},
    {'sum': {'>': 100.0}},
    {'date': {'>=': '2025-10-01 00:00:00'}, 'status_id': {'!=': 4}}
]

for i, filters in enumerate(test_filters, 1):
    print(f"\nTest {i}: {filters}")
    count = client.get_orders_count(filters)
    print(f"  Matching orders: {count}")
    
    if count > 0:
        orders = client.get_orders_page(limit=5, filters=filters)
        print(f"  Sample IDs: {[o['order_id'] for o in orders]}")
```

### Example 15: Debug Order Data

```python
import json

# Fetch order and print structure
order_id = 23456
order = client.get_order_details(order_id)

if order:
    print("Order structure:")
    print(json.dumps(order, indent=2, ensure_ascii=False))
    
    print("\nKey fields:")
    print(f"  order_id: {order.get('order_id')}")
    print(f"  email: {order.get('email')}")
    print(f"  sum: {order.get('sum')}")
    print(f"  status_id: {order.get('status_id')}")
    print(f"  paid: {order.get('paid')}")
    print(f"  updated_at: {order.get('updated_at')}")
    print(f"  date: {order.get('date')}")
```

---

## ⚡ Performance Tips

### Example 16: Batch Processing

```python
def sync_orders_in_batches(client, order_ids, batch_size=20):
    """Process orders in batches with commits"""
    total = len(order_ids)
    
    for i in range(0, total, batch_size):
        batch = order_ids[i:i+batch_size]
        print(f"Processing batch {i//batch_size + 1} ({len(batch)} orders)...")
        
        for order_id in batch:
            try:
                order_data = client.get_order_details(order_id)
                if order_data:
                    # Process order
                    process_order(env, order_data)
            except Exception as e:
                print(f"  Error: {order_id}: {e}")
        
        # Commit batch
        env.cr.commit()
        print(f"  Committed {len(batch)} orders")
    
    print("All batches processed!")

# Usage
all_order_ids = list(range(23000, 23100))  # 100 orders
sync_orders_in_batches(client, all_order_ids, batch_size=20)
```

### Example 17: Concurrent Requests (Advanced)

```python
from concurrent.futures import ThreadPoolExecutor
import time

def fetch_order_safe(client, order_id):
    """Thread-safe order fetching"""
    try:
        return client.get_order_details(order_id)
    except Exception as e:
        print(f"Error fetching #{order_id}: {e}")
        return None

def fetch_orders_concurrent(client, order_ids, max_workers=5):
    """Fetch multiple orders concurrently"""
    print(f"Fetching {len(order_ids)} orders with {max_workers} workers...")
    
    start = time.time()
    
    with ThreadPoolExecutor(max_workers=max_workers) as executor:
        orders = list(executor.map(
            lambda oid: fetch_order_safe(client, oid),
            order_ids
        ))
    
    elapsed = time.time() - start
    valid_orders = [o for o in orders if o]
    
    print(f"Fetched {len(valid_orders)} orders in {elapsed:.2f}s")
    return valid_orders

# Usage (be careful with API rate limits!)
order_ids = list(range(23000, 23050))
orders = fetch_orders_concurrent(client, order_ids, max_workers=3)
```

---

## 🛠️ Utility Functions

### Example 18: Helper Functions

```python
def get_recent_orders(env, days=7):
    """Get orders from last N days"""
    from datetime import datetime, timedelta
    date_from = datetime.now() - timedelta(days=days)
    
    return env['sale.order'].search([
        ('is_shoper_order', '=', True),
        ('shoper_date', '>=', date_from)
    ], order='shoper_date desc')

def get_unpaid_orders(env):
    """Get unpaid Shoper orders"""
    return env['sale.order'].search([
        ('is_shoper_order', '=', True),
        ('shoper_paid', '=', False)
    ])

def get_orders_by_status(env, status_id):
    """Get orders by Shoper status ID"""
    return env['sale.order'].search([
        ('is_shoper_order', '=', True),
        ('shoper_status_id', '=', status_id)
    ])

def export_orders_to_csv(orders, filename='orders.csv'):
    """Export orders to CSV"""
    import csv
    
    with open(filename, 'w', newline='', encoding='utf-8') as f:
        writer = csv.writer(f)
        writer.writerow(['Odoo ID', 'Shoper ID', 'Customer', 'Total', 'Status', 'Paid'])
        
        for order in orders:
            writer.writerow([
                order.name,
                order.shoper_order_id,
                order.partner_id.name,
                order.amount_total,
                order.shoper_status_name,
                'Yes' if order.shoper_paid else 'No'
            ])
    
    print(f"Exported {len(orders)} orders to {filename}")

# Usage
recent = get_recent_orders(env, days=30)
unpaid = get_unpaid_orders(env)
processing = get_orders_by_status(env, status_id=2)

export_orders_to_csv(unpaid, 'unpaid_orders.csv')
```

---

## 📝 Shell Commands

### Example 19: Odoo Shell Usage

```bash
# Start Odoo shell
odoo-bin shell -c /etc/odoo/odoo.conf -d your_database

# Then in shell:
```

```python
# Import modules
from odoo.addons.odoo_shoper_integration.models.shoper_api_client import ShoperAPIClient

# Get config
config = env['shoper.config'].get_active_config()

# Create client
client = ShoperAPIClient(config.shop_url, config.access_token)

# Test
print(client.test_connection())
print(f"Total orders: {client.get_orders_count()}")

# Sync
config.action_sync_orders()

# Exit
exit()
```

---

## 🎯 Common Tasks

### Task 1: Re-sync Failed Orders

```python
# Find orders without products
empty_orders = env['sale.order'].search([
    ('is_shoper_order', '=', True),
    ('order_line', '=', False)
])

print(f"Found {len(empty_orders)} empty orders")

# Re-fetch and update
config = env['shoper.config'].get_active_config()
client = ShoperAPIClient(config.shop_url, config.access_token)

for order in empty_orders:
    print(f"Re-syncing {order.name} (Shoper #{order.shoper_order_id})")
    order_data = client.get_order_details(order.shoper_order_id)
    
    if order_data:
        products = client.get_order_products(order.shoper_order_id)
        order_data['products'] = products
        
        # Re-create lines
        env['sale.order']._create_order_lines_from_shoper(order, products)
        print(f"  Added {len(products)} products")

env.cr.commit()
```

### Task 2: Update All Order Statuses

```python
# Force update all Shoper orders
shoper_orders = env['sale.order'].search([
    ('is_shoper_order', '=', True)
])

config = env['shoper.config'].get_active_config()
client = ShoperAPIClient(config.shop_url, config.access_token)

for order in shoper_orders:
    order_data = client.get_order_details(order.shoper_order_id)
    if order_data:
        order.update_from_shoper(order_data)
        print(f"Updated {order.name}")

env.cr.commit()
```

---

**For more examples, see EXAMPLES.md**
