import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useAuth } from "@/hooks/use-auth";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Switch } from "@/components/ui/switch";
import { Webhook, Trash2, Plus, FileText, Send, Pencil } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { useState } from "react";
import { useLocation } from "wouter";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Checkbox } from "@/components/ui/checkbox";
import { format } from "date-fns";
import { pl } from "date-fns/locale";

interface WebhookConfig {
  id: number;
  name: string;
  url: string;
  events: string[];
  is_active: boolean;
  retry_attempts: number;
  created_at: string;
}

const AVAILABLE_EVENTS = [
  { value: "order.created", label: "Nowe zamówienie" },
  { value: "order.updated", label: "Zaktualizowane zamówienie" },
  { value: "order.paid", label: "Zamówienie opłacone" },
  { value: "order.shipped", label: "Zamówienie wysłane" },
];

const createWebhookFormSchema = z.object({
  name: z.string().min(3, "Nazwa musi mieć minimum 3 znaki").max(255, "Nazwa może mieć maksymalnie 255 znaków"),
  url: z.string().url("Podaj prawidłowy URL (https://...)"),
  events: z.array(z.string()).min(1, "Wybierz co najmniej jedno wydarzenie"),
  secret: z.string().transform(val => val === "" ? undefined : val).optional(),
  retryAttempts: z.coerce.number().min(0, "Minimum 0 prób").max(10, "Maksimum 10 prób").default(3),
});

export default function WebhooksPage() {
  const { user } = useAuth();
  const { toast } = useToast();
  const [, setLocation] = useLocation();
  const [deleteWebhookId, setDeleteWebhookId] = useState<number | null>(null);
  const [showDialog, setShowDialog] = useState(false);
  const [editingWebhook, setEditingWebhook] = useState<WebhookConfig | null>(null);

  const form = useForm<z.infer<typeof createWebhookFormSchema>>({
    resolver: zodResolver(createWebhookFormSchema),
    defaultValues: {
      name: "",
      url: "",
      events: [],
      secret: "",
      retryAttempts: 3,
    },
  });

  const { data: webhooks, isLoading } = useQuery<WebhookConfig[]>({
    queryKey: ["/api/webhooks"],
    enabled: user?.role === 'admin',
  });

  const createWebhookMutation = useMutation({
    mutationFn: async (data: z.infer<typeof createWebhookFormSchema>) => {
      const response = await apiRequest("POST", "/api/webhooks", data);
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/webhooks"] });
      setShowDialog(false);
      form.reset();
      toast({
        title: "Sukces",
        description: "Webhook został utworzony",
      });
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się utworzyć webhooka",
        variant: "destructive",
      });
    },
  });

  const updateWebhookMutation = useMutation({
    mutationFn: async ({ webhookId, isActive }: { webhookId: number; isActive: boolean }) => {
      const response = await fetch(`/api/webhooks/${webhookId}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ isActive }),
        credentials: "include",
      });
      if (!response.ok) throw new Error("Failed to update webhook");
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/webhooks"] });
      toast({
        title: "Sukces",
        description: "Status webhooka został zaktualizowany",
      });
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się zaktualizować webhooka",
        variant: "destructive",
      });
    },
  });

  const editWebhookMutation = useMutation({
    mutationFn: async ({ webhookId, data }: { webhookId: number; data: Partial<z.infer<typeof createWebhookFormSchema>> }) => {
      const response = await apiRequest("PUT", `/api/webhooks/${webhookId}`, data);
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/webhooks"] });
      setShowDialog(false);
      setEditingWebhook(null);
      form.reset();
      toast({
        title: "Sukces",
        description: "Webhook został zaktualizowany",
      });
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się zaktualizować webhooka",
        variant: "destructive",
      });
    },
  });

  const deleteWebhookMutation = useMutation({
    mutationFn: async (webhookId: number) => {
      const response = await fetch(`/api/webhooks/${webhookId}`, {
        method: "DELETE",
        credentials: "include",
      });
      if (!response.ok) throw new Error("Failed to delete webhook");
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/webhooks"] });
      toast({
        title: "Sukces",
        description: "Webhook został usunięty",
      });
      setDeleteWebhookId(null);
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się usunąć webhooka",
        variant: "destructive",
      });
      setDeleteWebhookId(null);
    },
  });

  const testWebhookMutation = useMutation({
    mutationFn: async (webhookId: number) => {
      const response = await fetch(`/api/webhooks/${webhookId}/test`, {
        method: "POST",
        credentials: "include",
      });
      if (!response.ok) throw new Error("Failed to send test webhook");
      return response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["/api/webhooks/logs"] });
      toast({
        title: "Test webhook wysłany",
        description: data.message || "Sprawdź logi dostaw aby zobaczyć wynik",
      });
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się wysłać testowego webhooka",
        variant: "destructive",
      });
    },
  });

  if (user?.role !== 'admin') {
    return (
      <div className="flex items-center justify-center h-full">
        <Card>
          <CardHeader>
            <CardTitle>Brak dostępu</CardTitle>
            <CardDescription>
              Tylko administratorzy mają dostęp do zarządzania webhookami.
            </CardDescription>
          </CardHeader>
        </Card>
      </div>
    );
  }

  const getStatusBadge = (isActive: boolean) => {
    return isActive ? (
      <Badge variant="default" data-testid="badge-status-active">Aktywny</Badge>
    ) : (
      <Badge variant="secondary" data-testid="badge-status-inactive">Nieaktywny</Badge>
    );
  };

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Webhooks</h1>
          <p className="text-muted-foreground">Zarządzaj powiadomieniami push dla zdarzeń zamówień</p>
        </div>
        <div className="flex items-center gap-3">
          <Button
            variant="outline"
            onClick={() => setLocation("/webhooks/logs")}
            data-testid="button-view-logs"
          >
            <FileText className="w-4 h-4 mr-2" />
            Logi dostaw
          </Button>
          <Button
            onClick={() => setShowDialog(true)}
            data-testid="button-create-webhook"
          >
            <Plus className="w-4 h-4 mr-2" />
            Nowy webhook
          </Button>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Lista webhooków</CardTitle>
          <CardDescription>
            Webhooks wysyłają powiadomienia HTTP POST o wydarzeniach w czasie rzeczywistym
          </CardDescription>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="text-center py-8 text-muted-foreground">Ładowanie...</div>
          ) : webhooks && webhooks.length > 0 ? (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Nazwa</TableHead>
                  <TableHead>URL</TableHead>
                  <TableHead>Wydarzenia</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead>Retry</TableHead>
                  <TableHead>Utworzono</TableHead>
                  <TableHead className="text-right">Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {webhooks.map((webhook) => (
                  <TableRow key={webhook.id} data-testid={`row-webhook-${webhook.id}`}>
                    <TableCell className="font-medium">{webhook.name}</TableCell>
                    <TableCell>
                      <code className="text-sm text-muted-foreground" data-testid={`text-url-${webhook.id}`}>
                        {webhook.url.length > 40 ? webhook.url.substring(0, 40) + "..." : webhook.url}
                      </code>
                    </TableCell>
                    <TableCell>
                      <div className="flex flex-wrap gap-1">
                        {webhook.events.map((event) => (
                          <Badge key={event} variant="outline" className="text-xs" data-testid={`badge-event-${event}`}>
                            {event}
                          </Badge>
                        ))}
                      </div>
                    </TableCell>
                    <TableCell>{getStatusBadge(webhook.is_active)}</TableCell>
                    <TableCell className="text-muted-foreground">{webhook.retry_attempts}x</TableCell>
                    <TableCell className="text-muted-foreground">
                      {format(new Date(webhook.created_at), "dd.MM.yyyy", { locale: pl })}
                    </TableCell>
                    <TableCell className="text-right">
                      <div className="flex items-center justify-end gap-2">
                        <Switch
                          checked={webhook.is_active}
                          onCheckedChange={(checked) =>
                            updateWebhookMutation.mutate({ webhookId: webhook.id, isActive: checked })
                          }
                          data-testid={`switch-active-${webhook.id}`}
                        />
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => {
                            setEditingWebhook(webhook);
                            form.reset({
                              name: webhook.name,
                              url: webhook.url,
                              events: webhook.events,
                              secret: "",
                              retryAttempts: webhook.retry_attempts,
                            });
                            setShowDialog(true);
                          }}
                          data-testid={`button-edit-${webhook.id}`}
                          title="Edytuj webhook"
                        >
                          <Pencil className="w-4 h-4" />
                        </Button>
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => testWebhookMutation.mutate(webhook.id)}
                          disabled={testWebhookMutation.isPending}
                          data-testid={`button-test-${webhook.id}`}
                          title="Wyślij testowego webhoka"
                        >
                          <Send className="w-4 h-4" />
                        </Button>
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => setDeleteWebhookId(webhook.id)}
                          data-testid={`button-delete-${webhook.id}`}
                        >
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          ) : (
            <div className="text-center py-8 text-muted-foreground">
              <Webhook className="w-12 h-12 mx-auto mb-4 opacity-50" />
              <p>Brak webhooków</p>
              <p className="text-sm">Utwórz pierwszy webhook, aby otrzymywać powiadomienia o zdarzeniach</p>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Create/Edit Webhook Dialog */}
      <Dialog open={showDialog} onOpenChange={(open) => {
        setShowDialog(open);
        if (!open) {
          setEditingWebhook(null);
          form.reset();
        }
      }}>
        <DialogContent className="max-w-2xl" data-testid={editingWebhook ? "dialog-edit-webhook" : "dialog-create-webhook"}>
          <DialogHeader>
            <DialogTitle>{editingWebhook ? "Edytuj webhook" : "Nowy webhook"}</DialogTitle>
            <DialogDescription>
              {editingWebhook 
                ? "Zmodyfikuj konfigurację webhooka" 
                : "Skonfiguruj endpoint HTTP, który będzie otrzymywał powiadomienia o zdarzeniach"}
            </DialogDescription>
          </DialogHeader>
          <Form {...form}>
            <form onSubmit={form.handleSubmit((data) => {
              if (editingWebhook) {
                editWebhookMutation.mutate({ webhookId: editingWebhook.id, data });
              } else {
                createWebhookMutation.mutate(data);
              }
            })} className="space-y-4 py-4">
              <FormField
                control={form.control}
                name="name"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa</FormLabel>
                    <FormControl>
                      <Input
                        placeholder="np. Webhook Odoo - produkcja"
                        data-testid="input-webhook-name"
                        {...field}
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="url"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>URL endpointu</FormLabel>
                    <FormControl>
                      <Input
                        type="url"
                        placeholder="https://your-domain.com/webhook/alpma-orders"
                        data-testid="input-webhook-url"
                        {...field}
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="secret"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Secret (opcjonalny)</FormLabel>
                    <FormControl>
                      <Input
                        type="password"
                        placeholder="Pozostaw puste, aby wygenerować automatycznie"
                        data-testid="input-webhook-secret"
                        {...field}
                      />
                    </FormControl>
                    <FormDescription>
                      Secret jest używany do podpisywania payloadu HMAC-SHA256
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="events"
                render={() => (
                  <FormItem>
                    <FormLabel>Wydarzenia</FormLabel>
                    <div className="space-y-2">
                      {AVAILABLE_EVENTS.map((event) => (
                        <FormField
                          key={event.value}
                          control={form.control}
                          name="events"
                          render={({ field }) => (
                            <FormItem className="flex items-center space-x-2 space-y-0">
                              <FormControl>
                                <Checkbox
                                  checked={field.value?.includes(event.value)}
                                  onCheckedChange={(checked) => {
                                    return checked
                                      ? field.onChange([...field.value, event.value])
                                      : field.onChange(field.value?.filter((value) => value !== event.value))
                                  }}
                                  data-testid={`checkbox-event-${event.value}`}
                                />
                              </FormControl>
                              <FormLabel className="cursor-pointer font-normal">
                                <code className="text-sm">{event.value}</code> - {event.label}
                              </FormLabel>
                            </FormItem>
                          )}
                        />
                      ))}
                    </div>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="retryAttempts"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Liczba prób ponowienia (retry)</FormLabel>
                    <FormControl>
                      <Input
                        type="number"
                        min="0"
                        max="10"
                        data-testid="input-retry-attempts"
                        {...field}
                        onChange={(e) => field.onChange(parseInt(e.target.value) || 3)}
                      />
                    </FormControl>
                    <FormDescription>
                      Backoff: 2s, 4s, 8s (domyślnie: 3 próby)
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <DialogFooter>
                <Button variant="outline" type="button" onClick={() => setShowDialog(false)}>
                  Anuluj
                </Button>
                <Button
                  type="submit"
                  disabled={createWebhookMutation.isPending || editWebhookMutation.isPending}
                  data-testid={editingWebhook ? "button-confirm-edit" : "button-confirm-create"}
                >
                  {editingWebhook 
                    ? (editWebhookMutation.isPending ? "Zapisywanie..." : "Zapisz zmiany")
                    : (createWebhookMutation.isPending ? "Tworzenie..." : "Utwórz webhook")}
                </Button>
              </DialogFooter>
            </form>
          </Form>
        </DialogContent>
      </Dialog>

      {/* Delete Confirmation Dialog */}
      <AlertDialog open={deleteWebhookId !== null} onOpenChange={() => setDeleteWebhookId(null)}>
        <AlertDialogContent data-testid="dialog-delete-webhook">
          <AlertDialogHeader>
            <AlertDialogTitle>Czy na pewno usunąć webhook?</AlertDialogTitle>
            <AlertDialogDescription>
              Ta akcja jest nieodwracalna. Webhook przestanie otrzymywać powiadomienia o zdarzeniach.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-delete">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={() => deleteWebhookId && deleteWebhookMutation.mutate(deleteWebhookId)}
              className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
              data-testid="button-confirm-delete"
            >
              Usuń webhook
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
