import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Link2, CheckCircle2, AlertCircle, Info, Download } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";

export default function Settings() {
  const { toast } = useToast();
  const [autoRefresh, setAutoRefresh] = useState(true);
  const [refreshInterval, setRefreshInterval] = useState("3");
  const [shoperAutoRefresh, setShoperAutoRefresh] = useState(true);
  const [shoperRefreshInterval, setShoperRefreshInterval] = useState("5");

  const { data: connection, isLoading: connectionLoading } = useQuery<{
    isActive: boolean;
    hasCredentials: boolean;
    fromEnv: boolean;
    clientId?: string;
  }>({
    queryKey: ["/api/allegro/connection"],
  });

  const { data: syncSettings } = useQuery<{
    autoRefreshEnabled: boolean;
    refreshIntervalMinutes: string;
  }>({
    queryKey: ["/api/sync/settings"],
  });

  const { data: shoperSyncSettings } = useQuery<{
    shoperAutoRefreshEnabled: boolean;
    shoperRefreshIntervalMinutes: string;
  }>({
    queryKey: ["/api/shoper/sync/settings"],
  });

  useEffect(() => {
    if (syncSettings) {
      setAutoRefresh(syncSettings.autoRefreshEnabled ?? true);
      setRefreshInterval(syncSettings.refreshIntervalMinutes ?? "3");
    }
  }, [syncSettings]);

  useEffect(() => {
    if (shoperSyncSettings) {
      setShoperAutoRefresh(shoperSyncSettings.shoperAutoRefreshEnabled ?? true);
      setShoperRefreshInterval(shoperSyncSettings.shoperRefreshIntervalMinutes ?? "5");
    }
  }, [shoperSyncSettings]);

  const saveSyncSettingsMutation = useMutation({
    mutationFn: async (data: {
      autoRefreshEnabled: boolean;
      refreshIntervalMinutes: string;
    }) => {
      return await apiRequest("POST", "/api/sync/settings", data);
    },
    onSuccess: () => {
      toast({
        title: "Ustawienia zapisane",
        description: "Ustawienia synchronizacji zostały zaktualizowane.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/sync/settings"] });
    },
  });

  const saveShoperSyncSettingsMutation = useMutation({
    mutationFn: async (data: {
      shoperAutoRefreshEnabled: boolean;
      shoperRefreshIntervalMinutes: string;
    }) => {
      return await apiRequest("POST", "/api/shoper/sync/settings", data);
    },
    onSuccess: () => {
      toast({
        title: "Ustawienia Shoper zapisane",
        description: "Ustawienia automatycznej synchronizacji Shoper zostały zaktualizowane.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/shoper/sync/settings"] });
    },
  });

  const handleOAuthConnect = async () => {
    if (!connection?.hasCredentials) {
      toast({
        title: "Brak danych uwierzytelniających",
        description: "Dodaj ALLEGRO_CLIENT_ID i ALLEGRO_CLIENT_SECRET jako sekrety w Replit.",
        variant: "destructive",
      });
      return;
    }
      
    window.location.href = "/api/allegro/auth";
  };

  const handleSyncSettingsChange = () => {
    saveSyncSettingsMutation.mutate({
      autoRefreshEnabled: autoRefresh,
      refreshIntervalMinutes: refreshInterval,
    });
  };

  const handleShoperSyncSettingsChange = () => {
    saveShoperSyncSettingsMutation.mutate({
      shoperAutoRefreshEnabled: shoperAutoRefresh,
      shoperRefreshIntervalMinutes: shoperRefreshInterval,
    });
  };

  return (
    <div className="flex-1 overflow-auto">
      <div className="max-w-4xl mx-auto px-6 py-8">
        <div className="mb-8">
          <h1 className="text-3xl font-semibold tracking-tight">Konfiguracja</h1>
          <p className="text-sm text-muted-foreground mt-1">
            Zarządzaj połączeniem z Allegro i ustawieniami synchronizacji
          </p>
        </div>

        <div className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>Połączenie z Allegro</CardTitle>
              <CardDescription>
                Skonfiguruj dane OAuth do połączenia z API Allegro
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              {connection?.isActive ? (
                <div className="flex items-center gap-3 p-4 rounded-lg border bg-chart-3/5 border-chart-3/20">
                  <CheckCircle2 className="h-5 w-5 text-chart-3" />
                  <div className="flex-1">
                    <p className="font-medium text-sm">Połączenie aktywne</p>
                    <p className="text-xs text-muted-foreground mt-0.5">
                      Twoje konto jest pomyślnie połączone z Allegro
                    </p>
                  </div>
                </div>
              ) : (
                <div className="flex items-center gap-3 p-4 rounded-lg border bg-chart-4/5 border-chart-4/20">
                  <AlertCircle className="h-5 w-5 text-chart-4" />
                  <div className="flex-1">
                    <p className="font-medium text-sm">Brak połączenia</p>
                    <p className="text-xs text-muted-foreground mt-0.5">
                      Połącz swoje konto Allegro, aby rozpocząć
                    </p>
                  </div>
                </div>
              )}

              {!connection?.hasCredentials && (
                <div className="flex items-start gap-3 p-4 rounded-lg border bg-muted/50">
                  <Info className="h-5 w-5 text-muted-foreground mt-0.5" />
                  <div className="flex-1 space-y-2">
                    <p className="font-medium text-sm">Wymagane sekrety</p>
                    <p className="text-xs text-muted-foreground">
                      Dodaj następujące sekrety w ustawieniach Replit (Tools → Secrets):
                    </p>
                    <ul className="text-xs text-muted-foreground list-disc list-inside space-y-1">
                      <li><code className="bg-muted px-1 py-0.5 rounded">ALLEGRO_CLIENT_ID</code></li>
                      <li><code className="bg-muted px-1 py-0.5 rounded">ALLEGRO_CLIENT_SECRET</code></li>
                    </ul>
                  </div>
                </div>
              )}

              {connection?.hasCredentials && (
                <div className="flex items-start gap-3 p-4 rounded-lg border bg-muted/50">
                  <CheckCircle2 className="h-5 w-5 text-chart-3 mt-0.5" />
                  <div className="flex-1">
                    <p className="font-medium text-sm">Dane uwierzytelniające skonfigurowane</p>
                    <p className="text-xs text-muted-foreground mt-0.5">
                      {connection.fromEnv 
                        ? "Client ID i Secret pobrane ze zmiennych środowiskowych"
                        : "Client ID i Secret dostępne w systemie"}
                    </p>
                  </div>
                </div>
              )}

              <Button
                variant="default"
                onClick={handleOAuthConnect}
                disabled={connectionLoading || !connection?.hasCredentials}
                data-testid="button-connect-allegro"
                className="w-full"
              >
                <Link2 className="h-4 w-4" />
                Połącz z Allegro
              </Button>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Połączenie z Shoper</CardTitle>
              <CardDescription>
                Skonfiguruj dane API do połączenia z platformą Shoper
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="flex items-start gap-3 p-4 rounded-lg border bg-muted/50">
                <Info className="h-5 w-5 text-muted-foreground mt-0.5" />
                <div className="flex-1 space-y-2">
                  <p className="font-medium text-sm">Wymagane sekrety</p>
                  <p className="text-xs text-muted-foreground">
                    Dodaj następujące sekrety w ustawieniach Replit (Tools → Secrets):
                  </p>
                  <ul className="text-xs text-muted-foreground list-disc list-inside space-y-1">
                    <li><code className="bg-muted px-1 py-0.5 rounded">SHOPER_SHOP_URL</code> - URL sklepu (np. https://sklep.myshoper.com)</li>
                    <li><code className="bg-muted px-1 py-0.5 rounded">SHOPER_USERNAME</code> - Nazwa użytkownika WebAPI</li>
                    <li><code className="bg-muted px-1 py-0.5 rounded">SHOPER_PASSWORD</code> - Hasło użytkownika WebAPI</li>
                  </ul>
                </div>
              </div>

              <ShoperSyncButton />
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Automatyczna synchronizacja Shoper</CardTitle>
              <CardDescription>
                Skonfiguruj cykliczne pobieranie zamówień z Shoper
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label htmlFor="shoper-auto-refresh">Automatyczne odświeżanie</Label>
                  <p className="text-sm text-muted-foreground">
                    Pobieraj zaktualizowane zamówienia automatycznie z Shoper
                  </p>
                </div>
                <Switch
                  id="shoper-auto-refresh"
                  checked={shoperAutoRefresh}
                  onCheckedChange={setShoperAutoRefresh}
                  data-testid="switch-shoper-auto-refresh"
                />
              </div>

              {shoperAutoRefresh && (
                <div className="space-y-2">
                  <Label htmlFor="shoper-refresh-interval">Częstotliwość odświeżania</Label>
                  <Select
                    value={shoperRefreshInterval}
                    onValueChange={setShoperRefreshInterval}
                  >
                    <SelectTrigger
                      id="shoper-refresh-interval"
                      data-testid="select-shoper-refresh-interval"
                    >
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="1">Co 1 minutę</SelectItem>
                      <SelectItem value="3">Co 3 minuty</SelectItem>
                      <SelectItem value="5">Co 5 minut</SelectItem>
                      <SelectItem value="10">Co 10 minut</SelectItem>
                    </SelectContent>
                  </Select>
                  <p className="text-xs text-muted-foreground">
                    Synchronizacja używa filtru updated_at - pobierane są tylko zamówienia zmodyfikowane od ostatniej synchronizacji (uwzględnia wszystkie zmiany: statusy, produkty, ceny, adresy)
                  </p>
                </div>
              )}

              <Button
                onClick={handleShoperSyncSettingsChange}
                disabled={saveShoperSyncSettingsMutation.isPending}
                data-testid="button-save-shoper-sync-settings"
              >
                Zapisz ustawienia Shoper
              </Button>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Ustawienia synchronizacji Allegro</CardTitle>
              <CardDescription>
                Skonfiguruj automatyczne odświeżanie zamówień z Allegro
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label htmlFor="auto-refresh">Automatyczne odświeżanie</Label>
                  <p className="text-sm text-muted-foreground">
                    Pobieraj nowe zamówienia automatycznie
                  </p>
                </div>
                <Switch
                  id="auto-refresh"
                  checked={autoRefresh}
                  onCheckedChange={setAutoRefresh}
                  data-testid="switch-auto-refresh"
                />
              </div>

              {autoRefresh && (
                <div className="space-y-2">
                  <Label htmlFor="refresh-interval">Częstotliwość odświeżania</Label>
                  <Select
                    value={refreshInterval}
                    onValueChange={setRefreshInterval}
                  >
                    <SelectTrigger
                      id="refresh-interval"
                      data-testid="select-refresh-interval"
                    >
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="1">Co 1 minutę</SelectItem>
                      <SelectItem value="3">Co 3 minuty</SelectItem>
                      <SelectItem value="5">Co 5 minut</SelectItem>
                      <SelectItem value="10">Co 10 minut</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              )}

              <Button
                onClick={handleSyncSettingsChange}
                disabled={saveSyncSettingsMutation.isPending}
                data-testid="button-save-sync-settings"
              >
                Zapisz ustawienia
              </Button>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Zdjęcia produktów</CardTitle>
              <CardDescription>
                Pobierz zdjęcia produktów z Allegro API
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <p className="text-sm text-muted-foreground">
                Kliknij poniżej, aby pobrać zdjęcia wszystkich produktów z zamówień.
                Zdjęcia zostaną zapisane w katalogu products/ z nazwami według SKU.
              </p>
              <DownloadImagesButton />
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}

function ShoperSyncButton() {
  const { toast } = useToast();
  
  const syncMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/shoper/sync", {});
    },
    onSuccess: (data: any) => {
      toast({
        title: "Synchronizacja zakończona",
        description: `Zsynchronizowano ${data.ordersFetched} zamówień z Shoper (${data.ordersNew} nowych, ${data.ordersUpdated} zaktualizowanych)`,
      });
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd synchronizacji",
        description: error.message || "Nie udało się zsynchronizować zamówień z Shoper",
        variant: "destructive",
      });
    },
  });

  return (
    <Button
      onClick={() => syncMutation.mutate()}
      disabled={syncMutation.isPending}
      data-testid="button-sync-shoper"
      className="w-full"
      variant="default"
    >
      <Link2 className="h-4 w-4" />
      {syncMutation.isPending ? "Synchronizacja..." : "Synchronizuj zamówienia Shoper"}
    </Button>
  );
}

function DownloadImagesButton() {
  const { toast } = useToast();
  
  const downloadMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/products/download-images", {});
    },
    onSuccess: (data: any) => {
      toast({
        title: "Zdjęcia pobrane",
        description: `Pobrano ${data.success}/${data.total} zdjęć produktów`,
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd pobierania",
        description: error.message || "Nie udało się pobrać zdjęć produktów",
        variant: "destructive",
      });
    },
  });

  return (
    <Button
      onClick={() => downloadMutation.mutate()}
      disabled={downloadMutation.isPending}
      data-testid="button-download-images"
    >
      <Download className="h-4 w-4" />
      {downloadMutation.isPending ? "Pobieranie..." : "Pobierz zdjęcia produktów"}
    </Button>
  );
}
