import { useQuery } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { Clock, Package, TrendingUp, ArrowRight } from "lucide-react";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { format } from "date-fns";
import { pl } from "date-fns/locale";

export default function RecentUpdates() {
  const [, navigate] = useLocation();

  const { data: orders, isLoading } = useQuery<any[]>({
    queryKey: ['/api/recent-updates'],
    refetchInterval: 30000,
  });

  if (isLoading) {
    return (
      <div className="flex-1 overflow-auto">
        <div className="max-w-7xl mx-auto px-6 py-8">
          <div className="h-8 w-64 bg-muted rounded animate-pulse mb-8" />
          <div className="space-y-4">
            {[1, 2, 3].map((i) => (
              <div key={i} className="h-24 bg-muted rounded animate-pulse" />
            ))}
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="flex-1 overflow-auto">
      <div className="max-w-7xl mx-auto px-6 py-8">
        <div className="mb-8">
          <div className="flex items-center gap-3 mb-2">
            <Clock className="h-8 w-8 text-primary" />
            <h1 className="text-3xl font-semibold tracking-tight">
              Ostatnie zmiany
            </h1>
          </div>
          <p className="text-muted-foreground">
            Zamówienia zmodyfikowane podczas ostatniej synchronizacji
          </p>
        </div>

        {!orders || orders.length === 0 ? (
          <Card>
            <CardContent className="flex flex-col items-center justify-center py-12">
              <Package className="h-16 w-16 text-muted-foreground mb-4" />
              <h3 className="text-xl font-semibold mb-2">Brak ostatnich zmian</h3>
              <p className="text-muted-foreground text-center max-w-md">
                Nie wykryto żadnych zmian w zamówieniach podczas ostatniej synchronizacji.
              </p>
            </CardContent>
          </Card>
        ) : (
          <div className="space-y-4">
            {orders.map((order: any) => {
              const source = order.source || 'ALLEGRO';
              const totalAmount = parseFloat(order.total_to_pay_amount || "0");
              
              return (
                <Card 
                  key={order.order_id} 
                  className="hover-elevate cursor-pointer transition-all"
                  onClick={() => navigate(`/order/${order.order_number}`)}
                  data-testid={`card-recent-order-${order.order_number}`}
                >
                  <CardHeader className="pb-3">
                    <div className="flex items-start justify-between gap-4">
                      <div className="flex-1">
                        <div className="flex items-center gap-2 mb-2">
                          <CardTitle className="text-lg font-mono">
                            {order.order_code || order.order_number}
                          </CardTitle>
                          <Badge 
                            variant="outline"
                            className={
                              source === 'SHOPER' 
                                ? 'bg-blue-500/10 text-blue-600 dark:text-blue-400 border-blue-500/20' 
                                : 'bg-orange-500/10 text-orange-600 dark:text-orange-400 border-orange-500/20'
                            }
                          >
                            {source}
                          </Badge>
                        </div>
                        <CardDescription className="space-y-1">
                          {order.buyer_login && (
                            <div className="font-medium">
                              {order.buyer_login}
                            </div>
                          )}
                          <div className="flex flex-col gap-1 text-xs text-muted-foreground">
                            {order.order_date && (
                              <div>
                                📅 Zamówienie: {format(new Date(order.order_date), "dd.MM.yyyy, HH:mm", { locale: pl })}
                              </div>
                            )}
                            {(order.updated_at || order.created_at) && (
                              <div>
                                🔄 Aktualizacja: {format(
                                  new Date(order.updated_at > order.created_at ? order.updated_at : order.created_at), 
                                  "dd.MM.yyyy, HH:mm", 
                                  { locale: pl }
                                )}
                              </div>
                            )}
                          </div>
                        </CardDescription>
                      </div>
                      <div className="text-right">
                        <div className="text-2xl font-bold">
                          {totalAmount.toFixed(2)} {order.total_to_pay_currency || 'PLN'}
                        </div>
                        {order.status && (
                          <div className="text-xs text-muted-foreground mt-1">
                            Status: {order.status}
                          </div>
                        )}
                      </div>
                    </div>
                  </CardHeader>
                  <CardContent className="pt-0">
                    <div className="flex items-center justify-between">
                      <div className="text-sm text-muted-foreground">
                        {order.change_description || "Wykryto zmianę w zamówieniu"}
                      </div>
                      <Button 
                        variant="ghost" 
                        size="sm"
                        onClick={(e) => {
                          e.stopPropagation();
                          navigate(`/order/${order.order_number}`);
                        }}
                        data-testid={`button-view-order-${order.order_number}`}
                      >
                        Zobacz szczegóły
                        <ArrowRight className="ml-2 h-4 w-4" />
                      </Button>
                    </div>
                  </CardContent>
                </Card>
              );
            })}
          </div>
        )}
      </div>
    </div>
  );
}
