import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { History, FileText, ArrowRight, Clock, Package, CheckCircle2, XCircle, TrendingUp } from "lucide-react";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { format } from "date-fns";
import { pl } from "date-fns/locale";

interface OrderChange {
  id: string;
  order_id: string;
  order_number: string;
  order_code: string;
  source: string;
  change_type: string;
  field_changed: string;
  old_value: string;
  new_value: string;
  detected_at: string;
}

interface SyncLog {
  id: number;
  sync_started_at: string;
  sync_completed_at: string | null;
  status: 'RUNNING' | 'SUCCESS' | 'ERROR';
  orders_fetched: number;
  orders_new: number;
  orders_updated: number;
  error_message: string | null;
  sync_duration_ms: number | null;
}

const fieldLabels: Record<string, string> = {
  status: "Status zamówienia",
  payment_status: "Status płatności",
  payment_amount: "Kwota płatności",
  refund_amount: "Kwota zwrotu",
  has_returns: "Zwroty produktów",
  tracking_numbers: "Numery przesyłek",
  buyer_address: "Adres kupującego",
  buyer_city: "Miasto",
};

export default function OrderChanges() {
  const [, navigate] = useLocation();
  const [activeTab, setActiveTab] = useState("changes");

  const { data: changes, isLoading: changesLoading } = useQuery<OrderChange[]>({
    queryKey: ['/api/order-changes'],
    refetchInterval: 30000,
  });

  const { data: recentOrders, isLoading: recentLoading } = useQuery<any[]>({
    queryKey: ['/api/recent-updates'],
    refetchInterval: 30000,
  });

  const { data: logs, isLoading: logsLoading } = useQuery<SyncLog[]>({
    queryKey: ["/api/sync/logs"],
  });

  const formatDuration = (ms: number | null) => {
    if (!ms) return "-";
    if (ms < 1000) return `${ms}ms`;
    return `${(ms / 1000).toFixed(1)}s`;
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'SUCCESS':
        return <CheckCircle2 className="h-4 w-4 text-green-500" />;
      case 'ERROR':
        return <XCircle className="h-4 w-4 text-red-500" />;
      case 'RUNNING':
        return <Clock className="h-4 w-4 text-blue-500" />;
      default:
        return null;
    }
  };

  const getStatusBadge = (status: string) => {
    switch (status) {
      case 'SUCCESS':
        return <Badge variant="default" className="bg-green-600">Sukces</Badge>;
      case 'ERROR':
        return <Badge variant="destructive">Błąd</Badge>;
      case 'RUNNING':
        return <Badge variant="secondary">W trakcie</Badge>;
      default:
        return <Badge variant="outline">{status}</Badge>;
    }
  };

  return (
    <div className="flex-1 overflow-auto">
      <div className="max-w-7xl mx-auto px-6 py-8">
        <div className="mb-8">
          <div className="flex items-center gap-3 mb-2">
            <History className="h-8 w-8 text-primary" />
            <h1 className="text-3xl font-semibold tracking-tight">
              Historia Zmian
            </h1>
          </div>
          <p className="text-muted-foreground">
            Wszystkie zmiany, aktualizacje i logi synchronizacji
          </p>
        </div>

        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
          <TabsList className="grid w-full max-w-md grid-cols-3">
            <TabsTrigger value="changes" data-testid="tab-changes">
              Wykryte zmiany
            </TabsTrigger>
            <TabsTrigger value="recent" data-testid="tab-recent">
              Ostatnie aktualizacje
            </TabsTrigger>
            <TabsTrigger value="logs" data-testid="tab-logs">
              Logi synchronizacji
            </TabsTrigger>
          </TabsList>

          {/* Tab: Wykryte zmiany */}
          <TabsContent value="changes" className="space-y-4">
            {changesLoading ? (
              <Card>
                <CardContent className="py-12">
                  <div className="h-96 bg-muted rounded animate-pulse" />
                </CardContent>
              </Card>
            ) : !changes || changes.length === 0 ? (
              <Card>
                <CardContent className="flex flex-col items-center justify-center py-12">
                  <FileText className="h-16 w-16 text-muted-foreground mb-4" />
                  <h3 className="text-xl font-semibold mb-2">Brak zmian</h3>
                  <p className="text-muted-foreground text-center max-w-md">
                    Nie wykryto jeszcze żadnych zmian w zamówieniach.
                  </p>
                </CardContent>
              </Card>
            ) : (
              <Card>
                <CardHeader>
                  <CardTitle>Wykryte zmiany ({changes.length})</CardTitle>
                  <CardDescription>
                    Lista wszystkich automatycznie wykrytych zmian w zamówieniach
                  </CardDescription>
                </CardHeader>
                <CardContent className="p-0">
                  <div className="overflow-x-auto">
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead>Data i czas</TableHead>
                          <TableHead>Zamówienie</TableHead>
                          <TableHead>Pole</TableHead>
                          <TableHead>Stara wartość</TableHead>
                          <TableHead>Nowa wartość</TableHead>
                          <TableHead className="text-right">Akcje</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {changes.map((change) => (
                          <TableRow 
                            key={change.id}
                            data-testid={`row-change-${change.id}`}
                          >
                            <TableCell className="text-sm text-muted-foreground whitespace-nowrap">
                              {format(new Date(change.detected_at), "dd.MM.yyyy HH:mm", { locale: pl })}
                            </TableCell>
                            <TableCell>
                              <div className="flex items-center gap-2">
                                <span className="font-mono text-sm font-medium">
                                  {change.order_code}
                                </span>
                                <Badge 
                                  variant="outline" 
                                  className={
                                    change.source === 'SHOPER' 
                                      ? 'bg-blue-500/10 text-blue-600 dark:text-blue-400 border-blue-500/20' 
                                      : 'bg-orange-500/10 text-orange-600 dark:text-orange-400 border-orange-500/20'
                                  }
                                >
                                  {change.source}
                                </Badge>
                              </div>
                            </TableCell>
                            <TableCell className="font-medium">
                              {fieldLabels[change.field_changed] || change.field_changed}
                            </TableCell>
                            <TableCell>
                              <code className="text-xs bg-muted px-2 py-1 rounded">
                                {change.old_value}
                              </code>
                            </TableCell>
                            <TableCell>
                              <code className="text-xs bg-primary/10 px-2 py-1 rounded">
                                {change.new_value}
                              </code>
                            </TableCell>
                            <TableCell className="text-right">
                              <Button
                                variant="ghost"
                                size="sm"
                                onClick={() => navigate(`/order/${change.order_number}`)}
                                data-testid={`button-view-order-${change.order_number}`}
                              >
                                Zobacz
                                <ArrowRight className="ml-2 h-4 w-4" />
                              </Button>
                            </TableCell>
                          </TableRow>
                        ))}
                      </TableBody>
                    </Table>
                  </div>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          {/* Tab: Ostatnie aktualizacje */}
          <TabsContent value="recent" className="space-y-4">
            {recentLoading ? (
              <div className="space-y-4">
                {[1, 2, 3].map((i) => (
                  <div key={i} className="h-24 bg-muted rounded animate-pulse" />
                ))}
              </div>
            ) : !recentOrders || recentOrders.length === 0 ? (
              <Card>
                <CardContent className="flex flex-col items-center justify-center py-12">
                  <Package className="h-16 w-16 text-muted-foreground mb-4" />
                  <h3 className="text-xl font-semibold mb-2">Brak ostatnich zmian</h3>
                  <p className="text-muted-foreground text-center max-w-md">
                    Nie wykryto żadnych zmian w zamówieniach podczas ostatniej synchronizacji.
                  </p>
                </CardContent>
              </Card>
            ) : (
              <div className="space-y-4">
                {recentOrders.map((order: any) => {
                  const source = order.source || 'ALLEGRO';
                  const totalAmount = parseFloat(order.total_to_pay_amount || "0");
                  
                  return (
                    <Card 
                      key={order.order_id} 
                      className="hover-elevate cursor-pointer transition-all"
                      onClick={() => navigate(`/order/${order.order_number}`)}
                      data-testid={`card-recent-order-${order.order_number}`}
                    >
                      <CardHeader className="pb-3">
                        <div className="flex items-start justify-between gap-4">
                          <div className="flex-1">
                            <div className="flex items-center gap-2 mb-2">
                              <CardTitle className="text-lg font-mono">
                                {order.order_code || order.order_number}
                              </CardTitle>
                              <Badge 
                                variant="outline"
                                className={
                                  source === 'SHOPER' 
                                    ? 'bg-blue-500/10 text-blue-600 dark:text-blue-400 border-blue-500/20' 
                                    : 'bg-orange-500/10 text-orange-600 dark:text-orange-400 border-orange-500/20'
                                }
                              >
                                {source}
                              </Badge>
                            </div>
                            <CardDescription className="space-y-1">
                              {order.buyer_login && (
                                <div className="font-medium">
                                  {order.buyer_login}
                                </div>
                              )}
                              <div className="flex flex-col gap-1 text-xs text-muted-foreground">
                                {order.order_date && (
                                  <div>
                                    📅 Zamówienie: {format(new Date(order.order_date), "dd.MM.yyyy, HH:mm", { locale: pl })}
                                  </div>
                                )}
                                {(order.updated_at || order.created_at) && (
                                  <div>
                                    🔄 Aktualizacja: {format(
                                      new Date(order.updated_at > order.created_at ? order.updated_at : order.created_at), 
                                      "dd.MM.yyyy, HH:mm", 
                                      { locale: pl }
                                    )}
                                  </div>
                                )}
                              </div>
                            </CardDescription>
                          </div>
                          <div className="text-right">
                            <div className="text-2xl font-bold">
                              {totalAmount.toFixed(2)} {order.total_to_pay_currency || 'PLN'}
                            </div>
                            {order.status && (
                              <div className="text-xs text-muted-foreground mt-1">
                                Status: {order.status}
                              </div>
                            )}
                          </div>
                        </div>
                      </CardHeader>
                    </Card>
                  );
                })}
              </div>
            )}
          </TabsContent>

          {/* Tab: Logi synchronizacji */}
          <TabsContent value="logs" className="space-y-4">
            {logsLoading ? (
              <div className="space-y-4">
                {[...Array(5)].map((_, i) => (
                  <div key={i} className="h-32 bg-muted rounded animate-pulse" />
                ))}
              </div>
            ) : logs && logs.length > 0 ? (
              <div className="space-y-4">
                {logs.map((log) => (
                  <Card key={log.id} data-testid={`card-sync-log-${log.id}`}>
                    <CardHeader>
                      <div className="flex items-start justify-between">
                        <div className="flex items-center gap-3">
                          {getStatusIcon(log.status)}
                          <div>
                            <CardTitle className="text-lg" data-testid={`text-sync-time-${log.id}`}>
                              {format(new Date(log.sync_started_at), "PPP 'o' HH:mm:ss", { locale: pl })}
                            </CardTitle>
                            <CardDescription data-testid={`text-sync-duration-${log.id}`}>
                              Czas trwania: {formatDuration(log.sync_duration_ms)}
                            </CardDescription>
                          </div>
                        </div>
                        <div data-testid={`badge-sync-status-${log.id}`}>
                          {getStatusBadge(log.status)}
                        </div>
                      </div>
                    </CardHeader>
                    <CardContent>
                      <div className="grid grid-cols-3 gap-4">
                        <div>
                          <p className="text-sm text-muted-foreground">Pobrano zamówień</p>
                          <p className="text-2xl font-bold" data-testid={`text-orders-fetched-${log.id}`}>
                            {log.orders_fetched}
                          </p>
                        </div>
                        <div>
                          <p className="text-sm text-muted-foreground">Nowych</p>
                          <p className="text-2xl font-bold text-green-600" data-testid={`text-orders-new-${log.id}`}>
                            {log.orders_new}
                          </p>
                        </div>
                        <div>
                          <p className="text-sm text-muted-foreground">Zaktualizowanych</p>
                          <p className="text-2xl font-bold text-blue-600" data-testid={`text-orders-updated-${log.id}`}>
                            {log.orders_updated}
                          </p>
                        </div>
                      </div>
                      {log.error_message && (
                        <div className="mt-4 p-3 bg-red-50 dark:bg-red-950 rounded-md">
                          <p className="text-sm font-medium text-red-800 dark:text-red-200">Błąd:</p>
                          <p className="text-sm text-red-600 dark:text-red-300 mt-1" data-testid={`text-error-message-${log.id}`}>
                            {log.error_message}
                          </p>
                        </div>
                      )}
                    </CardContent>
                  </Card>
                ))}
              </div>
            ) : (
              <Card>
                <CardContent className="flex flex-col items-center justify-center py-12">
                  <FileText className="h-16 w-16 text-muted-foreground mb-4" />
                  <h3 className="text-xl font-semibold mb-2">Brak logów</h3>
                  <p className="text-muted-foreground text-center max-w-md">
                    Nie znaleziono żadnych logów synchronizacji.
                  </p>
                </CardContent>
              </Card>
            )}
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}
