import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { CheckCircle2, XCircle, Clock } from "lucide-react";

interface SyncLog {
  id: number;
  sync_started_at: string;
  sync_completed_at: string | null;
  status: 'RUNNING' | 'SUCCESS' | 'ERROR';
  orders_fetched: number;
  orders_new: number;
  orders_updated: number;
  error_message: string | null;
  sync_duration_ms: number | null;
}

export default function Logs() {
  const { data: logs, isLoading } = useQuery<SyncLog[]>({
    queryKey: ["/api/sync/logs"],
  });

  const formatDuration = (ms: number | null) => {
    if (!ms) return "-";
    if (ms < 1000) return `${ms}ms`;
    return `${(ms / 1000).toFixed(1)}s`;
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'SUCCESS':
        return <CheckCircle2 className="h-4 w-4 text-green-500" />;
      case 'ERROR':
        return <XCircle className="h-4 w-4 text-red-500" />;
      case 'RUNNING':
        return <Clock className="h-4 w-4 text-blue-500" />;
      default:
        return null;
    }
  };

  const getStatusBadge = (status: string) => {
    switch (status) {
      case 'SUCCESS':
        return <Badge variant="default" className="bg-green-600">Sukces</Badge>;
      case 'ERROR':
        return <Badge variant="destructive">Błąd</Badge>;
      case 'RUNNING':
        return <Badge variant="secondary">W trakcie</Badge>;
      default:
        return <Badge variant="outline">{status}</Badge>;
    }
  };

  return (
    <div className="h-full overflow-auto">
      <div className="container mx-auto p-6 space-y-6">
        <div>
          <h1 className="text-3xl font-bold" data-testid="text-page-title">Historia Synchronizacji</h1>
          <p className="text-muted-foreground mt-1" data-testid="text-page-description">
            Logi automatycznej synchronizacji z Allegro API
          </p>
        </div>

        {isLoading ? (
          <div className="space-y-4">
            {[...Array(5)].map((_, i) => (
              <Card key={i}>
                <CardHeader>
                  <Skeleton className="h-6 w-1/3" />
                  <Skeleton className="h-4 w-1/2 mt-2" />
                </CardHeader>
              </Card>
            ))}
          </div>
        ) : logs && logs.length > 0 ? (
          <div className="space-y-4">
            {logs.map((log) => (
              <Card key={log.id} data-testid={`card-sync-log-${log.id}`}>
                <CardHeader>
                  <div className="flex items-start justify-between">
                    <div className="flex items-center gap-3">
                      {getStatusIcon(log.status)}
                      <div>
                        <CardTitle className="text-lg" data-testid={`text-sync-time-${log.id}`}>
                          {format(new Date(log.sync_started_at), "PPP 'o' HH:mm:ss", { locale: pl })}
                        </CardTitle>
                        <CardDescription data-testid={`text-sync-duration-${log.id}`}>
                          Czas trwania: {formatDuration(log.sync_duration_ms)}
                        </CardDescription>
                      </div>
                    </div>
                    <div data-testid={`badge-sync-status-${log.id}`}>
                      {getStatusBadge(log.status)}
                    </div>
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-3 gap-4">
                    <div>
                      <p className="text-sm text-muted-foreground">Pobrano zamówień</p>
                      <p className="text-2xl font-bold" data-testid={`text-orders-fetched-${log.id}`}>
                        {log.orders_fetched}
                      </p>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Nowych</p>
                      <p className="text-2xl font-bold text-green-600" data-testid={`text-orders-new-${log.id}`}>
                        {log.orders_new}
                      </p>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Zaktualizowanych</p>
                      <p className="text-2xl font-bold text-blue-600" data-testid={`text-orders-updated-${log.id}`}>
                        {log.orders_updated}
                      </p>
                    </div>
                  </div>
                  {log.error_message && (
                    <div className="mt-4 p-3 bg-red-50 dark:bg-red-950 rounded-md">
                      <p className="text-sm font-medium text-red-800 dark:text-red-200">Błąd:</p>
                      <p className="text-sm text-red-600 dark:text-red-300 mt-1" data-testid={`text-error-message-${log.id}`}>
                        {log.error_message}
                      </p>
                    </div>
                  )}
                </CardContent>
              </Card>
            ))}
          </div>
        ) : (
          <Card>
            <CardHeader>
              <CardTitle data-testid="text-no-logs">Brak logów synchronizacji</CardTitle>
              <CardDescription>
                Uruchom pierwszą synchronizację aby zobaczyć logi
              </CardDescription>
            </CardHeader>
          </Card>
        )}
      </div>
    </div>
  );
}
