import { useState } from "react";
import { useMutation } from "@tanstack/react-query";
import { Calendar, Download, RefreshCw } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";

export default function DateSync() {
  const { toast } = useToast();
  
  const [allegroStartDate, setAllegroStartDate] = useState("");
  const [allegroEndDate, setAllegroEndDate] = useState("");
  
  const [shoperStartDate, setShoperStartDate] = useState("");
  const [shoperEndDate, setShoperEndDate] = useState("");

  const allegroSyncMutation = useMutation({
    mutationFn: async (data: { startDate: string; endDate: string }) => {
      return await apiRequest("POST", "/api/sync/date-range", data);
    },
    onSuccess: (data: any) => {
      toast({
        title: "Synchronizacja Allegro zakończona",
        description: `Pobrano ${data.ordersFetched} zamówień (${data.ordersNew} nowych, ${data.ordersUpdated} zaktualizowanych)`,
      });
      setAllegroStartDate("");
      setAllegroEndDate("");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd synchronizacji Allegro",
        description: error.message || "Nie udało się zsynchronizować zamówień",
        variant: "destructive",
      });
    },
  });

  const shoperSyncMutation = useMutation({
    mutationFn: async (data: { startDate: string; endDate: string }) => {
      return await apiRequest("POST", "/api/shoper/sync/date-range", data);
    },
    onSuccess: (data: any) => {
      toast({
        title: "Synchronizacja Shoper zakończona",
        description: `Pobrano ${data.ordersFetched} zamówień (${data.ordersNew} nowych, ${data.ordersUpdated} zaktualizowanych${data.ordersFailed ? `, ${data.ordersFailed} błędów` : ''})`,
      });
      setShoperStartDate("");
      setShoperEndDate("");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd synchronizacji Shoper",
        description: error.message || "Nie udało się zsynchronizować zamówień",
        variant: "destructive",
      });
    },
  });

  const handleAllegroSync = () => {
    if (!allegroStartDate || !allegroEndDate) {
      toast({
        title: "Brak dat",
        description: "Wybierz datę początkową i końcową",
        variant: "destructive",
      });
      return;
    }

    if (new Date(allegroStartDate) > new Date(allegroEndDate)) {
      toast({
        title: "Błędny zakres dat",
        description: "Data początkowa musi być wcześniejsza niż data końcowa",
        variant: "destructive",
      });
      return;
    }

    allegroSyncMutation.mutate({
      startDate: allegroStartDate,
      endDate: allegroEndDate,
    });
  };

  const handleShoperSync = () => {
    if (!shoperStartDate || !shoperEndDate) {
      toast({
        title: "Brak dat",
        description: "Wybierz datę początkową i końcową",
        variant: "destructive",
      });
      return;
    }

    if (new Date(shoperStartDate) > new Date(shoperEndDate)) {
      toast({
        title: "Błędny zakres dat",
        description: "Data początkowa musi być wcześniejsza niż data końcowa",
        variant: "destructive",
      });
      return;
    }

    shoperSyncMutation.mutate({
      startDate: shoperStartDate,
      endDate: shoperEndDate,
    });
  };

  return (
    <div className="flex-1 overflow-auto">
      <div className="max-w-4xl mx-auto px-6 py-8">
        <div className="mb-8">
          <h1 className="text-3xl font-semibold tracking-tight">Synchronizacja według dat</h1>
          <p className="text-sm text-muted-foreground mt-1">
            Pobierz i zaktualizuj zamówienia z wybranego przedziału czasowego
          </p>
        </div>

        <div className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Calendar className="h-5 w-5" />
                Synchronizacja Allegro
              </CardTitle>
              <CardDescription>
                Pobierz zamówienia Allegro z określonego przedziału dat. Zamówienia zostaną zaktualizowane z aktualnymi danymi płatności.
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="grid gap-4 sm:grid-cols-2">
                <div className="space-y-2">
                  <Label htmlFor="allegro-start-date">Data początkowa</Label>
                  <Input
                    id="allegro-start-date"
                    type="date"
                    value={allegroStartDate}
                    onChange={(e) => setAllegroStartDate(e.target.value)}
                    data-testid="input-allegro-start-date"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="allegro-end-date">Data końcowa</Label>
                  <Input
                    id="allegro-end-date"
                    type="date"
                    value={allegroEndDate}
                    onChange={(e) => setAllegroEndDate(e.target.value)}
                    data-testid="input-allegro-end-date"
                  />
                </div>
              </div>
              
              <Button
                onClick={handleAllegroSync}
                disabled={allegroSyncMutation.isPending}
                className="w-full sm:w-auto"
                data-testid="button-allegro-sync"
              >
                {allegroSyncMutation.isPending ? (
                  <>
                    <RefreshCw className="mr-2 h-4 w-4 animate-spin" />
                    Synchronizacja...
                  </>
                ) : (
                  <>
                    <Download className="mr-2 h-4 w-4" />
                    Synchronizuj Allegro
                  </>
                )}
              </Button>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Calendar className="h-5 w-5" />
                Synchronizacja Shoper
              </CardTitle>
              <CardDescription>
                Pobierz zamówienia Shoper z określonego przedziału dat. Zamówienia zostaną zaktualizowane z aktualnymi danymi płatności.
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="grid gap-4 sm:grid-cols-2">
                <div className="space-y-2">
                  <Label htmlFor="shoper-start-date">Data początkowa</Label>
                  <Input
                    id="shoper-start-date"
                    type="date"
                    value={shoperStartDate}
                    onChange={(e) => setShoperStartDate(e.target.value)}
                    data-testid="input-shoper-start-date"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="shoper-end-date">Data końcowa</Label>
                  <Input
                    id="shoper-end-date"
                    type="date"
                    value={shoperEndDate}
                    onChange={(e) => setShoperEndDate(e.target.value)}
                    data-testid="input-shoper-end-date"
                  />
                </div>
              </div>
              
              <Button
                onClick={handleShoperSync}
                disabled={shoperSyncMutation.isPending}
                className="w-full sm:w-auto"
                data-testid="button-shoper-sync"
              >
                {shoperSyncMutation.isPending ? (
                  <>
                    <RefreshCw className="mr-2 h-4 w-4 animate-spin" />
                    Synchronizacja...
                  </>
                ) : (
                  <>
                    <Download className="mr-2 h-4 w-4" />
                    Synchronizuj Shoper
                  </>
                )}
              </Button>
            </CardContent>
          </Card>

          <Card className="bg-muted/50">
            <CardHeader>
              <CardTitle className="text-base">Informacje</CardTitle>
            </CardHeader>
            <CardContent className="space-y-2 text-sm text-muted-foreground">
              <p>
                • Synchronizacja pobiera zamówienia zaktualizowane w wybranym przedziale czasowym
              </p>
              <p>
                • Istniejące zamówienia zostaną zaktualizowane z najnowszymi danymi
              </p>
              <p>
                • Nowe zamówienia zostaną dodane do bazy danych
              </p>
              <p>
                • Użyj tej funkcji do naprawy nieprawidłowych danych płatności w starszych zamówieniach
              </p>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
