import { useMemo, useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { RefreshCw, Search, ChevronLeft, ChevronRight, RotateCcw, Calendar as CalendarIcon } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Calendar } from "@/components/ui/calendar";
import { Badge } from "@/components/ui/badge";
import { MetricsCards } from "@/components/metrics-cards";
import { OrdersTable } from "@/components/orders-table";
import { ThemeToggle } from "@/components/theme-toggle";
import { useToast } from "@/hooks/use-toast";
import { useDashboardSettings } from "@/hooks/use-dashboard-settings";
import { apiRequest, queryClient } from "@/lib/queryClient";
import type { AllegroOrder } from "@shared/schema";
import { LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer } from "recharts";
import { format } from "date-fns";
import { pl } from "date-fns/locale";

interface FilterButtonGroupProps {
  label: string;
  value: string;
  options: Array<{ value: string; label: string; testId: string }>;
  onChange: (value: string) => void;
}

function FilterButtonGroup({ label, value, options, onChange }: FilterButtonGroupProps) {
  return (
    <div className="flex items-center gap-2">
      <span className="text-sm text-muted-foreground">{label}:</span>
      {options.map(opt => (
        <Button
          key={opt.value}
          variant={value === opt.value ? "default" : "outline"}
          size="sm"
          onClick={() => onChange(opt.value)}
          data-testid={opt.testId}
        >
          {opt.label}
        </Button>
      ))}
    </div>
  );
}

export default function Dashboard() {
  const { settings, updateSettings, resetSettings } = useDashboardSettings();
  const { toast } = useToast();
  const [location, setLocation] = useLocation();

  const {
    searchQuery,
    paymentFilter,
    fulfillmentFilter,
    invoiceFilter,
    paymentTypeFilter,
    sourceFilter,
    dateRangeType,
    customDateFrom,
    customDateTo,
    currentPage,
    itemsPerPage,
    sortBy,
    sortOrder,
    chartDays,
  } = settings;

  // Odczytaj filtry z URL przy montowaniu
  useEffect(() => {
    const params = new URLSearchParams(window.location.search);
    const updates: any = {};
    
    if (params.has('search')) updates.searchQuery = params.get('search') || '';
    if (params.has('source')) updates.sourceFilter = params.get('source') || 'all';
    if (params.has('dateRange')) updates.dateRangeType = params.get('dateRange') || 'all';
    if (params.has('dateFrom')) updates.customDateFrom = params.get('dateFrom');
    if (params.has('dateTo')) updates.customDateTo = params.get('dateTo');
    if (params.has('payment')) updates.paymentFilter = params.get('payment') || 'all';
    if (params.has('fulfillment')) updates.fulfillmentFilter = params.get('fulfillment') || 'all';
    if (params.has('invoice')) updates.invoiceFilter = params.get('invoice') || 'all';
    if (params.has('paymentType')) updates.paymentTypeFilter = params.get('paymentType') || 'all';
    if (params.has('page')) updates.currentPage = parseInt(params.get('page') || '1');
    if (params.has('perPage')) updates.itemsPerPage = parseInt(params.get('perPage') || '25');
    if (params.has('sortBy')) updates.sortBy = params.get('sortBy') || 'order_number';
    if (params.has('sortOrder')) updates.sortOrder = (params.get('sortOrder') === 'ASC' ? 'ASC' : 'DESC') as 'ASC' | 'DESC';
    
    if (Object.keys(updates).length > 0) {
      updateSettings(updates);
    }
  }, []);

  // Aktualizuj URL przy każdej zmianie filtrów
  useEffect(() => {
    const params = new URLSearchParams();
    
    if (searchQuery) params.set('search', searchQuery);
    if (sourceFilter !== 'all') params.set('source', sourceFilter);
    if (dateRangeType !== 'all') params.set('dateRange', dateRangeType);
    if (dateRangeType === 'custom' && customDateFrom) params.set('dateFrom', customDateFrom);
    if (dateRangeType === 'custom' && customDateTo) params.set('dateTo', customDateTo);
    if (paymentFilter !== 'all') params.set('payment', paymentFilter);
    if (fulfillmentFilter !== 'all') params.set('fulfillment', fulfillmentFilter);
    if (invoiceFilter !== 'all') params.set('invoice', invoiceFilter);
    if (paymentTypeFilter !== 'all') params.set('paymentType', paymentTypeFilter);
    if (currentPage !== 1) params.set('page', currentPage.toString());
    if (itemsPerPage !== 25) params.set('perPage', itemsPerPage.toString());
    if (sortBy !== 'order_number') params.set('sortBy', sortBy);
    if (sortOrder !== 'DESC') params.set('sortOrder', sortOrder);
    
    const newSearch = params.toString();
    const newUrl = newSearch ? `/?${newSearch}` : '/';
    
    if (window.location.pathname + window.location.search !== newUrl) {
      window.history.replaceState({}, '', newUrl);
    }
  }, [searchQuery, sourceFilter, dateRangeType, customDateFrom, customDateTo, paymentFilter, fulfillmentFilter, invoiceFilter, paymentTypeFilter, currentPage, itemsPerPage, sortBy, sortOrder]);

  // Client-side filters (search, sourceFilter, dateRangeType są server-side)
  const hasClientFilters = paymentFilter !== "all" || fulfillmentFilter !== "all" || 
                           invoiceFilter !== "all" || paymentTypeFilter !== "all";
  
  // Server-side filters
  const hasServerFilters = searchQuery || sourceFilter !== "all" || dateRangeType !== "all";
  
  // Jakiekolwiek filtry (dla UI)
  const hasAnyFilters = hasServerFilters || hasClientFilters;

  // Buduj URL z parametrami
  const buildOrdersUrl = () => {
    const params = new URLSearchParams();
    params.set('sortBy', sortBy);
    params.set('sortOrder', sortOrder);
    
    if (searchQuery) {
      params.set('search', searchQuery);
    }
    
    if (sourceFilter && sourceFilter !== 'all') {
      params.set('sourceFilter', sourceFilter);
    }
    
    if (dateRangeType && dateRangeType !== 'all') {
      params.set('dateRangeType', dateRangeType);
      if (dateRangeType === 'custom' && customDateFrom && customDateTo) {
        params.set('customDateFrom', customDateFrom);
        params.set('customDateTo', customDateTo);
      }
    }
    
    // Server-side pagination tylko gdy brak client-side filtrów
    if (!hasClientFilters) {
      params.set('limit', itemsPerPage.toString());
      params.set('offset', ((currentPage - 1) * itemsPerPage).toString());
    }
    
    return `/api/orders?${params.toString()}`;
  };

  const { data: orders = [], isLoading } = useQuery<AllegroOrder[]>({
    queryKey: [buildOrdersUrl()],
  });

  const buildCountUrl = () => {
    const params = new URLSearchParams();
    if (searchQuery) {
      params.set('search', searchQuery);
    }
    if (sourceFilter && sourceFilter !== 'all') {
      params.set('sourceFilter', sourceFilter);
    }
    if (dateRangeType && dateRangeType !== 'all') {
      params.set('dateRangeType', dateRangeType);
      if (dateRangeType === 'custom' && customDateFrom && customDateTo) {
        params.set('customDateFrom', customDateFrom);
        params.set('customDateTo', customDateTo);
      }
    }
    return `/api/orders/count?${params.toString()}`;
  };

  const { data: totalCountData } = useQuery<{ count: number }>({
    queryKey: [buildCountUrl()],
    enabled: !hasClientFilters,
  });

  const { data: syncStatus } = useQuery<{
    lastSyncAt: string | null;
    isRefreshing: boolean;
  }>({
    queryKey: ["/api/sync/status"],
    refetchInterval: 5000,
  });

  const { data: statistics } = useQuery<{
    totalOrders: number;
    paidOrders: number;
    unpaidOrders: number;
    totalRevenue: number;
  }>({
    queryKey: ["/api/orders/statistics"],
    refetchInterval: 5000,
  });

  const { data: chartData = [], isLoading: isChartLoading } = useQuery<{
    date: string;
    orders: number;
    paidOrders: number;
    revenue: number;
  }[]>({
    queryKey: [`/api/orders/chart?days=${chartDays}`],
  });

  const { data: todayStats, isLoading: isTodayStatsLoading } = useQuery<{
    revenueToday: number;
    paidToday: number;
    codCount: number;
    codAmount: number;
    unpaidNonCod: number;
    paidAllegro: number;
    paidShoper: number;
    ordersCount: number;
    ordersAllegro: number;
    ordersShoper: number;
    revenueYesterday: number;
    dayName: string;
    dateFormatted: string;
    revenueChangePercent: number;
  }>({
    queryKey: ["/api/orders/today-stats"],
    refetchInterval: 5000,
  });

  const filteredOrders = useMemo(() => {
    // Search, sourceFilter, dateRangeType są teraz server-side, więc tutaj tylko pozostałe filtry
    let filtered = [...orders];

    if (paymentFilter !== "all") {
      const statusMap: Record<string, string> = {
        'PAID': 'PAID',
        'UNPAID': 'UNPAID',
        'PENDING': 'PENDING'
      };
      filtered = filtered.filter((order) => {
        const o = order as any;
        const paidAmount = o.payment?.paidAmount?.amount;
        const actualStatus = paidAmount && parseFloat(paidAmount) > 0 ? 'PAID' : 'UNPAID';
        return actualStatus === statusMap[paymentFilter];
      });
    }

    if (fulfillmentFilter !== "all") {
      filtered = filtered.filter((order) => {
        const o = order as any;
        const status = o.status?.toUpperCase() || "NEW";
        return status === fulfillmentFilter;
      });
    }

    if (invoiceFilter !== "all") {
      filtered = filtered.filter((order) => {
        const o = order as any;
        const invoiceRequired = o.invoice?.required || false;
        return invoiceFilter === "yes" ? invoiceRequired : !invoiceRequired;
      });
    }

    if (paymentTypeFilter !== "all") {
      filtered = filtered.filter((order) => {
        const o = order as any;
        const paymentType = o.payment?.type || "";
        return paymentType === paymentTypeFilter;
      });
    }

    return filtered;
  }, [orders, paymentFilter, fulfillmentFilter, invoiceFilter, paymentTypeFilter]);

  const totalCount = hasClientFilters ? filteredOrders.length : (totalCountData?.count || 0);
  const totalPages = Math.ceil(totalCount / itemsPerPage);
  
  const paginatedOrders = useMemo(() => {
    if (hasClientFilters) {
      const startIndex = (currentPage - 1) * itemsPerPage;
      return filteredOrders.slice(startIndex, startIndex + itemsPerPage);
    }
    return filteredOrders;
  }, [filteredOrders, currentPage, itemsPerPage, hasClientFilters]);

  // Statystyki zależne od filtrów
  const filteredStats = useMemo(() => {
    // Jeśli nie ma żadnych filtrów, używamy globalnych statystyk
    if (!hasAnyFilters) {
      return {
        totalOrders: statistics?.totalOrders || 0,
        paidOrders: statistics?.paidOrders || 0,
        unpaidOrders: statistics?.unpaidOrders || 0,
        totalRevenue: statistics?.totalRevenue || 0,
      };
    }

    // Z jakimikolwiek filtrami obliczamy statystyki z filteredOrders
    let paid = 0;
    let unpaid = 0;
    let revenue = 0;

    filteredOrders.forEach((order) => {
      const o = order as any;
      const paidAmount = o.payment?.paidAmount?.amount;
      const totalAmount = o.payment?.totalToPay?.amount;
      
      if (paidAmount && parseFloat(paidAmount) > 0) {
        paid++;
        revenue += parseFloat(paidAmount);
      } else {
        unpaid++;
      }
    });

    return {
      totalOrders: filteredOrders.length,
      paidOrders: paid,
      unpaidOrders: unpaid,
      totalRevenue: revenue,
    };
  }, [filteredOrders, statistics, hasAnyFilters]);

  const totalOrders = filteredStats.totalOrders;
  const paidOrders = filteredStats.paidOrders;
  const unpaidOrders = filteredStats.unpaidOrders;
  const totalRevenue = filteredStats.totalRevenue.toFixed(2);

  const handleRefresh = async () => {
    try {
      await apiRequest("POST", "/api/sync/manual", {});
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
      queryClient.invalidateQueries({ queryKey: ["/api/sync/status"] });
      toast({
        title: "Synchronizacja ukończona",
        description: "Zamówienia zostały zaktualizowane pomyślnie.",
      });
    } catch (error) {
      toast({
        title: "Błąd synchronizacji",
        description: "Nie udało się pobrać zamówień z Allegro.",
        variant: "destructive",
      });
    }
  };

  return (
    <div className="flex-1 overflow-y-auto">
      <div className="w-full px-6 py-8">
        <div className="flex items-center justify-between gap-4 mb-6 flex-wrap">
          <div className="flex items-center gap-3">
            <h1 className="text-2xl font-semibold tracking-tight">Zamówienia</h1>
            <span className="text-sm text-muted-foreground">
              Przegląd i zarządzanie zamówieniami z Allegro
            </span>
          </div>
          <div className="flex items-center gap-2 flex-wrap">
            {syncStatus?.lastSyncAt && (
              <div className="text-sm text-muted-foreground">
                Ostatnia synchronizacja:{" "}
                {new Date(syncStatus.lastSyncAt).toLocaleString("pl-PL")}
              </div>
            )}
            <Button
              variant="outline"
              size="default"
              onClick={handleRefresh}
              disabled={syncStatus?.isRefreshing}
              data-testid="button-refresh"
            >
              <RefreshCw
                className={`h-4 w-4 ${syncStatus?.isRefreshing ? "animate-spin" : ""}`}
              />
              Odśwież
            </Button>
            <ThemeToggle />
          </div>
        </div>

        <MetricsCards
          totalOrders={totalOrders}
          paidOrders={paidOrders}
          unpaidOrders={unpaidOrders}
          totalRevenue={totalRevenue}
          isLoading={isLoading}
        />

        <div className="mt-6">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle>Statystyki zamówień</CardTitle>
                <div className="flex gap-2">
                  <Button
                    variant={chartDays === 7 ? "default" : "outline"}
                    size="sm"
                    onClick={() => updateSettings({ chartDays: 7 })}
                    data-testid="button-chart-7days"
                  >
                    7 dni
                  </Button>
                  <Button
                    variant={chartDays === 10 ? "default" : "outline"}
                    size="sm"
                    onClick={() => updateSettings({ chartDays: 10 })}
                    data-testid="button-chart-10days"
                  >
                    10 dni
                  </Button>
                  <Button
                    variant={chartDays === 30 ? "default" : "outline"}
                    size="sm"
                    onClick={() => updateSettings({ chartDays: 30 })}
                    data-testid="button-chart-30days"
                  >
                    30 dni
                  </Button>
                  <Button
                    variant={chartDays === 90 ? "default" : "outline"}
                    size="sm"
                    onClick={() => updateSettings({ chartDays: 90 })}
                    data-testid="button-chart-90days"
                  >
                    90 dni
                  </Button>
                  <Button
                    variant={chartDays === 180 ? "default" : "outline"}
                    size="sm"
                    onClick={() => updateSettings({ chartDays: 180 })}
                    data-testid="button-chart-180days"
                  >
                    180 dni
                  </Button>
                  <Button
                    variant={chartDays === 360 ? "default" : "outline"}
                    size="sm"
                    onClick={() => updateSettings({ chartDays: 360 })}
                    data-testid="button-chart-360days"
                  >
                    360 dni
                  </Button>
                </div>
              </div>
            </CardHeader>
            <CardContent>
              {isTodayStatsLoading ? (
                <div className="mb-6">
                  <div className="h-6 w-48 bg-muted rounded animate-pulse mb-4" />
                  <div className="grid grid-cols-4 gap-3">
                    {[1, 2, 3, 4, 5, 6, 7, 8].map((i) => (
                      <div key={i} className="p-3 rounded-lg border">
                        <div className="h-3 w-16 bg-muted rounded animate-pulse mb-2" />
                        <div className="h-5 w-20 bg-muted rounded animate-pulse" />
                      </div>
                    ))}
                  </div>
                </div>
              ) : todayStats ? (
                <div className="mb-6">
                  <div className="flex items-center gap-3 mb-4">
                    <h3 className="text-lg font-semibold">
                      {todayStats.dayName}, {todayStats.dateFormatted}
                    </h3>
                    <div className={`text-sm font-medium px-2 py-1 rounded ${
                      todayStats.revenueChangePercent >= 0 
                        ? 'bg-green-500/10 text-green-600 dark:text-green-400' 
                        : 'bg-red-500/10 text-red-600 dark:text-red-400'
                    }`}>
                      {todayStats.revenueChangePercent >= 0 ? '+' : ''}{todayStats.revenueChangePercent}% vs wczoraj
                    </div>
                  </div>
                  <div className="grid grid-cols-4 gap-3">
                    <div className="p-3 rounded-lg border hover-elevate bg-green-500/10" data-testid="card-today-revenue">
                      <div className="text-xs text-green-600 dark:text-green-400 mb-1">
                        Dzienny obrót
                      </div>
                      <div className="text-lg font-bold text-green-600 dark:text-green-400">
                        {todayStats.revenueToday.toFixed(2)} PLN
                      </div>
                    </div>
                    <div className="p-3 rounded-lg border hover-elevate" data-testid="card-paid-today">
                      <div className="text-xs text-muted-foreground mb-1">
                        Zapłacono
                      </div>
                      <div className="text-lg font-bold">
                        {todayStats.paidToday.toFixed(2)} PLN
                      </div>
                    </div>
                    <div className="p-3 rounded-lg border hover-elevate" data-testid="card-cod">
                      <div className="text-xs text-muted-foreground mb-1">
                        Pobrania ({todayStats.codCount})
                      </div>
                      <div className="text-lg font-bold">
                        {todayStats.codAmount.toFixed(2)} PLN
                      </div>
                    </div>
                    <div className="p-3 rounded-lg border hover-elevate" data-testid="card-unpaid">
                      <div className="text-xs text-muted-foreground mb-1">
                        Niezapłacono (poza pobr.)
                      </div>
                      <div className="text-lg font-bold">
                        {todayStats.unpaidNonCod.toFixed(2)} PLN
                      </div>
                    </div>
                    <div className="p-3 rounded-lg border hover-elevate" data-testid="card-paid-allegro">
                      <div className="text-xs text-muted-foreground mb-1">
                        Zapłacono Allegro
                      </div>
                      <div className="text-lg font-bold text-orange-600 dark:text-orange-400">
                        {todayStats.paidAllegro.toFixed(2)} PLN
                      </div>
                    </div>
                    <div className="p-3 rounded-lg border hover-elevate" data-testid="card-paid-shoper">
                      <div className="text-xs text-muted-foreground mb-1">
                        Zapłacono Shoper
                      </div>
                      <div className="text-lg font-bold text-blue-600 dark:text-blue-400">
                        {todayStats.paidShoper.toFixed(2)} PLN
                      </div>
                    </div>
                    <div className="p-3 rounded-lg border hover-elevate" data-testid="card-orders-allegro">
                      <div className="text-xs text-muted-foreground mb-1">
                        Zamówienia Allegro
                      </div>
                      <div className="text-lg font-bold text-orange-600 dark:text-orange-400">
                        {todayStats.ordersAllegro}
                      </div>
                    </div>
                    <div className="p-3 rounded-lg border hover-elevate" data-testid="card-orders-shoper">
                      <div className="text-xs text-muted-foreground mb-1">
                        Zamówienia Shoper
                      </div>
                      <div className="text-lg font-bold text-blue-600 dark:text-blue-400">
                        {todayStats.ordersShoper}
                      </div>
                    </div>
                  </div>
                </div>
              ) : null}
              {isChartLoading ? (
                <div className="h-[300px] flex items-center justify-center">
                  <RefreshCw className="h-8 w-8 animate-spin text-muted-foreground" />
                </div>
              ) : (
                <ResponsiveContainer width="100%" height={300}>
                  <LineChart data={chartData}>
                    <CartesianGrid strokeDasharray="3 3" className="stroke-muted" />
                    <XAxis 
                      dataKey="date" 
                      tick={{ fontSize: 12 }}
                      tickFormatter={(value) => {
                        const date = new Date(value);
                        return `${date.getDate()}/${date.getMonth() + 1}`;
                      }}
                    />
                    <YAxis yAxisId="left" tick={{ fontSize: 12 }} />
                    <YAxis yAxisId="right" orientation="right" tick={{ fontSize: 12 }} />
                    <Tooltip 
                      labelFormatter={(value) => {
                        const date = new Date(value as string);
                        return date.toLocaleDateString("pl-PL");
                      }}
                      formatter={(value: any, name: string) => {
                        if (name === "Przychód") {
                          return [`${parseFloat(value).toFixed(2)} PLN`, name];
                        }
                        return [value, name];
                      }}
                    />
                    <Legend />
                    <Line 
                      yAxisId="left"
                      type="monotone" 
                      dataKey="orders" 
                      stroke="hsl(var(--primary))" 
                      name="Zamówienia"
                      strokeWidth={2}
                    />
                    <Line 
                      yAxisId="left"
                      type="monotone" 
                      dataKey="paidOrders" 
                      stroke="hsl(var(--chart-2))" 
                      name="Opłacone"
                      strokeWidth={2}
                    />
                    <Line 
                      yAxisId="right"
                      type="monotone" 
                      dataKey="revenue" 
                      stroke="hsl(var(--chart-3))" 
                      name="Przychód"
                      strokeWidth={2}
                    />
                  </LineChart>
                </ResponsiveContainer>
              )}
            </CardContent>
          </Card>

          <div className="sticky top-0 z-10 bg-background/95 backdrop-blur supports-[backdrop-filter]:bg-background/60 pt-6 pb-4 space-y-4 mt-6">
            <div className="flex items-center gap-3 flex-wrap">
              <div className="relative flex-1 min-w-[250px]">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="Szukaj po ID, imię, nazwisko, email..."
                  value={searchQuery}
                  onChange={(e) => updateSettings({ searchQuery: e.target.value })}
                  className="pl-9"
                  data-testid="input-search-orders"
                />
              </div>
              
              <FilterButtonGroup
                label="Źródło"
                value={sourceFilter}
                options={[
                  { value: "all", label: "Wszystkie", testId: "button-filter-source-all" },
                  { value: "ALLEGRO", label: "Allegro", testId: "button-filter-source-allegro" },
                  { value: "SHOPER", label: "Shoper", testId: "button-filter-source-shoper" },
                ]}
                onChange={(val) => updateSettings({ sourceFilter: val })}
              />

              <FilterButtonGroup
                label="Płatność"
                value={paymentFilter}
                options={[
                  { value: "all", label: "Wszystkie", testId: "button-filter-payment-all" },
                  { value: "PAID", label: "Opłacone", testId: "button-filter-payment-paid" },
                  { value: "UNPAID", label: "Nieopłacone", testId: "button-filter-payment-unpaid" },
                ]}
                onChange={(val) => updateSettings({ paymentFilter: val })}
              />

              <div className="flex items-center gap-2">
                <span className="text-sm text-muted-foreground">Okres:</span>
                {[
                  { value: "all", label: "Wszystkie" },
                  { value: "today", label: "Dzisiaj" },
                  { value: "yesterday", label: "Wczoraj" },
                  { value: "7days", label: "7 dni" },
                  { value: "30days", label: "30 dni" },
                ].map(opt => (
                  <Button
                    key={opt.value}
                    variant={dateRangeType === opt.value ? "default" : "outline"}
                    size="sm"
                    onClick={() => updateSettings({ dateRangeType: opt.value })}
                    data-testid={`button-filter-date-${opt.value}`}
                  >
                    {opt.label}
                  </Button>
                ))}
                <Popover>
                  <PopoverTrigger asChild>
                    <Button
                      variant={dateRangeType === "custom" ? "default" : "outline"}
                      size="sm"
                      data-testid="button-filter-date-custom"
                    >
                      <CalendarIcon className="h-4 w-4 mr-2" />
                      {dateRangeType === "custom" && customDateFrom && customDateTo
                        ? `${format(new Date(customDateFrom), "dd.MM", { locale: pl })} - ${format(new Date(customDateTo), "dd.MM", { locale: pl })}`
                        : "Własny zakres"}
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-auto p-0" align="start">
                    <div className="p-3 space-y-3">
                      <div className="space-y-2">
                        <div className="text-sm font-medium">Data od:</div>
                        <Calendar
                          mode="single"
                          selected={customDateFrom ? new Date(customDateFrom) : undefined}
                          onSelect={(date) => {
                            if (date) {
                              updateSettings({ 
                                customDateFrom: format(date, "yyyy-MM-dd"),
                                dateRangeType: customDateTo ? "custom" : dateRangeType
                              });
                            }
                          }}
                          locale={pl}
                          data-testid="calendar-date-from"
                        />
                      </div>
                      <div className="space-y-2">
                        <div className="text-sm font-medium">Data do:</div>
                        <Calendar
                          mode="single"
                          selected={customDateTo ? new Date(customDateTo) : undefined}
                          onSelect={(date) => {
                            if (date) {
                              updateSettings({ 
                                customDateTo: format(date, "yyyy-MM-dd"),
                                dateRangeType: customDateFrom ? "custom" : dateRangeType
                              });
                            }
                          }}
                          locale={pl}
                          data-testid="calendar-date-to"
                        />
                      </div>
                      {customDateFrom && customDateTo && (
                        <Button
                          size="sm"
                          className="w-full"
                          onClick={() => updateSettings({ dateRangeType: "custom" })}
                          data-testid="button-apply-custom-date"
                        >
                          Zastosuj
                        </Button>
                      )}
                    </div>
                  </PopoverContent>
                </Popover>
              </div>

              <Select value={fulfillmentFilter} onValueChange={(value) => updateSettings({ fulfillmentFilter: value })}>
                <SelectTrigger className="w-[150px]" data-testid="select-fulfillment-filter">
                  <SelectValue placeholder="Realizacja" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszystkie</SelectItem>
                  <SelectItem value="NEW">Nowe</SelectItem>
                  <SelectItem value="PROCESSING">W realizacji</SelectItem>
                  <SelectItem value="SENT">Wysłane</SelectItem>
                  <SelectItem value="CANCELLED">Anulowane</SelectItem>
                </SelectContent>
              </Select>
              <Select value={paymentTypeFilter} onValueChange={(value) => updateSettings({ paymentTypeFilter: value })}>
                <SelectTrigger className="w-[150px]" data-testid="select-payment-type-filter">
                  <SelectValue placeholder="Typ płatności" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszystkie</SelectItem>
                  <SelectItem value="CASH_ON_DELIVERY">Pobranie</SelectItem>
                  <SelectItem value="ONLINE">Online</SelectItem>
                </SelectContent>
              </Select>
              <Select value={invoiceFilter} onValueChange={(value) => updateSettings({ invoiceFilter: value })}>
                <SelectTrigger className="w-[130px]" data-testid="select-invoice-filter">
                  <SelectValue placeholder="Faktura" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszystkie</SelectItem>
                  <SelectItem value="yes">Z fakturą</SelectItem>
                  <SelectItem value="no">Bez faktury</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="flex items-center justify-between gap-3 flex-wrap">
              <div className="flex items-center gap-3">
                <Badge variant="secondary" className="text-base px-3 py-1" data-testid="badge-results-count">
                  Wyniki: {totalCount.toLocaleString('pl-PL')}
                </Badge>
                <Button
                  variant="outline"
                  size="default"
                  onClick={resetSettings}
                  data-testid="button-reset-filters"
                >
                  <RotateCcw className="h-4 w-4 mr-2" />
                  Resetuj
                </Button>
              </div>

              <div className="flex items-center gap-2">
                <div className="text-sm text-muted-foreground">
                  {paginatedOrders.length > 0 ? (currentPage - 1) * itemsPerPage + 1 : 0} - {Math.min(currentPage * itemsPerPage, totalCount)} z {totalCount}
                  {hasAnyFilters && totalCount !== totalOrders && ` (filtr. z ${totalOrders})`}
                </div>
                <Select
                  value={itemsPerPage.toString()}
                  onValueChange={(value) => {
                    updateSettings({ itemsPerPage: Number(value), currentPage: 1 });
                  }}
                >
                  <SelectTrigger className="w-[90px]" data-testid="select-items-per-page">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="10">10</SelectItem>
                    <SelectItem value="25">25</SelectItem>
                    <SelectItem value="50">50</SelectItem>
                    <SelectItem value="100">100</SelectItem>
                    <SelectItem value="250">250</SelectItem>
                    <SelectItem value="500">500</SelectItem>
                  </SelectContent>
                </Select>
                <div className="flex items-center gap-1">
                  <Button
                    variant="outline"
                    size="icon"
                    onClick={() => updateSettings({ currentPage: Math.max(1, currentPage - 1) })}
                    disabled={currentPage === 1}
                    data-testid="button-prev-page"
                  >
                    <ChevronLeft className="h-4 w-4" />
                  </Button>
                  <div className="text-sm px-2">
                    {currentPage} / {totalPages || 1}
                  </div>
                  <Button
                    variant="outline"
                    size="icon"
                    onClick={() => updateSettings({ currentPage: Math.min(totalPages, currentPage + 1) })}
                    disabled={currentPage === totalPages || totalPages === 0}
                    data-testid="button-next-page"
                  >
                    <ChevronRight className="h-4 w-4" />
                  </Button>
                </div>
              </div>
            </div>

            <OrdersTable 
              orders={paginatedOrders} 
              isLoading={isLoading}
              sortBy={sortBy}
              sortOrder={sortOrder}
              onSort={(column) => {
                if (sortBy === column) {
                  updateSettings({ sortOrder: sortOrder === 'ASC' ? 'DESC' : 'ASC' });
                } else {
                  updateSettings({ sortBy: column, sortOrder: 'DESC' });
                }
              }}
            />
          </div>
        </div>
      </div>
    </div>
  );
}
