import { Badge } from "@/components/ui/badge";
import { CheckCircle, Clock, XCircle, Package, Truck } from "lucide-react";
import type { OrderStatus, FulfillmentStatus } from "@shared/schema";

interface StatusBadgeProps {
  type: "payment" | "fulfillment";
  status: OrderStatus | FulfillmentStatus;
}

export function StatusBadge({ type, status }: StatusBadgeProps) {
  if (type === "payment") {
    const paymentStatus = status as OrderStatus;
    
    const statusConfig = {
      PAID: {
        label: "Opłacone",
        icon: CheckCircle,
        className: "bg-green-500/10 text-green-600 dark:text-green-400 border-green-500/20",
      },
      REFUNDED: {
        label: "Zwrócone",
        icon: XCircle,
        className: "bg-red-500/10 text-red-600 dark:text-red-400 border-red-500/20",
      },
      UNPAID: {
        label: "Nieopłacone",
        icon: XCircle,
        className: "bg-red-500/10 text-red-600 dark:text-red-400 border-red-500/20",
      },
      CASH_ON_DELIVERY: {
        label: "Za pobraniem",
        icon: Clock,
        className: "bg-orange-500/10 text-orange-600 dark:text-orange-400 border-orange-500/20",
      },
      PENDING: {
        label: "Oczekujące",
        icon: Clock,
        className: "bg-red-500/10 text-red-600 dark:text-red-400 border-red-500/20",
      },
      CANCELLED: {
        label: "Anulowane",
        icon: XCircle,
        className: "bg-muted text-muted-foreground border-border",
      },
    };

    const config = statusConfig[paymentStatus];
    const Icon = config.icon;

    return (
      <Badge
        variant="outline"
        className={`${config.className} flex items-center gap-1.5 w-fit`}
        data-testid={`badge-payment-${paymentStatus.toLowerCase()}`}
      >
        <Icon className="h-3 w-3" />
        {config.label}
      </Badge>
    );
  }

  const fulfillmentStatus = status as FulfillmentStatus;
  
  const statusConfig = {
    NEW: {
      label: "Nowe",
      icon: Package,
      className: "bg-chart-2/10 text-chart-2 border-chart-2/20",
    },
    PROCESSING: {
      label: "W realizacji",
      icon: Clock,
      className: "bg-chart-4/10 text-chart-4 border-chart-4/20",
    },
    READY_FOR_SHIPMENT: {
      label: "Gotowe do wysyłki",
      icon: Package,
      className: "bg-chart-1/10 text-chart-1 border-chart-1/20",
    },
    SENT: {
      label: "Wysłane",
      icon: Truck,
      className: "bg-purple-500/10 text-purple-600 dark:text-purple-400 border-purple-500/20",
    },
    DELIVERED: {
      label: "Dostarczone",
      icon: CheckCircle,
      className: "bg-chart-3/10 text-chart-3 border-chart-3/20",
    },
    CANCELLED: {
      label: "Anulowane",
      icon: XCircle,
      className: "bg-muted text-muted-foreground border-border",
    },
  };

  const config = statusConfig[fulfillmentStatus] || {
    label: fulfillmentStatus || "Nieznany",
    icon: Package,
    className: "bg-muted text-muted-foreground border-border",
  };
  const Icon = config.icon;

  return (
    <Badge
      variant="outline"
      className={`${config.className} flex items-center gap-1.5 w-fit`}
      data-testid={`badge-fulfillment-${(fulfillmentStatus || 'unknown').toLowerCase()}`}
    >
      <Icon className="h-3 w-3" />
      {config.label}
    </Badge>
  );
}
